C************************************************************************
C --  This code calculates the ionization of, and the line emission 
C --  from the circumstellar gas around SN 1987A.
C
C --  This version is from September, 1997.
C --  Most recent update, Jan. 24, 2001 (H I collisions).
C
C --  Recent updates:
C --  ---------------
C --  New atomic data and Blinn. & Bart. nommxf.ph included (July, 1997).
C --  Blinn. & Bart. ut87af.ph included (August, 1997).
C --  Better 2-gamma routines for He I and He II (September, 1997).
C --  Function expfn() corrected (April, 2000)
C --  Blinn. et al. nmxfso17.ph included (July, 2000).
C --  Hydrogen collisions updated (Anderson et al. 2000), and corrected 
C      (January, 2001), gives good H-alpha, H-beta and Pa-alpha predictions.
C
C --  The photospheric properties from Woosley's 10L-model,
C --  the Shigeyama et al.'s 11E0.8-model, or Ensman & Burrows'
C --  500full1 and 500full2-models. A model with higher color
C --  temperature than E & B's models, but the same late evolution
C --  is called 500full3. Also included are Blinn. & Bart. models.
C
C --  Read inputs from file    inpshell
c
C  INPUT VARIABLES: 
C  shmass = shell mass (in Mo)
C  mshell = the shell around 87A is divided into mshell subshells
C  den1 = number density (assuming hydrogen only)
C  rs15 = shell radius (in 1E15 cm)
C  ishiwo = ionizing spectrum (0 = 11E1Y6, 1 = 10L,  3 = 500full1,
C           4 = 500full2,  6 = nommxf,  7 = ut87af,  8 = nmxfso17.ph)
C  zzcosm = metal abundance relative to Allen's cosmic abundances.
C  xhexh = AB(He)/AB(H)
C  xnxc = AB(N)/AB(C)
C  xnxo = AB(N)/AB(O)
C  teinn = initial gas temperature in the shell
C
C --  Outputs:  sluvtot (general outputs)
C               sloptdep (line optical depths)
C               outteden  (temp. and abundances for all shells)
C               outemiss (temp. and emissivities for all shells)
C               outradii (radii for all she#lls)
C************************************************************************
C --   Enumeration of the lines --
C
C  (Emissivities for each shell are stored in vector STOLIN,
C   and given in units of erg/(s cm3 sr)).
C
C   1/ H Ly-alpha                    2/ H Ly-alpha (2-phot)
C   3/ H Ly-beta                     4/ H Ly-gamma
C   5/ H H-alpha                     6/ H H-beta
C   7/ H Pa-alpha                    8/ He II 304
C   9/ He II 1640                   10/ He II 4686
C  11/ He I 5876                    12/ He I 10830
C  13/ He I 584                     14/ He I 3188
C  15/ He I 3889                    16/ He I 4471
C  17/ He I 4713                    18/ He I 7065
C  19/ C I 609.8 mu                 20/ C I 370.4 mu
C  21/ C I 9853                     22/ C I 4623
C  23/ C I 2966-68                  24/ C I 8729
C  25/ C II 858.4                   26/ C II 904.1
C  27/ C II 1335                    28/ C II 157.7 mu
C  29/ C II 2326                    30/ C II 454.6 mu
C  31/ C II 353.4 mu                32/ C III 977 
C  33/ C III 1909                   34/ C IV 1549
C  35/ N I 5199-5202                36/ N I 3468
C  37/ N I 10401-410                38/ N II 916.4
C  39/ N II 204 mu                  40/ N II 122 mu
C  41/ N II 6548-83                 42/ N II 3063
C  43/ N II 2143                    44/ N II 5755
C  45/ N III 685.7                  46/ N III 991
C  47/ N III 57.33 mu               48/ N III 1750
C  49/ N III 167.5 mu               50/ N III 123.3 mu
C  51/ N IV 765.2                   52/ N IV 1486
C  53/ N V 1240                     54/ O I 63.17 mu
C  55/ O I 145.6 mu                 56/ O I 6300-64
C  57/ O I 2958-72                  58/ O I 5577
C  59/ O II 833.8                   60/ O II 3726-29
C  61/ O II 2470                    62/ O II 7319-31
C  63/ O III 507.9
C  64/ O III 703.4                  65/ O III 834.5
C  66/ O III 88.34 mu               67/ O III 51.81 mu
C  68/ O III 4959-5007              69/ O III 2321
C  70/ O III 1663                   71/ O III 4363
C  72/ O III 3133 (Bowen)           73/ O III 3444 (Bowen)
C  74/ O IV 554.4                   75/ O IV 609.4
C  76/ O IV 789.4                   77/ O IV 25.89 mu
C  78/ O IV 1401                    79/ O IV 76.10 mu
C  80/ O IV 54.20 mu                81/ O V 629.7
C  82/ O V 1218                     83/ O VI 1034
C  84/ S I 25.25 mu                 85/ S I 56.31 mu
C  86/ S I 10824-11309              87/ S I 4509-90
C  88/ S I 7727                     89/ S II 6716-31
C  90/ S II 4069-76                 91/ S II 10287-370
C  92/ S III 1197.5                 93/ S III 33.5 mu
C  94/ S III 18.7 mu                95/ S III 9069-9531
C  96/ S III 3722                   97/ S III 1713-29
C  98/ S III 6312                   99/ S IV 1069
C 100/ S IV 10.52 mu               101/ S IV 1393
C 102/ S IV 29.07 mu               103/ S IV 18.28 mu
C 104/ S V 786.5                   105/ S VI 937.1
C 106/ Ne III 15.6 mu              107/ Ne III 36.0 mu
C 108/ Ne III 3869-3968            109/ Ne III 1815
C 110/ Ne III 3342                 111/ Ne IV 2422-24
C 112/ Ne IV 1602                  113/ Ne IV 4714-24
C 114/ Ne V 24.3 mu                115/ Ne V 14.3 mu
C 116/ Ne V 3346-3426              117/ Ne V 1575
C 118/ Ne V 1137-46                119/ Ne V 2975
C 120/ Ne VIII 773.7               121/ Si II 34.8 mu
C 122/ Si II 2341                  123/ Si III 1206
C 124/ Si III 1892                 125/ Si IV 1394
C 126/ Na IV 3242-3417             127/ Na IV 1504-29
C 128/ Na IV 2805                  129/ Mg II 2796-2802
C 130/ Mg V 2783-2993              131/ Mg V 1294-1324
C 132/ Mg V 2418                   133/ Mg VII 2510-2630
C 134/ Mg VII 1190                 135/ Mg VII 2262
C 136/ Al II 1671                  137/ Al II 2660-69
C 138/ Ar III 7136-7751            139/ Ar III 3005-3109
C 140/ Ar III 5192                 141/ Ar IV 4711-40
C 142/ Ar IV 2854-68               143/ Ar IV 7171-7331
C 144/ Ar V 6435-7006              145/ Ar V 2691
C 146/ Ar V 4626                   147/ Ca II 3934-68
C 148/ Ca II 7291-7324             149/ Ca V 5311-6429  
C 150/ Ca V 2281-2414              151/ Ca V 3999
C 152/ Ca VII 4572-5622            153/ Ca VII 2112-2226
C 154/ Ca VII 3687                 155/ Fe X 6374
C 156/ Fe VII 5722 (corrected March 18, 2009)
C 156/ Fe XIII 2579-3388
C
C - Fe II-IV,VII lines may be calculated separately.
C - Only the strongest (UV & optical) Na, Mg, Al, Ar & Ca-lines 
C   listed. (More are included in the code.)
C
C******************************************************************
C
      IMPLICIT REAL*8(A-H,O-Z)
      character*80 fil1,fil2,fil3,fil4,fil5,fil6,fil7,fil8,fil9,
     &fil10,fil11,fil12
C
      parameter (md=75,mmatr=80)
C
      COMMON/ABS/DRQ,DMDTQ,R1Q,D2,D2S,DRS,NSHELL,ISABS
      COMMON/FRE/JMIN,JJ
      COMMON/YIELDA/YLA(md),YB(md),YO2(md),YA(md),IYIELD,KYIELD
      COMMON/PLUMP/XMIN,STE1,TOL1,TOL2,TOL13,KWRI                       
      COMMON/VAIJU/IJU,IJUMAX
      COMMON/LINE/XLINE
      COMMON/QSOM/QSO
      COMMON/SIMPL/ISIMP,IGAMM,GAHE,CR
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      COMMON/SPH/ISPH
      COMMON/HYDROS/HSCALE,DEN1
      COMMON/CHG/CHGH,CHGHE,CHTE
      COMMON/THER/A1,B1,TIN,E10,E20,PHMIN
      COMMON/MOD/DMODE,MODE
      COMMON/ABUN/AB(15)
      COMMON/A12/ITER
      COMMON/A10/BETA,AV
      COMMON/NFE/NFE1,NFE2,NPR
      COMMON/VARI11/ICOM1,ICOM2,TEMP11,TEMP12,TEMPIN
      COMMON/VARI12/TOLA,TOLB,TOLC,JAPEX,JCOMMAX,IMAXUS
      COMMON/TIMES/DELTT,DEPE,MAXT
      COMMON/NHY/NH
      COMMON/CHESIM/INNN
C
      BETA=12.
      AV=1.
      TAUS=0.05
      FLUM=2.5D-6
      TEX=2.5D3
      TSH=1.84D5
      MODE=1
      DMODE=1.
      XLINE=-1.
      NPRINT=999
      IPAR=-1
      A1=1.D4
      B1=1.D8
      CHGH=0.2
      CHGHE=0.2
      CHTE=0.22
      IOMAX=0
 100  FORMAT(1X,8E11.4)
 110  FORMAT(1X,10I4)
      dmdt=0.91d-1
      DMDTQ=DMDT
 120  FORMAT(1X,7E11.4,I5)
      QSO=-1.
      E20=0.02
      E10=1.D-9
      CHG=0.2
C
      OPEN(13,FILE='inpshelld_3e4_new',STATUS='UNKNOWN')
      read(13,9277)fil1
      read(13,9277)fil4
      read(13,9277)fil5
      read(13,9277)fil8
      read(13,9277)fil9
      read(13,9277)fil10
      read(13,9277)fil11
      read(13,9277)fil12
9277  format(a)
      READ(13,*)toutmin,mshell,den1,rs15,ishiwo,zzcosm,xhexh,
     &xnxc,xnxo,teinn,mwind,fudfed,burmul,den2,mcut,den3,mcut3
C
      OPEN(6,FILE=fil1,STATUS='UNKNOWN')
      OPEN(9,FILE=fil5,STATUS='UNKNOWN')
      OPEN(11,FILE='recdata_big',STATUS='old')
      OPEN(14,FILE=fil4,STATUS='UNKNOWN')
      open(28,file=fil10,STATUS='UNKNOWN')
      open(29,file='feIV.22b',STATUS='UNKNOWN')
      open(30,file=fil8,STATUS='UNKNOWN')
      open(31,file=fil9,STATUS='UNKNOWN')
      open(32,file='hcoll_new.dat',STATUS='UNKNOWN')
      open(33,file='hecoll.dat',STATUS='UNKNOWN')
      open(34,file='feII.16b',STATUS='UNKNOWN')
      open(35,file=fil11,STATUS='UNKNOWN')
      open(36,file=fil12,STATUS='UNKNOWN')
      open(41,file='recrm.ca',STATUS='UNKNOWN')
      open(42,file='recrm.ni',STATUS='UNKNOWN')
      open(43,file='recrm.ox',STATUS='UNKNOWN')
      open(44,file='recrm.ne',STATUS='UNKNOWN')
      open(45,file='recrm.si',STATUS='UNKNOWN')
      open(46,file='recrm.su',STATUS='UNKNOWN')
      open(47,file='recrm.fe',STATUS='UNKNOWN')
      open(50,file='sltemp',STATUS='UNKNOWN')
      open(54,file='cout11',STATUS='UNKNOWN')
      open(55,file='nout11',STATUS='UNKNOWN')
      open(56,file='oout11',STATUS='UNKNOWN')
      open(57,file='naout11',STATUS='UNKNOWN')
      open(58,file='mgout11',STATUS='UNKNOWN')
      open(59,file='feout11',STATUS='UNKNOWN')
      open(60,file='alout11',STATUS='UNKNOWN')
      open(61,file='arout11',STATUS='UNKNOWN')
      open(62,file='caout11',STATUS='UNKNOWN')
      open(63,file='he1_cross',STATUS='UNKNOWN')
      open(64,file='nahar_rec1',STATUS='UNKNOWN')
      open(65,file='slcrate',STATUS='UNKNOWN')
      open(66,file='nommxf.ph',STATUS='UNKNOWN')
      open(67,file='ut87af.ph',STATUS='UNKNOWN')
      open(68,file='nmxfso17.ph',STATUS='UNKNOWN')
C
      pi=3.14159d0
      JMIN=-3
      JJ=57
      if(ishiwo.eq.0.or.ishiwo.ge.3) tsec=6.912000001d3
      if(ishiwo.eq.1) tsec=8.640000001d3
      if(ishiwo.eq.2) tsec=0.0d0
      if(ishiwo.eq.6) tsec=5.879500001d3
      if(ishiwo.eq.7) tsec=8.151000001d3
      if(ishiwo.eq.8) tsec=6.56208501d3
      if(ishiwo.le.2) shmass=1.5d0
C      if(ishiwo.eq.3) shmass=0.5d0
      if(ishiwo.eq.3) shmass=2.2d0
      if(ishiwo.ge.4) shmass=4.0d0
      if(ishiwo.eq.6) shmass=3.3d0
      if(ishiwo.eq.7) shmass=1.75d0
      if(ishiwo.eq.8) shmass=2.2d0
      shmass=burmul*shmass
      imaxus=mshell-5
C--  adjust shmass for sphericity
      dshell=(rs15/dble(mshell))*((1.d0+(2.8448d11*shmass)/(den1*
     &rs15**3))**.333333-1.d0)
      dshell2=(rs15/dble(mshell))*((1.d0+(2.8448d11*shmass)/(den2*
     &rs15**3))**.333333-1.d0)
      dshell3=(rs15/dble(mshell))*((1.d0+(2.8448d11*shmass)/(den3*
     &rs15**3))**.333333-1.d0)
C     dshell2=(shmass2*9.4826d10)/(dble(mwind)*den2*rs152**2)
      TI=TSEC/3.6D3
      tdays=tsec/8.64d4
      INNN=3
      IN=INNN
      if(ishiwo.eq.0) CALL NOMSHI(TDAYS,RPA15,TEB)
      if(ishiwo.eq.1) CALL WOS10L(TDAYS,RPA15,TEB)
      if(ishiwo.eq.2) then
       RPA15=1.d-5
       TEB=1.5D5
      endif
      if(ishiwo.eq.3) CALL ENSBUR(TDAYS,RPA15,TEB,qtc)
      if(ishiwo.eq.4) CALL ENSBUR2(TDAYS,RPA15,TEB,qtc)
      if(ishiwo.eq.5) CALL ENSBUR3(TDAYS,RPA15,TEB,qtc)
      if(ishiwo.ge.6) CALL BARBLI1(TDAYS,RPA15,qtc,ishiwo)
C     DEN1=3.D6*DMDT/RS15**2
C     RS15=1.3D3
c      DEN1=1.0D4
C     rs15=rpa15
      R1=RS15
      BETA=2.D0
      AV=0.0D0
      TAUS=1.D-2
      MAXT=10000
      STE1=1.d-1
      TOL1=2.d-2
      TOL2=1.d-1
C
      AB(1)=(1.d0-xz-AB(6)-AB(7)-AB(8)-AB(9)-AB(10)-AB(11)-AB(12)-
     &AB(13)-AB(14))/(1.d0+xhexh)
      AB(2)=AB(1)*xhexh
C xzcosm = X(C+N+O) for cosmic abundances (Allen -73)
C xz = [X(C+N+O) for SN 1987A] / xzcosm
      xzcosm=9.97d-4
      xzfix=0.3d0
      xz=zzcosm*xzcosm
      xzcorr=zzcosm/xzfix
      ABNI=xz/(1.d0+1.d0/xnxc+1.d0/xnxo)
      ABCA=ABNI/xnxc
      ABOX=ABNI/xnxo
      AB(3)=ABOX
      AB(4)=ABCA
      AB(5)=ABNI
C     AB(3)=0.62d0*AB(3)
C     AB(4)=2.10d0*AB(4)
C     AB(5)=1.28d0*AB(5)
C - Sulphur estimate from Pagel, Edmunds & Fosbury 1978 (MNRAS, 184, 569)
C   (lg O/S = 1.6 in both LMC and the Galaxy. Scaled to the cosmic abundance 
C    by Allen.)
C      AB(&)=6.61d-6
      AB(6)=4.5D-6*xzcorr
      AB(6)=AB(6)*0.89d0
C - Neon from mean value of Table I & II in Barlow (IAU-Symp on PNe, 1989).
C   (lg Ne/H = -4.32, scaled to absolute abundance. 0.58 x Allen's cosmic ab.)
      AB(7)=4.41D-5*xzcorr
C -- Iron, lg (Fe/H)= -4.58 for LMC (Russell, Bessell and Dopita), scaled to
C     absolute abundance. (0.66 x Allen's cosmic ab.)
      AB(8)=2.42D-5*xzcorr
C -- Silicon assumed to follow sulphur, i.e., O/Si (Galaxy) = O/Si (LMC).
C     (0.40 x Allen's cosmic abundances.)
      xzsidep=0.5d0
      AB(9)=1.39D-5*xzcorr*xzsidep
C -- Na, Mg, Al, Ar, Ca x 0.4 of solar abundances (Allen '73).
      AB(10)=7.11D-7*xzcorr
      AB(11)=1.05D-5*xzcorr
      AB(12)=9.82D-7*xzcorr
      AB(12)=AB(12)*0.87d0
      AB(13)=2.52D-6*xzcorr
      AB(14)=7.98D-7*xzcorr
C
      AB(1)=(1.d0-xz-AB(6)-AB(7)-AB(8)-AB(9)-AB(10)-AB(11)-AB(12)-
     &AB(13)-AB(14))/(1.d0+xhexh)
      AB(2)=AB(1)*xhexh
      AB(5)=1.05d0*2.15d0*AB(5)
      AB(3)=1.05d0*AB(3)
      AB(6)=1.05d0*2.2d0*AB(6)
      AB(7)=1.05d0*1.4d0*AB(7)
      AB(13)=1.05d0*0.50d0*AB(13)
      AB(14)=1.05d0*4.00d0*AB(14)
C
      XMIN=1.D-4
      QSO=-999.
      XLINE=-1.
      FLUM=1.D-14
      TEX=2.5D3
      TSH=1.84D5
      TOL13=0.05
      TEMPIN=2.0D4
      MODE=-1
      DMODE=-1.
      NMAX=0
      MAX=13
      ITER=0
      NPRINT=0
      IPAR=1
      ISPH=1
      IGAMM=0
      GAHE=0.
      CR=0.
      ISIMP=0
      A1=8.D3
      B1=1.25D4
      PHMIN=10.
      TIN=1.8D4
      E10=1.D-5
      E20=0.02
      CHGH=0.2
      CHGHE=0.2
      CHTE=0.2
      IOMAX=-1
      MQMAX=3
      NPR=0
      ITERM=0
      ISPEC=0
      TINIT=0.
      XIN=1.
      INP=2
      EPS=1.D-30
      ICOM1=3
      ICOM2=1
      TEMP11=0.9
      TEMP12=1.1
      TOLA=0.15
      TOLB=1.015
      TOLC=1.015
      JAPEX=2
      JCOMMAX=10
      TRULS=2.
      IJUMAX=1
C
C     CHANGE DEN1 TO THE TOTAL NUMBER OF NUCLEI PER VOLUME
C
      AMEAN=AB(1)+4.*AB(2)+12.*AB(4)+14.*AB(5)+16.*AB(3)
     &+32.1*AB(6)+20.2*AB(7)+55.85*AB(8)+28.086*AB(9)+
     &22.99*AB(10)+24.305*AB(11)+26.982*AB(13)+39.948*AB(13)+
     &40.08*AB(14)
      DEN1=DEN1/AMEAN
      DEN2=DEN2/AMEAN
      DEN3=DEN3/AMEAN
      CALL TRANS(NMAX,MAX,IOMAX,NPRINT,IPAR,TAUS,FLUM,TEX,TSH,DR,
     &R1,MQMAX,ITERM,ISPEC,tinit,xin,inp,TI,dshell,cbur,tbur,dtbur,
     &ishiwo,teinn,toutmin,denwd,mwind,mshell,dshell2,den2,fudfed,
     &burmul,mcut,dshell3,den3,mcut3)
C
      STOP
      END
C****************************************************************
C****************************************************************
      SUBROUTINE TRANS(NMAX,MAX,IOMAX,NPRINT,IPAR,TAUS,FLUM,TEX,
     &TSH,DR,R1,MQMAX,ITERM,ISPEC,tinit,xin,inp,TI,dshell,cbur,
     &tbur,dtbur,ishiwo,teinn,toutmin,denwd,mwind,mshell,dshell2,
     &den2,fudfed,burmul,mcut,dshell3,den3,mcut3)
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      parameter (md=75,levhmx=55,levh=levhmx+1,levhemx=71,
     &           levhe=levhemx+1,nnhmax=10,mmatr=80,levhe1=16)
C
      common/hydaij/ahydij(levhmx,levhmx),ahydsm(levhmx)
      common/helaij/ahelij(levhmx,levhmx),ahelsm(levhmx)
      common/hrecij/rechij(25,levhmx),terech(25),rechsum(25)
      common/herecij/recheij(25,levhmx),tereche(25),rechesum(25)
      common/heiila/heii21
      common/folines/redc(10),redn(10),redo(10),rede(10),reds(10),
     &               forc(10),forn(10),foro(10),fore(11),fors(10)
      common/folines2/redsi(10),UVSI(md,4),forsi(10)
      common/folines3/redna(2),redmg(4),redar(4),redca(4),
     &                forna(6),formg(11),forar(10),forca(9),
     &                foralu(2),fecol10,forbfe1
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      COMMON/FEIIC/OMFE2(16,16),AIJFE2(16,16),DLFE2(16,16) 
      common/FEIIC2/admpfe2(16,16),asmfe2(16)
      common/columns/columi(117),colum(117)
      common/column/colni(md,117),colno(md,117)
      common/newhy/hy2,hy3,hyco2,hyco3,hytot
      common/hyddt/qxh(md,2,levh),hradm(levh,levh),c(levh,levh)
      common/hedt/qxhe(md,2,levhe),heradm(levhe,levhe),
     &            hecoll(levhe,levhe)
      common/arglin/farg(md,6)
      COMMON/FBFE7/FEADD(17)
      COMMON/BWLOSS/FOIII(md,8)
      COMMON/COLCNO/COLLC,COLLN,COLLO,COOLFR(175)
      COMMON/HEOPT/OPTHEL(10,md)
      COMMON/YIELDA/YLA(md),YB(md),YO2(md),YA(md),IYIELD,KYIELD
      common/yieldc/ela(md),plinla(md),pconla(md),poo(md)
      COMMON/UVSUVS/UVHELI(md,1),UVCARB(md,6),UVOXYG(md,5),UVNITR(md,6)
      COMMON/PHOTOS/TEB,RPA15
      COMMON/CHESIM/INNN
      COMMON/TRES/EL(117),EK(117)
      COMMON/LUMI/FSURF(-13:75)
      COMMON/RSTART/RQ11
      COMMON/IND/I
      COMMON/TAUXX/TAXA(md,-13:75)
      COMMON/ABS/DRQ,DMDTQ,R1Q,D2,D2S,DRS,NSHELL,ISABS
      COMMON/SHELLX/RSF15,DXU,DXU1
      COMMON/VARI11/ICOM1,ICOM2,TEMP11,TEMP12,TEMPIN
      COMMON/PHONUM/V1,VV1
      COMMON/VARI12/TOLA,TOLB,TOLC,JAPEX,JCOMMAX,IMAXUS
      COMMON/VAIJU/IJU,IJUMAX
      COMMON/ABC/AL(7),ALN(8),ALO(9),ALC(7),ALS(15),ALMG(11),
     &           ALAL(12),COO(8),COC(6),CON(7)
      common/abc2/alneo(9),alsul(11),cone(8),cosu(10),zneo(8),
     &            zsul(10)
      common/abc3/alfe(16),cofe(15),zfe(15),alsi(11),cosi(10),zsi(10)
      common/abc4/alna(10),alar(11),alca(11),cona(9),zna(9),
     &            comg(10),zmg(10),coal(11),zal(11),coar(10),zar(10),
     &            coca(10),zca(10)
      COMMON/REC/AL2(7)
      COMMON/HESTUFF/ALDHE,CHEO,CHE
      COMMON/FRE/JMIN,JJ
      COMMON/SPH/ISPH
      COMMON/THER/A1,B1,TIN,E10,E20,PHMIN
      COMMON/MOD/DMODE,MODE
      COMMON/SPOT/OTSP(7)
      COMMON/MPCONST/CC1(-13:76),CC2(-13:76),CC3(-13:76)
      COMMON/ASP1/DAYMAX,DAYIN,JASPP,IASP
      COMMON/COL/RE(4),FF,HY,HE,C131,C142,C151,C161,C231,C241,COH,
     &           COHE,C351,C321,C331,C341,C222,C232,C221,C332,C441
      COMMON/COOLVA/CEX(15),C311,C322,C111,C121,COCARB,CONI,COOX
      COMMON/EQUIV/W(30),CIN(30),FB(md,15),TWOP(md)
      COMMON/EQUIH/WEH(40)
      COMMON/T/TES,SS
      COMMON/REHEL/REHE21,REHE22,REC31,REN41
      COMMON/PHY/DEN(md)
      COMMON/DXA/DX(md)
      COMMON/CHG/CHGH,CHGHE,CHTE
      COMMON/HEA/CO,PH,PHEO,PHEI,PO(8),PC(6),PN(7),PMG,PSI,PFE
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)
      COMMON/PHQ/ZE(117),GE(117),ZK(117),GET(117)
      COMMON/ION/XB(md,117)
      COMMON/ABU/XA(2,117)
      COMMON/OTS/NOTS
      COMMON/DIF/TAU(md,-13:75)
      COMMON/TEM/TE(md)
      COMMON/TIMES/DELTT,DEPE,MAXT
      COMMON/ICOMMO/ICOM,NCOUN
      COMMON/PELEC/QXB(md,2,131),XU(131),JUP
      COMMON/HYDROS/HSCALE,DEN1
      COMMON/RADIE/R(md)
      COMMON/ELEC/DEL(md),MEL
      COMMON/EDAYY/TDAY
      COMMON/SIK/SK(117,-13:75)
      COMMON/FT/FTOT(1,-13:75)
      COMMON/ABUN/AB(15)
      common/abun2/abunda(117)
      COMMON/NHY/NH
      COMMON/BOLD/TOLD
      COMMON/ZAETA/ZA(8),ZB(6),QXA1,ZC(7)
      COMMON/HPOP/XN1,XN2,XN3
      COMMON/A5/TAUE,ALFA,EN,TSN,XL40,TEXA,R15
      COMMON/A7/C3,C33
      COMMON/A10/BETA,AV
      COMMON/A11/R11,CV,FLUX
      COMMON/PLUMP/XMIN,STE1,TOL1,TOL2,TOL13,KWRI                       
      COMMON/ITRIM/ITEX
      COMMON/A12/ITER
      COMMON/HRAT/XEL,HYR,HEAT,COOL,RADR
      COMMON/HEB/PHET,PCT,PNT,POXT,PMGT,PSIT,PFET,HEATT
      COMMON/SPECT/TEL,FD(1,1),F0(-13:75),FEDD(md,-13:75),IPARA
      COMMON/DTAU/FLUXED(md,-13:75)
      COMMON/CHION/CIONH,HYCO,HEI,HEII,HEII32,HEIIBW
      COMMON/ABSH/ABS12(md),ABS13(md),ABSHE2(md),ABSHE3(md),
     &            ABSO1(md),ABSHE4(MD),ABSHE5(MD),ABSHE6(MD),
     &            ABSHE7(MD),ABSHE8(MD)
      COMMON/ABSCNO/ABSEUV(md,99)
      COMMON/BOWHEL/BWHELI(md,2)
      COMMON/ABSHN/ABSNV(md)
      COMMON/ADDL/ADDLIN(md,75)
      common/radabs/rfix(md),rbound(md),dxabs(3,md,md)
      common/euvlin/w1eu(99),eeveu(99),alqeu(99),beqeu(99),tmeu(99),
     &              ioeu(99),iityp(99),nline,jqeu(99),jweu(99)
      common/euvli2/rcont(79),leuv(79),iskipeu(79)
      common/fe4/faij(22,22),fome(22,22)
      common/twexps/twoexp(-13:75)
      common/oldtee/TEALD(md)
      common/aggarw/teagh(9),teaghe(10),agh(15,15,9),aghe(10,10,10)
      common/anglemu/xnjmu(100),njmu
      common/robcum/fro21,fro41,frs21,frs41,frr21,frr41
      COMMON/NVLEV/x1n5(md),p21in(md),p21out(md)
      COMMON/RADADD/SQW(md),TEMPST(md),WARM(md),DENT(md),DENTK(md),
     &              COLD(md),HYD(md),DXA(md),tsecc(md),RAAA(md),
     &              RAAB(md),RAAC(md),STOLIN(md,200),frro21(md),
     &              frro41(md),frrs21(md),frrs41(md),icomp(md),
     &              itee(md)
      common/recnah/tenah(81),alnah(29,81)
      common/nontherm/zexc(2),inonth(117)
C
      DIMENSION OPTEUV(26),CDC(15),ia(26),lleuv(79),sumlin(200),
     &          inont(117)
C
      data lleuv/17,13,11,6,6,5,5,4,4,3,3,2,2,2,2,14,12,10,9,7,7,6,
     &5,3,3,3,17,16,16,16,16,16,16,16,16,15,15,15,15,15,15,15,15,15,
     &15,14,14,14,13,12,12,11,10,10,10,10,9,14,13,12,11,11,10,27,26,
     &25,25,25,25,12,12,11,10,9,9,9,8,8,5/
      data inont/1,1,1,0,0,0,0,0,0,0,0,1,1,1,1,0,0,1,1,0,0,0,0,0,1,
     &1,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,1,0,0,0,0,0,0,0,0,0,0,0,0,
     &0,1,1,0,0,0,0,0,0,0,0,1,1,0,0,0,0,0,0,0,1,1,0,0,0,0,0,0,0,0,1,
     &1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,1,0,0,0,0,0,0,0,0/
C
      do iaaa=1,117
       inonth(iaaa)=inont(iaaa)
      enddo
      do iaaa=1,26
       ia(iaaa)=0
      enddo
      do iaaa=-13,75
       twoexp(iaaa)=1.d0
      enddo
      do iaaa=1,99
       eeveu(iaaa)=0.0d0
       w1eu(iaaa)=0.0d0
       alqeu(iaaa)=0.0d0
       beqeu(iaaa)=0.0d0
       tmeu(iaaa)=0.0d0
       jqeu(iaaa)=0
       jweu(iaaa)=0
       ioeu(iaaa)=0
       iityp(iaaa)=0
       if(iaaa.le.79) leuv(iaaa)=0
      enddo
      nline=94
      IA(1)=1                                                           
      IA(2)=2                                                           
      IA(6)=4                                                           
      IA(7)=5                                                           
      IA(8)=3                                                           
      IA(10)=7                                                          
      IA(16)=6                                                          
      IA(26)=8
      IA(14)=9
      IA(11)=10
      IA(12)=11
      IA(13)=12
      IA(18)=13
      IA(20)=14
      DO 5001 KK=1,NLINE                                                
      READ(11,*)W1EU(KK),JQEU(KK),JWEU(KK),IITYP(KK),EEVEU(KK),
     &ALQEU(KK),BEQEU(KK),TMEU(KK),IOEU(KK)
      JQEU(KK)=IA(JQEU(KK))
      if(kk.le.79) then
       leuv(kk)=lleuv(kk)
      endif
 5001 CONTINUE
C  ---- Recomb.data from Nahar (Oct. '95; April '96; July '97)
      do inah=1,5
	do jnah=1,81
          if(inah.le.2) then
            read(64,*)tenah(jnah),(alnah(5*(inah-1)+knah,jnah),
     &	                           knah=1,5)
	    do knah=1,5
	      alnah(5*(inah-1)+knah,jnah)=
     & 	                       dlog10(alnah(5*(inah-1)+knah,jnah))
	    enddo
          elseif(inah.eq.3) then
            read(64,*)tenah(jnah),(alnah(10+knah,jnah),knah=1,6)
	    do knah=1,6
	      alnah(10+knah,jnah)=dlog10(alnah(10+knah,jnah))
	    enddo
          elseif(inah.eq.4) then
            read(64,*)tenah(jnah),(alnah(16+knah,jnah),knah=1,6)
	    do knah=1,6
	      alnah(16+knah,jnah)=dlog10(alnah(16+knah,jnah))
	    enddo
          elseif(inah.eq.5) then
            read(64,*)tenah(jnah),(alnah(22+knah,jnah),knah=1,7)
	    do knah=1,7
	      alnah(22+knah,jnah)=dlog10(alnah(22+knah,jnah))
	    enddo
	  endif
	enddo
      enddo
C  ---- Fe II data ---
       do ii=1,16
       asmfe2(ii)=0.0d0
        do jjj=1,16
        omfe2(ii,jjj)=0.0d0
        aijfe2(ii,jjj)=0.0d0
        admpfe2(ii,jjj)=0.0d0
        dlfe2(ii,jjj)=0.0d0
        enddo
       enddo
       do ii=2,16
        do jjj=1,ii-1
        read(34,*)iq,jq,dlfe2(ii,jjj),aijfe2(ii,jjj),omfe2(jjj,ii)
        asmfe2(ii)=asmfe2(ii)+aijfe2(ii,jjj)
        enddo
       enddo
       do ii=2,16
        do jjj=1,ii-1
        if(aijfe2(ii,jjj).ne.0.0d0) then
         admpfe2(ii,jjj)=asmfe2(ii)+asmfe2(jjj)
        endif
        enddo
       enddo
C  ---- Fe IV data ---
       do 603 ii=1,22
        do 604 kk=1,22
        faij(ii,kk)=0.0d0
        fome(ii,kk)=0.0d0
  604   continue
  603  continue
       do 601 ii=2,22
        do 602 kk=1,ii-1
        read(29,*)iq,jq,dlair,faij(ii,kk),fome(kk,ii)
  602   continue
  601  continue
C
C  H I and He II A-values and grid for recombination coeffs.
C
       call HYDIJ
C
C  H I and He II effective collision strengths
C
       do 5391 kh=1,15
        do 5392 ki=1,15
         do 5393 kj=1,9
         agh(kh,ki,kj)=0.0d0
 5393    continue
 5392   continue
 5391  continue
       do 5394 kh=1,10
        do 5395 ki=1,10
         do 5396 kj=1,10
         aghe(kh,ki,kj)=0.0d0
 5396    continue
 5395   continue
 5394  continue
      read(32,*)(teagh(kh),kh=1,8)
       do 5401 kh=2,15
       read(32,*)(agh(1,kh,ki),ki=1,8)
 5401  continue
       do 5402 kh=4,15
       read(32,*)(agh(2,kh,ki),ki=1,8)
 5402  continue
       do 5403 kh=4,15
       read(32,*)(agh(3,kh,ki),ki=1,8)
 5403  continue
       do 5404 kh=7,15
       read(32,*)(agh(4,kh,ki),ki=1,8)
 5404  continue
       do 5405 kh=7,15
       read(32,*)(agh(5,kh,ki),ki=1,8)
 5405  continue
       do 5406 kh=7,15
       read(32,*)(agh(6,kh,ki),ki=1,8)
 5406  continue
       do 5407 kh=11,15
       read(32,*)(agh(7,kh,ki),ki=1,8)
 5407  continue
       do 5408 kh=11,15
       read(32,*)(agh(8,kh,ki),ki=1,8)
 5408  continue
       do 5409 kh=11,15
       read(32,*)(agh(9,kh,ki),ki=1,8)
 5409  continue
       do 5410 kh=11,15
       read(32,*)(agh(10,kh,ki),ki=1,8)
 5410  continue
C      read(32,*)(agh(11,13,ki),ki=1,8)
C      read(32,*)(agh(11,14,ki),ki=1,8)
C      read(32,*)(agh(11,15,ki),ki=1,8)
C      read(32,*)(agh(12,14,ki),ki=1,8)
C      read(32,*)(agh(12,15,ki),ki=1,8)
C      read(32,*)(agh(13,15,ki),ki=1,8)
       read(33,*)(teaghe(kh),kh=1,10)
       do 9411 kh=2,10
      read(33,*)(aghe(1,kh,ki),ki=1,10)
 9411  continue
       do 9412 kh=4,10
       read(33,*)(aghe(2,kh,ki),ki=1,10)
 9412  continue
       do 9413 kh=4,10
       read(33,*)(aghe(3,kh,ki),ki=1,10)
 9413  continue
       do 9414 kh=6,10
       read(33,*)(aghe(4,kh,ki),ki=1,10)
 9414  continue
       do 9415 kh=7,10
       read(33,*)(aghe(5,kh,ki),ki=1,10)
 9415  continue
       do 9416 kh=7,10
       read(33,*)(aghe(6,kh,ki),ki=1,10)
 9416  continue
      read(33,*)(aghe(7,9,ki),ki=1,10)
      read(33,*)(aghe(7,10,ki),ki=1,10)
      read(33,*)(aghe(8,10,ki),ki=1,10)
C
      CALL TRINIT
C
C     *************************************************************
C     EVALUATION OF CROSS SECTIONS FOR EACH ENERGY
C     (also define energy vector in CROSS.)
C     *************************************************************
C
      CALL CROSS
C
      IMAX=IMAXUS
      do 8406 i=1,imaxus
      itee(i)=0
       do k=1,131
        qxb(i,1,k)=1.d-30
        qxb(i,2,k)=0.0d0
       enddo
       do k=1,levhe
        qxhe(i,1,k)=1.d-30
        qxhe(i,2,k)=0.0d0
       enddo
       do k=1,levh
        qxh(i,1,k)=1.d-30
        qxh(i,2,k)=0.0d0
       enddo
      qxb(i,1,1)=1.d0
      qxh(i,1,1)=1.d0
      qxhe(i,1,1)=1.d0
      qxb(i,1,3)=1.d0
      qxb(i,1,6)=1.d0
      if(i.le.60) then
C	qxb(i,1,1)=1.d-30
C	qxb(i,1,2)=1.0d0
C        qxb(i,1,3)=0.d0
C        qxb(i,1,4)=1.d0
C         qxh(i,1,1)=1.d-30
C        qxhe(i,1,1)=0.d0
C         qxh(i,1,levh)=1.d0
C        qxhe(i,1,levhe1+1)=1.d0
      endif
      qxb(i,1,13)=1.d0
      qxb(i,1,22)=1.d0
      qxb(i,1,33)=1.d0
      qxb(i,1,41)=1.d0
      qxb(i,1,50)=1.d0
      qxb(i,1,66)=1.d0
      qxb(i,1,77)=1.d0
      qxb(i,1,87)=1.d0
      qxb(i,1,98)=1.d0
      qxb(i,1,110)=1.d0
      qxb(i,1,121)=1.d0
      te(i)=teinn
      if(i.le.60) then
	te(i)=1.d2
C	te(i)=1.d4
      endif
      do k=1,117
       xb(i,k)=0.0d0
      enddo
      XB(I,1)=QXB(I,1,1)
      XB(I,2)=QXB(I,1,3)
      XB(I,3)=QXB(I,1,4)
      DO 372 K=4,6
  372 XB(I,K)=QXB(I,1,K+14)
      XB(I,7)=QXB(I,1,17)
      DO 1938 K=8,11
 1938 XB(I,K)=QXB(I,1,K)
      XB(I,12)=QXB(I,1,6)
      XB(I,13)=QXB(I,1,7)
      DO 7902 K=14,17
 7902 XB(I,K)=QXB(I,1,K-1)
      DO 7909 K=18,24
 7909 XB(I,K)=QXB(I,1,K+15)
      DO 5609 K=25,34
 5609 XB(I,K)=QXB(I,1,K-3)
      DO 3909 K=35,42
 3909 XB(I,K)=QXB(I,1,K+6)
      DO 3910 K=43,57
 3910 XB(I,K)=QXB(I,1,K+7)
      DO 3911 K=58,67
 3911 XB(I,K)=QXB(I,1,K+8)
      DO K=68,76
         XB(I,K)=QXB(I,1,K+9)
      ENDDO
      DO K=77,86
         XB(I,K)=QXB(I,1,K+10)
      ENDDO
      DO K=87,97
         XB(I,K)=QXB(I,1,K+11)
      ENDDO
      DO K=98,107
         XB(I,K)=QXB(I,1,K+12)
      ENDDO
      DO K=108,117
         XB(I,K)=QXB(I,1,K+13)
      ENDDO
      TEMPST(I)=TEMPIN
      x1n5(i)=1.d0
 8406 continue
C 
      DO 7678 IJ=1,117
      ABUNDA(ij)=AB(3)                                                     
      IF(IJ.EQ.1) ABUNDA(ij)=AB(1)                                         
      IF(IJ.EQ.2.OR.IJ.EQ.3) ABUNDA(ij)=AB(2)                              
      IF(IJ.GE.8.AND.IJ.LE.13) ABUNDA(ij)=AB(4)                            
      IF(IJ.GE.18) ABUNDA(ij)=AB(5)                                        
      IF(IJ.GE.25) ABUNDA(ij)=AB(6)                                        
      IF(IJ.GE.35) ABUNDA(ij)=AB(7)                                        
      IF(IJ.GE.43) ABUNDA(ij)=AB(8)
      IF(IJ.GE.58) ABUNDA(ij)=AB(9)
      IF(IJ.GE.68) ABUNDA(ij)=AB(10)
      IF(IJ.GE.77) ABUNDA(ij)=AB(11)
      IF(IJ.GE.87) ABUNDA(ij)=AB(12)
      IF(IJ.GE.98) ABUNDA(ij)=AB(13)
      IF(IJ.GE.108) ABUNDA(ij)=AB(14)
 7678 CONTINUE
C
      njmu=25
      dnjmu=1.d0/(dble(njmu-1))
       do nnjmu=1,njmu
       xnjmu(nnjmu)=dnjmu*dble(nnjmu-1)
       enddo
C     ***********************************************************
C     *****
C     INITIALIZE PARAMETERS
C     *****
C     ***********************************************************
      IMIN=1
      MAX=IMAX
C     IMAX=IMIN
c     write(14)imin,imax
      PI=3.14159D0
c cf!
      tout=toutmin
      TOLD=TIN
      IOITER=0
C
C     R15=PHOTOSPHERIC RADIUS IN 1E15 CM
C     R15 EQ.START VALUE OF COUNTINGS IN 1E15 CM
C
      R15=R1
      NUM=0
      NOTS=1
      DO 120 K=1,117
      ZE(K)=0.0D0
      GE(K)=0.0D0
      XA(1,K)=0.0D0
  120 XA(2,K)=0.0D0
 1672 continue
C--------------------------------------------------------
C---  initialize the relative abundances of the ions
      DO 7365 K=1,117
      XA(2,K)=XB(2,K)
      XA(1,K)=XB(2,K)
 7365 CONTINUE
C--------------------------------------------------------
C -- On-the-spot ?
C
      OTSP(1)=0.
      OTSP(2)=0.
      OTSP(3)=0.
 8360 CONTINUE
      JUP=2
      do 1928 i=1,imaxus
      CALL ELDENS(3)
      DEL(I)=DEPE
 1928 CONTINUE
      MELECT=1
      I=1
C
C     DENSITY AT INNER BOUNDARY
C
c     **************************************************************
C     *****
C     R(1)= RADIUS IN 1.E15 CM
C     DX(I) = SHELL THICKNESS IN  CM
C     DXA(I) = SHELL THICKNESS IN 1E15 CM
C     DEN(I) = NUMBER DENSITY IN SHELL I.
C     *****
C     **************************************************************
      R(1)=R1
      IF(INP.NE.1) RQ11=R(1)
      RS15=R1
      RSF15=R1
      DEN(1)=DEN1
       do 6259 i=1,mshell-5
       if(i.le.mcut) then
          dxa(i)=dshell
       elseif(i.gt.mcut.and.i.le.mcut3) then
          dxa(i)=dshell2
       else
          dxa(i)=dshell3
       endif
       if(i.le.10.and.i.le.mcut) dxa(i)=dshell/2.d0
       if(i.le.10.and.i.gt.mcut) then
	 if(i.gt.mcut.and.i.le.mcut3) then
            dxa(i)=dshell2/2.d0
	 else
            dxa(i)=dshell3/2.d0
	 endif
       endif
       if(i.gt.(mshell-15).and.i.le.mcut) dxa(i)=2.d0*dshell
       if(i.gt.(mshell-15).and.i.gt.mcut) then
	 if(i.gt.mcut.and.i.le.mcut3) then
	   dxa(i)=2.d0*dshell2
	 else
	   dxa(i)=2.d0*dshell3
	 endif
       endif
       r(i+1)=r(i)+dxa(i)
       dx(i)=dxa(i)*1.d15
       if(i.le.mcut) den(i)=den1
       if(i.gt.mcut.and.i.le.mcut3) den(i)=den2
       if(i.gt.mcut3) den(i)=den3
       write(6,9572)i,r(i),dxa(i),dx(i),den(i) 
       write(28,9573)r(i)
 6259  CONTINUE
       write(28,9573)r(mshell-4)
 9572 format(1x,i4,5e14.7)
 9573 format(1x,5e16.8)
       do 7603 k=1,3
       if(k.eq.2) goto 7603
       if(k.eq.1) then
        do 7604 i=1,imax
         do 7605 j=1,i
         if(i.eq.j) then
          dxabs(k,i,j)=5.d14*(r(i+1)-r(i))
         else
          dxabs(k,i,j)=1.d15*(r(j+1)-r(j))
         endif
 7605    continue
 7604   continue
       else
        do 7606 i=1,imax
         do 7607 j=i,imax
         if(i.eq.j) then
          dxabs(k,i,j)=5.d14*(r(i+1)-r(i))
         else
          dxabs(k,i,j)=1.d15*(r(j+1)-r(j))
         endif
 7607    continue
 7606   continue
       endif
 7603  continue
C     ***********************************************************
C     *****
C     PRINT OUT INPUT PARAMETERS
C     *****
C     ***********************************************************
      WRITE(6,525)
      WRITE(6,9100)
      WRITE(6,9100)
      WRITE(6,525)
      WRITE(6,105)RCGS,DX(2)
      WRITE(6,525)
      WRITE(6,9130)DEN1,BETA
      IF(ISPEC.NE.1) WRITE(6,525)
      if(ishiwo.eq.0) write(6,*)' Shigeyama & Nomoto 11E0.8 spectrum'
      if(ishiwo.eq.1) write(6,*)' Woosley 10 L spectrum'
      if(ishiwo.eq.3) write(6,*)' Ensman & Burrows full1 '
      if(ishiwo.eq.4) write(6,*)' Ensman & Burrows full2 '
      if(ishiwo.eq.5) write(6,*)' Ensman & Burrows full3 '
      if(ishiwo.eq.6) write(6,*)' Blinnikov & Bartunov nommxf '
      if(ishiwo.eq.7) write(6,*)' Blinnikov & Bartunov ut87af '
      if(ishiwo.eq.8) write(6,*)' Blinnikov et al. nmxfso17 '
      IF(ISPEC.NE.1) WRITE(6,525)
      WRITE(6,113)
      WRITE(6,112)(AB(mmmm)/ab(1),mmmm=1,14)
      WRITE(6,525)
 9346 FORMAT(1X,'T11,T12,STE1,XMIN',5E10.4)
C
C     goto 3882
      ITEX=0
      IP=2
      JASPP=0
      I=1
      enonth=3.d2
C--------------------------------------------
C     SEARCHING FOR APPROPRIATE DELTT (START)
      TSEC=3600.*TI
      TDAY=TI/24.
C -1987a---------------------------------------------------
      if(ishiwo.eq.0) CALL NOMSHI(TDAY,RPA15,TEB)
      if(ishiwo.eq.1) CALL WOS10L(TDAY,RPA15,TEB)
      if(ishiwo.eq.2) then
       RPA15=1.d-5
       TEB=1.5D5
      endif
      qtc=1.d0
      if(ishiwo.eq.3) CALL ENSBUR(TDAY,RPA15,TEB,qtc)
      if(ishiwo.eq.4) CALL ENSBUR2(TDAY,RPA15,TEB,qtc)
      if(ishiwo.eq.5) CALL ENSBUR3(TDAY,RPA15,TEB,qtc)
      if(ishiwo.ge.6) CALL BARBLI(TDAY,RPA15,qtc)
      IF(RPA15.GE.RS15) RPA15=0.999990*RS15
      if(ishiwo.eq.0) tdmin=0.08d0
      if(ishiwo.eq.1) tdmin=0.10d0
      if(ishiwo.eq.3) tdmin=0.08d0
      if(ishiwo.eq.4) tdmin=0.08d0
      if(ishiwo.eq.5) tdmin=0.08d0
      if(ishiwo.eq.6) tdmin=0.06805d0
      if(ishiwo.eq.7) tdmin=0.09434d0
      if(ishiwo.eq.8) tdmin=0.07595d0
      tday0=tday-tdmin
      if(tday0.le.0.0d0) tday0=0.0d0
      tsec0=tday0*8.64d4
C     CALL OPWIND(tsec0,tfail)
      DO 7779 J=JMIN,JJ
      tau(1,j)=0.0d0
      if(ishiwo.le.5) then
         AAEX=(1.1605D4*E1(J))/TEB
         IF(AAEX.GT.1.D2) GOTO 3947
           FLFF=(1.26D10*burmul*qtc*E1(J)**3)/(DEXP(AAEX)-1.0)
           GOTO 3948
 3947    FLFF=1.D-50
 3948    CONTINUE
      else
         flff=fsurf(j)
      endif
      FLFB=0.0D0
      FL(1,J)=(FLFF+FLFB)*(RPA15/R(1))**2
      fl(1,j)=expfn(tau(1,j))*fl(1,j)
 7779 CONTINUE
C -1987a---------------------------------------------------
      CALL RATE(i,enonth)
      ZB(1)=ZE(12)+ZK(12)
      ZA(1)=ZE(14)+ZK(14)
      ZC(1)=ZE(18)+ZK(18)
      zsu=ze(25)
      zne=ze(35)
      zfee=ze(43)
      zsii=ze(58)
      znaaa=ze(68)
      zmgaa=ze(77)
      zalaa=ze(87)
      zaraa=ze(98)
      zcaaa=ze(108)
      B31=DMAX1(ZK(1),ZK(2),ZB(1),ZA(1),ZC(1),zsu,zne,zfee,zsii,znaaa,
     &zmgaa,zalaa,zaraa,zcaaa)
      DELTT=0.002/B31
      if(deltt.gt.1.d0) deltt=1.d0
 7891 FORMAT(1X,10E10.3)
C---------------------------------------------
      JUP=2
      JAP=2
      JUPIT=2
      ITE=0
      ktemp=7
      itemp=ktemp
      itex=0
      DELQU=DELTT
      DELTTT=DELTT
      QQV=0.
      QQVV=0.
      QQVVV=0.
      QQVVVV=0.
      GOTO 9515
C---------------------------------------------------------------
C----- start of time-loop
C
 5734 ITE=ITE+1
      itemp=itemp+1
      JUP=JUP+1
      JAP=JAP+1
      JUPIT=JUPIT+1
C     ***********************************************************
 9515 TI=TI+DELTTT/3.6D3
      TDAY=TI/24.
      if(ite.eq.0) tdinit=tday
c cf  output times      
      tday0=tday-tdinit
      IF(TDAY0.GT.1.D4) GOTO 3882
C      IF(TDAY0.GT.3.D0) GOTO 3882 
      TSEC=TI*3600.
      if(ite.eq.0) then
        do 7395 i=imin,imax
 7395   tsecc(i)=tsec
      endif
C     IF(ITEX.GT.0) WRITE(17,8393)TSEC
 8393 FORMAT(1X,E20.12)
      I=1
C -- Ionizing spectrum --
      if(ishiwo.le.5) then
         call flux1j(ishiwo,tdmin,tday,rpa15,burmul)
      else
         CALL BARBLI(TDAY,RPA15,qtc)
      endif
C      goto 5934
C --- Circumstellar emission (brems.spectrum; Hasinger etal. A&A, '96).
      if(tday.gt.9.d2) then
C        call shock(tday,clum,estart,gam2)
      clum=1.364d32*((tday-9.d2)/2.1d3)
C      clum=1.d3*clum
	do j=jmin,jj
	fsurf(j)=0.0d0
C	if(e1(j).gt.estart) then
C	 fsurf(j)=fsurf(j)+(clum/(4.d15*pi*rpa15)**2)*(estart/e1(j))
C     &           **gam2
        fsurf(j)=fsurf(j)+(clum/(4.d15*pi*rpa15)**2)
     &           *expfn(e1(j)/440.d0)
C	endif
	enddo
      endif
C5934 continue
C -1987a---------------------------------------------------
      V=0.
      VV=0.
      VVV=0.
      VVVV=0.
      VVENE=0.0D0
      VVENE1=0.0D0
      VVENE2=0.0D0
      VVENE3=0.0D0
      DO 3281 J=2,13
      VVENE1=VVENE1+FSURF(J)*(E(J+1)-E(J))
 3281 V=V+(FSURF(J)*(E(J+1)-E(J)))/(E1(J)*1.6022D-12)
C      DO 3282 J=14,37
      DO 3282 J=14,18
      VVENE2=VVENE2+FSURF(J)*(E(J+1)-E(J))
 3282 VV=VV+(FSURF(J)*(E(J+1)-E(J)))/(E1(J)*1.6022D-12)
C      DO 3283 J=38,JJ
      DO 3283 J=19,JJ
      VVENE3=VVENE3+FSURF(J)*(E(J+1)-E(J))
 3283 VVV=VVV+(FSURF(J)*(E(J+1)-E(J)))/(E1(J)*1.6022D-12)
      V=V*((4.D15*PI*RPA15)**2)
      VVENE1=VVENE1*((4.D15*PI*RPA15)**2)
      VVENE2=VVENE2*((4.D15*PI*RPA15)**2)
      VVENE3=VVENE3*((4.D15*PI*RPA15)**2)
      VVENE=VVENE1+VVENE2+VVENE3
      VV=VV*((4.D15*PI*RPA15)**2)
      VVV=VVV*((4.D15*PI*RPA15)**2)
      VVVV=VVV+VV+V
      QQV=QQV+V*DELTTT
      QQVV=QQVV+VV*DELTTT
      QQVVV=QQVVV+VVV*DELTTT
      QQVVVV=QQVVVV+VVVV*DELTTT
      QQENE1=QQENE1+VVENE1*DELTTT
      QQENE2=QQENE2+VVENE2*DELTTT
      QQENE3=QQENE3+VVENE3*DELTTT
      QQENE=QQENE+VVENE*DELTTT
      IF(ITEX.eq.0) GOTO 555
      TYEAR=TDAY0/3.6525D2
      WRITE(6,1451)
      WRITE(6,525)
      WRITE(6,9988)TYEAR,DELTTT,ITE
C      if(ite.ne.0) WRITE(7,8787)TYEAR
 8787 format(1x,3e20.11)
      WRITE(6,4438)V,VV,VVV,VVVV
      WRITE(6,4439)QQV,QQVV,QQVVV,QQVVVV
 4438 FORMAT(1X,'# Ion.Phot',4E12.5)
 4439 FORMAT(1X,'Tot.Phot',4E12.5)
      WRITE(6,9438)vvene1,vvene2,vvene3,vvene
      WRITE(6,9439)qqene1,qqene2,qqene3,qqene
 9438 FORMAT(1X,'Ion.Energy',4E12.5)
 9439 FORMAT(1X,'Tot.Energy',4E12.5)
 1451 FORMAT(1X,'----------------------------------------------')
 9988 FORMAT(1X,'TIME,dt,ITE',2E13.5,I6)
  555 CONTINUE
      TEL=0.
      DIFFEM=0.
C
C*************************
C ----start of depth-loop
C*************************
      iprint=1
      DO 700 I=IMIN,IMAX
      iprint=iprint+1
      if(iprint.gt.1) iprint=1
      if(imax.le.50) iprint=1
      if(i.eq.imax) iprint=1
      if(ite.eq.0) then
       deltt=delttt
       icomp(i)=1
       dent(i)=delttt
        itee(i)=itee(i)+1
      else
       if(icomp(i).eq.1) then
        deltt=dent(i)
        itee(i)=itee(i)+1
       endif
      endif
      if(icomp(i).eq.1) then
       QXA1=QXB(I,1,1)
       IF(DEL(I).LE.0.D0) DEL(I)=1.D-10
        DO 7416 K=1,117
 7416   XA(2,K)=XB(I,K)
C --  Column densities  --
       call colden(imax)
        do 9046 k=1,117
        columi(k)=colni(i,k)
        colum(k)=colni(i,k)+colno(i,k)
        if(columi(k).le.0.0d0) columi(k)=0.0d0
        if(colum(k).le.0.0d0) colum(k)=0.0d0
 9046   continue
      endif
      CALL TAUI(i)
      if(icomp(i).ne.1) goto 5638
      CALL SPEC(i)
      CALL RATE(i,enonth)
C
C     NEW TEMPERATURES AND IONIZATION DEGREES FOR SLAB I
C---   When calculating the new temps we linearize explicitly
C
      TEALD(I)=TE(I)
      BEX=4.829D15*DELTT*DEN(I)
      corat1=0.
      corat2=0.
      corat3=0.
      corat4=0.
      corat5=0.
      corat6=0.
      corat7=0.
      corat8=0.
      XEL=DEL(I)
      icom=0
      dtrad=deltt
      SQW(I)=RAD(TE(I),MAX,CORAT1,CORAT2,ITE,dtrad,teinn,corat3,
     &corat4,corat5,corat6,corat7,corat8,fudfed)
      frro21(i)=fro21
      frro41(i)=fro41
      frrs21(i)=frs21
      frrs41(i)=frs41
      CALL IONS
      XELNEW=DEL(I)
      DXEDT=(XELNEW-XEL)/DELTT
      IF(ITE.EQ.0.or.te(i).le.teinn) DXEDT=0.0D0
      DETEMP=(BEX*SQW(I)-TE(I)*DELTT*DXEDT)/(1.D0+XEL)
      TE(I)=TEALD(I)+DETEMP
 2419 CONTINUE
        do 9391 j=2,jj
        tauiii=0.0d0
        tauooo=0.0d0
        if(i.ne.1) then
         do 9392 iin=1,i-1
         dendx=den(iin)*dxabs(1,i,iin)
          do 9393 kk=1,117
          tauiii=tauiii+abunda(kk)*xb(iin,kk)*(si(kk,j)+sk(kk,j))*
     &    dendx
 9393     continue
 9392    continue
        endif
        if(i.ne.imax) then
         do 9394 iin=i+1,imax
         dendx=den(iin)*dxabs(3,i,iin)
          do 9395 kk=1,117
          tauooo=tauooo+abunda(kk)*xb(iin,kk)*(si(kk,j)+sk(kk,j))*
     &    dendx
 9395     continue
 9394    continue
        endif
        twoexp(j)=0.5d0*(expfn(tauiii)+expfn(tauooo))
 9391  continue
      CALL BOWOTS(imax,ite,CORAT1,CORAT2,corat3,corat4,corat5,corat6,
     &corat7,corat8)
      COLD(I)=COOL
      WARM(I)=HEAT
      HYD(I)=hytot
 5638 continue
C     ***********************************************************
C     *****
C     PRINT RESULTS FOR SHELL I
C     *****
C     ***********************************************************
      RCGS=R(I)*1.D15
      IF(ITEX.EQ.0) GOTO 8236
C---------------------------------------------------------------- 
C OPTICAL DEPTHS AT 2, 6 AND 20 CM 
C---------------------------------------------------------------- 
       CDC(1)=AB(1)*QXB(I,JUP,2)+AB(2)*QXB(I,JUP,4)+AB(3)*QXB(I, 
     &JUP,14)+AB(4)*QXB(I,JUP,7)+AB(5)*QXB(I,JUP,34) 
     &+ab(6)*qxb(i,jup,23)+ab(7)*qxb(i,jup,42)+ab(8)*qxb(i,jup,51)
     &+ab(9)*qxb(i,jup,67)+ab(10)*qxb(i,jup,78)+ab(11)*
     &qxb(i,jup,88)+ab(12)*qxb(i,jup,99)+ab(13)*qxb(i,jup,111)+
     &ab(14)*qxb(i,jup,122)
      CDC(2)=4.*(AB(2)*QXB(I,JUP,5)+AB(3)*QXB(I,JUP,15)+AB(4)* 
     &QXB(I,JUP,8)+AB(5)*QXB(I,JUP,35)+ab(6)*qxb(i,jup,24)+
     &ab(7)*qxb(i,jup,43)+ab(8)*qxb(i,jup,52)+ab(9)*qxb(i,jup,68)+
     &ab(10)*qxb(i,jup,79)+ab(11)*qxb(i,jup,89)+ab(12)*
     &qxb(i,jup,100)+ab(13)*qxb(i,jup,112)+ab(14)*qxb(i,jup,123))
      DO 7742 IWA=3,6 
      CDC(IWA)=((DBLE(IWA))**2)*(AB(3)*QXB(I,JUP,13+IWA)+AB(4)* 
     &QXB(I,JUP,6+IWA)+AB(5)*QXB(I,JUP,33+IWA)+ab(6)*qxb(i,jup,22+ 
     &iwa)+ab(7)*qxb(i,jup,iwa+41)+ab(8)*qxb(i,jup,iwa+50)+ab(9)*
     &qxb(i,jup,iwa+66)+ab(10)*qxb(i,jup,iwa+77)+ab(11)*
     &qxb(i,jup,iwa+87)+ab(12)*qxb(i,jup,iwa+98)+ab(13)*
     &qxb(i,jup,iwa+110)+ab(14)*qxb(i,jup,iwa+121))
 7742 CONTINUE 
      CDC(7)=49.*(AB(3)*QXB(I,JUP,20)+AB(5)*QXB(I,JUP,44)+ 
     &ab(6)*qxb(i,jup,29)+ab(7)*qxb(i,jup,48)+ab(8)*qxb(i,jup,57)+
     &ab(9)*qxb(i,jup,73)
     &+ab(10)*qxb(i,jup,84)+ab(11)*qxb(i,jup,94)+ab(12)*
     &qxb(i,jup,105)+ab(13)*qxb(i,jup,117)+ab(14)*qxb(i,jup,128))
      CDC(8)=64.*(AB(3)*QXB(I,JUP,21)+ab(6)*qxb(i,jup,30)+ab(7)*qxb 
     &(i,jup,49)+ab(8)*qxb(i,jup,58)+ab(9)*qxb(i,jup,74)
     &+ab(10)*qxb(i,jup,85)+ab(11)*qxb(i,jup,95)+ab(12)*
     &qxb(i,jup,106)+ab(13)*qxb(i,jup,118)+ab(14)*qxb(i,jup,129))
      cdc(9)=81.*(ab(6)*qxb(i,jup,31)+ab(8)*qxb(i,jup,59)+ab(9)*
     &qxb(i,jup,75)+ab(10)*qxb(i,jup,86)+ab(11)*qxb(i,jup,96)+ab(12)
     &*qxb(i,jup,107)+ab(13)*qxb(i,jup,119)+ab(14)*qxb(i,jup,130))
      cdc(10)=100.*(ab(6)*qxb(i,jup,32)+ab(8)*qxb(i,jup,60)+ab(9)*
     &qxb(i,jup,76)+ab(11)*qxb(i,jup,97)+ab(12)*qxb(i,jup,108)+
     &ab(13)*qxb(i,jup,120)+ab(14)*qxb(i,jup,131))
      cdc(11)=121.*(ab(8)*qxb(i,jup,61)+ab(12)*qxb(i,jup,109))
      DO 7942 IWA=12,15 
      cdc(iwa)=dble(iwa)**2*ab(8)*qxb(i,jup,iwa+50)
 7942 continue
      CDCA=0. 
      CDCB=0. 
      CDCC=0. 
      DO 7746 IWA=1,15 
      CDCA=CDCA+CDC(IWA)*(17.7+DLOG((3.336D-11*2.*TE(I)**1.5)/ 
     &DBLE(IWA))) 
      CDCB=CDCB+CDC(IWA)*(17.7+DLOG((3.336D-11*6.*TE(I)**1.5)/ 
     &DBLE(IWA))) 
      CDCC=CDCC+CDC(IWA)*(17.7+DLOG((3.336D-11*20.*TE(I)**1.5)/ 
     &DBLE(IWA))) 
 7746 CONTINUE 
      RAAA(I)=(1.104D-8*((2.*DEN(I))**2)*DEL(I)*(r(i+1)-r(i)))
     &/TE(I)**1.5 
      RAAB(I)=(1.104D-8*((6.*DEN(I))**2)*DEL(I)*(r(i+1)-r(i)))
     &/TE(I)**1.5 
      RAAC(I)=(1.104D-8*((20.*DEN(I))**2)*DEL(I)*(r(i+1)-r(i)))
     &/TE(I)**1.5 
      RAAA(I)=RAAA(I)*CDCA 
      RAAB(I)=RAAB(I)*CDCB 
      RAAC(I)=RAAC(I)*CDCC 
C----------------------------------------------------------------
C --- Line emissivities in lines (ergs / cm3 s sr)
C
      dendel=den(i)**2*del(i)
      dendelpi=dendel/(4.d0*pi)
      stolin(i,1)=hy*dendelpi
      stolin(i,2)=twop(i)*dendel
      stolin(i,3)=hy2*dendelpi
      stolin(i,4)=hy3*dendelpi
      stolin(i,5)=hyco*dendelpi
      stolin(i,6)=hyco2*dendelpi
      stolin(i,7)=hyco3*dendelpi
      stolin(i,8)=heii21*dendel
      stolin(i,9)=uvheli(i,1)*dendel
      do kk=1,9
       stolin(i,9+kk)=opthel(kk,i)*dendel
      enddo
      stolin(i,19)=redc(1)*dendel
      stolin(i,20)=redc(2)*dendel
      do kk=1,4
       stolin(i,20+kk)=forc(kk)*dendel
      enddo
      stolin(i,25)=coolfr(55)*dendelpi
      stolin(i,26)=coolfr(56)*dendelpi
      stolin(i,27)=uvcarb(i,1)*dendel
      stolin(i,28)=redc(3)*dendel
      stolin(i,29)=uvcarb(i,2)*dendel
      stolin(i,30)=redc(4)*dendel
      stolin(i,31)=redc(5)*dendel
      stolin(i,32)=uvcarb(i,3)*dendel
      stolin(i,33)=(uvcarb(i,4)+uvcarb(i,6))*dendel
      stolin(i,34)=uvcarb(i,5)*dendel
      do kk=1,3
       stolin(i,kk+34)=forn(kk)*dendel
      enddo
      stolin(i,38)=coolfr(57)*dendelpi
      stolin(i,39)=redn(1)*dendel
      stolin(i,40)=redn(2)*dendel
      stolin(i,41)=forn(4)*dendel
      stolin(i,42)=forn(5)*dendel
      stolin(i,43)=uvnitr(i,1)*dendel
      stolin(i,44)=forn(6)*dendel
      stolin(i,45)=coolfr(49)*dendelpi
      stolin(i,46)=uvnitr(i,3)*dendel
      stolin(i,47)=redn(3)*dendel
      stolin(i,48)=uvnitr(i,2)*dendel
      stolin(i,49)=redn(4)*dendel
      stolin(i,50)=redn(5)*dendel
      stolin(i,51)=coolfr(51)*dendelpi
      stolin(i,52)=(uvnitr(i,4)+uvnitr(i,6))*dendel
      stolin(i,53)=uvnitr(i,5)*dendel
      stolin(i,54)=redo(6)*dendel
      stolin(i,55)=redo(7)*dendel
      do kk=1,3
       stolin(i,kk+55)=foro(6+kk)*dendel
      enddo
      stolin(i,59)=coolfr(53)*dendelpi
      do kk=1,3
       stolin(i,kk+59)=foro(3+kk)*dendel
      enddo
      stolin(i,63)=coolfr(45)*dendelpi
      stolin(i,64)=coolfr(50)*dendelpi
      stolin(i,65)=coolfr(54)*dendelpi
      stolin(i,66)=redo(4)*dendel
      stolin(i,67)=redo(5)*dendel
      stolin(i,68)=foro(1)*dendel
      stolin(i,69)=foro(2)*dendel
      stolin(i,70)=uvoxyg(i,4)*dendel
      stolin(i,71)=foro(3)*dendel
      stolin(i,72)=foiii(i,3)*dendel
      stolin(i,73)=foiii(i,7)*dendel
      stolin(i,74)=coolfr(46)*dendelpi
      stolin(i,75)=coolfr(47)*dendelpi
      stolin(i,76)=coolfr(52)*dendelpi
      stolin(i,77)=redo(1)*dendel
      stolin(i,78)=uvoxyg(i,3)*dendel
      stolin(i,79)=redo(2)*dendel
      stolin(i,80)=redo(3)*dendel
      stolin(i,81)=coolfr(48)*dendelpi
      stolin(i,82)=uvoxyg(i,2)*dendel
      stolin(i,83)=uvoxyg(i,1)*dendel
      stolin(i,84)=reds(1)*dendel
      stolin(i,85)=reds(2)*dendel
      do kk=1,6
       stolin(i,85+kk)=fors(kk)*dendel
      enddo
      stolin(i,92)=coolfr(92)*dendelpi
      stolin(i,93)=reds(3)*dendel
      stolin(i,94)=reds(4)*dendel
      do kk=1,4
       stolin(i,94+kk)=fors(kk+6)*dendel
      enddo
      stolin(i,99)=coolfr(91)*dendelpi
      stolin(i,100)=reds(5)*dendel
      stolin(i,101)=coolfr(93)*dendelpi
      stolin(i,102)=reds(6)*dendel
      stolin(i,103)=reds(7)*dendel
      stolin(i,104)=coolfr(89)*dendelpi
      stolin(i,105)=coolfr(90)*dendelpi
      stolin(i,106)=rede(1)*dendel
      stolin(i,107)=rede(2)*dendel
      do kk=1,6
       stolin(i,107+kk)=fore(kk)*dendel
      enddo
      stolin(i,114)=rede(3)*dendel
      stolin(i,115)=rede(4)*dendel
      do kk=1,4
       stolin(i,115+kk)=fore(6+kk)*dendel
      enddo
      stolin(i,120)=coolfr(88)*dendelpi
      stolin(i,121)=redsi(1)*dendel
      do kk=1,4
       stolin(i,121+kk)=uvsi(i,kk)*dendel
      enddo
      do kk=1,3
       stolin(i,125+kk)=forna(kk)*dendel
      enddo
      do kk=1,4
       stolin(i,128+kk)=formg(kk)*dendel
      enddo
      stolin(i,133)=formg(8)*dendel
      stolin(i,134)=formg(9)*dendel
      stolin(i,135)=formg(11)*dendel
      stolin(i,136)=foralu(1)*dendel
      stolin(i,137)=foralu(2)*dendel
      do kk=1,8
       stolin(i,137+kk)=forar(kk)*dendel
      enddo
      stolin(i,146)=forar(10)*dendel
      stolin(i,147)=forca(1)*dendel
      stolin(i,148)=forca(2)*dendel
      do kk=1,5
       stolin(i,148+kk)=forca(kk+2)*dendel
      enddo
      stolin(i,154)=forca(9)*dendel
      stolin(i,155)=fecol10*dendelpi
      stolin(i,156)=forbfe1*dendelpi
C
      IF(I.NE.IMAX) GOTO 8394
C --- Column densities of UV-absorbing ions --
C --- and total radio optical depths --
      IMAZ=IMAX
      optsu1=0.
      optsu2=0.
      optsu3=0.
      TOPTO6=colum(18)
      TOPTN5=colum(37)
      TOPTC3=colum(8)
      TOPTC4=colum(9)
      TOPTH1=colum(1)
      DO 2968 IW=1,IMAZ
      optsu1=optsu1+raaa(iw)
      optsu2=optsu2+raab(iw)
      optsu3=optsu3+raac(iw)
 2968 CONTINUE
 8394 CONTINUE
C----------------------------------------------------------------
 4461 CONTINUE
      if(iprint.ne.1) goto 6394
      TINEW=TDAY+(R(I)-RS15)*0.3861
      WRITE(6,525)
      WRITE(6,5920)I,itee(i),r(i),tday0
c     WRITE(0,5920)I,r(i),tday0
 5920 FORMAT(1X,'RESULTS FOR SHELL:',I3,I5,' R: ',1pe10.3,' time: '
     &                                               ,e11.4,' days')
      WRITE(6,5542)DXA(I),DEN(I),DEL(I),TE(I)
      WRITE(6,7651)(QXB(I,2,K),K=1,5)
      WRITE(6,7652)(QXB(I,2,K),K=6,12)
      WRITE(6,7654)(QXB(I,2,K),K=33,40)
      WRITE(6,7653)(QXB(I,2,K),K=13,21)
      WRITE(6,8653)(QXB(I,2,K),K=41,49)
      WRITE(6,8664)(QXB(I,2,K),K=77,86)
      WRITE(6,8665)(QXB(I,2,K),K=87,97)
      WRITE(6,8666)(QXB(I,2,K),K=98,109)
      WRITE(6,8667)(QXB(I,2,K),K=66,76)
      WRITE(6,8654)(QXB(I,2,K),K=22,32)
      WRITE(6,8668)(QXB(I,2,K),K=110,120)
      WRITE(6,8669)(QXB(I,2,K),K=121,131)
      WRITE(6,8663)(QXB(I,2,K),K=50,65)
      write(6,4793)warm(i),cold(i)
 4793 format('Heating =',1x,1e13.5,7x,'Cooling =',1x,1e13.5)
7651  FORMAT(' H,He ',9e11.3)
7652  FORMAT(' C    ',9e11.3)
7653  FORMAT(' O    ',9e11.3)
7654  FORMAT(' N    ',9e11.3)
8653  FORMAT(' Ne   ',11e11.3)
8654  FORMAT(' S    ',11e11.3)
8663  FORMAT(' Fe   ',16e11.3)
8664  FORMAT(' Na   ',12e11.3)
8665  FORMAT(' Mg   ',12e11.3)
8666  FORMAT(' Al   ',12e11.3)
8667  FORMAT(' Si   ',12e11.3)
8668  FORMAT(' Ar   ',12e11.3)
8669  FORMAT(' Ca   ',12e11.3)
 5432 FORMAT(1X,9E12.4)
 8929 FORMAT(1X,85E12.4)
 6394 continue
C
      IF(I.NE.IMAX) GOTO 2749
      WRITE(6,5433)I,TDAY0,RS15,OPTSU1,OPTSU2,OPTSU3
 9386 FORMAT(1X,50E12.4)
      if(ite.eq.0) goto 5079
       do 9368 kk=1,156
       sumlin(kk)=0.0d0
        do 7079 iz=1,mshell-5
        dshl=3.33333d44*(4.d0*pi)**2*(r(iz+1)**3-r(iz)**3)
        sumlin(kk)=sumlin(kk)+dshl*stolin(iz,kk)
 7079   continue
 9368  continue
C      write(7,4509)(sumlin(kk),kk=1,156)
      write(31,4509)tday0
      write(14,4509)tday0
      write(36,4509)tday0
       do 7081 iz=1,mshell-5
       write(31,4509)te(iz),(stolin(iz,kk),kk=1,156)
       write(14,4509)te(iz),(qxb(iz,2,kk),kk=1,131)
       write(36,4509)te(iz),frro21(iz),frro41(iz),frrs21(iz),
     & frrs41(iz)
 7081  continue
 4509 format(1x,160e13.5)
 5079 continue
 9898 FORMAT(1X,20E12.5)
      WRITE(6,5922)TOPTC3,TOPTC4,TOPTN5,TOPTO6,TOPTH1
      if(ite.eq.0) goto 2749
C     WRITE(7,9898)TOPTC3,TOPTC4,TOPTN5,TOPTO6,TOPTH1
 5922 FORMAT(1X,'COLUMN:C3,C4,N5,O6,H1',5E13.4)
C     DO 5819 JK=JMIN,JJ
C     FEUM=16.*PI**2*FLUXED(I,JK)*1.D30*R(I)**2
C     IF(TDAY.GT.22.) WRITE(6,7920)E(JK),FEUM
C5819 CONTINUE
 7920 FORMAT(1X,3E13.5)
 2749 CONTINUE
C
 6929 CONTINUE
 9991 FORMAT(2E14.6,4E12.4)
 5431 FORMAT(' Heat, Cool, Hyd',1X,I5,8E13.5)
 5542 FORMAT(1X,'dx, dens, x, te: ',1pe12.4,8E12.4)
 5433 FORMAT(1X,'TIME',I5,8E13.5)
 8573 CONTINUE
 6491 FORMAT(1X,'ELEC.TEMP',4(1X,E12.6))
 8236 CONTINUE
C---- OUTGOING NUMBER OF PHOTONS (cumulated number)
      IF(I.NE.IMAX) GOTO 556
      V=0.
      VV=0.
      VVV=0.
      VVVV=0.
      DO 6781 J=2,13
 6781 V=V+(FL(IMAX,J)*(E(J+1)-E(J)))/(E1(J)*1.6022D-12)
C      DO 6782 J=14,37
      DO 6782 J=14,18
 6782 VV=VV+(FL(IMAX,J)*(E(J+1)-E(J)))/(E1(J)*1.6022D-12)
      DO 6783 J=19,JJ
C      DO 6783 J=38,JJ
 6783 VVV=VVV+(FL(IMAX,J)*(E(J+1)-E(J)))/(E1(J)*1.6022D-12)
      V=V*((4.D15*PI*R(IMAX))**2)
      VV=VV*((4.D15*PI*R(IMAX))**2)
      VVV=VVV*((4.D15*PI*R(IMAX))**2)
      VVVV=VVV+VV+V
      QIMAX=QIMAX+V*DELTTT
      QIMAXV=QIMAXV+VV*DELTTT
      QIMAVV=QIMAVV+VVV*DELTTT
      QIMVVV=QIMVVV+VVVV*DELTTT
      IF(ITEX.EQ.0.OR.TDAY.LT.1.25D-2) GOTO 556
      WRITE(6,4438)V,VV,VVV,VVVV
      WRITE(6,4439)QIMAX,QIMAXV,QIMAVV,QIMVVV
  556 CONTINUE
  700 CONTINUE
C
C-------end of depth-loop
C
C--------------------------------------------------------------------
c     IF(TDAY.LT.0.5D0) GOTO 7985
c     print abund. for all shells
c     write(14)tday0
c     do 6327 i=imin,imax
c6327 WRITE(14)r(i),i,TE(i),(QXB(i,2,K),K=1,21),(QXB(i,2,K),K=
c    &33,40)
c7985 CONTINUE
      IF(ITEX.EQ.0) GOTO 7938
      SSOPT1=0.0D0
      SSOPT2=0.0D0
      SSOPT3=0.0D0
      SSOPT4=0.0D0
      SSOPT5=0.0D0
      SSOPT6=0.0D0
      DO 7797 IEUV=1,26
 7797 OPTEUV(IEUV)=0.0D0
      DO 9976 IIJJ=1,IMAX
      DQQQ=1.D15*(R(IIJJ+1)-R(IIJJ))
      if(iijj.eq.1.or.iijj.eq.imax) dqqq=dqqq/2.d0
      DO 8797 IEUV=1,26
 8797 OPTEUV(IEUV)=OPTEUV(IEUV)+DQQQ*ABSEUV(IIJJ,IEUV)
      SSOPT1=SSOPT1+DQQQ*ABS12(IIJJ)
      SSOPT2=SSOPT2+DQQQ*ABS13(IIJJ)
      SSOPT3=SSOPT3+DQQQ*ABSHE2(IIJJ)
      SSOPT4=SSOPT4+DQQQ*ABSHE3(IIJJ)
      SSOPT5=SSOPT5+DQQQ*ABSO1(IIJJ)
      SSOPT6=SSOPT6+DQQQ*ABSNV(IIJJ)
 9976 CONTINUE
      WRITE(9,7939)TDAY0,SSOPT1,SSOPT2,SSOPT3,SSOPT4,SSOPT5,SSOPT6
     &,(OPTEUV(IEUV),IEUV=1,15)
 7939 FORMAT(1X,30E12.4)
 7938 CONTINUE
C
      itex=0
      if(tday0.gt.tout) itex=1000
      tout1=tout
      if(tday0.gt.tout1.and.tday0.lt.(0.1)) tout=8.*tout1
      if(tday0.gt.tout1.and.tday0.gt.(0.1)) tout=4.*tout1
      if(tday0.gt.tout1.and.tday0.gt.(0.5)) tout=2.*tout1
      if(tday0.gt.tout1.and.tday0.gt.3.) tout=1.3*tout1
      if(tday0.gt.tout1.and.tday0.gt.100.) tout=1.1*tout1
c
      deltk=delttt
      delttt=1.3d0*deltk
      if(tday0.lt.0.005d0) delttt=1.2d0*deltk
C  2 % temperature increase per time step during initial ionization
C  (i.e., roughly 0.05 days), 0.5 % thereafter.
C  Always include the shell with the most swift changes (i=iident).
      tolte=5.d-3
      if(tday0.lt.0.05d0) then
       tolte=tolte*(1.d0+3.d0*(1.d0-tday0/0.05d0))
      endif
c
       DO 715 I=IMIN,IMAX
       if(icomp(i).eq.1) then
        dentk(i)=dent(i)
        tux=0.0d0
         DO 716 K=1,131
         IF(QXB(I,JUP,K).GT.TOL1) THEN
          TOX=DABS((QXB(I,JUP,K)-QXB(I,JUP-1,K))/QXB(I,JUP,K))
          tux=dmax1(tux,tox)
         endif
  716    CONTINUE
        tuxte=dabs((te(i)-teald(i))/te(i))
        deltio=(TOL2*dentk(i))/tux
        deltte=(TOLTE*dentk(i))/tuxte
        dent(i)=dmin1(deltio,deltte)
        tsecc(i)=tsecc(i)+dent(i)
        if(dent(i).le.delttt) then
         ixprint=i
         delttt=dent(i)
        endif
       endif
  715  CONTINUE
c
      tsecci=tsec+delttt
      denti=1.d50
      if(itex.ne.0) then
       DO 718 I=IMIN,IMAX
       icomp(i)=1
       dent(i)=dent(i)+tsecci-tsecc(i)
       tsecc(i)=tsecci
  718  continue
      else
       DO 717 I=IMIN,IMAX
       if(dent(i).le.denti) then
        iident=i
        denti=dent(i)
       endif
       icomp(i)=0
       if(tsecc(i).le.tsecci.or.itemp.eq.ktemp) then
        icomp(i)=1
        dent(i)=dent(i)+tsecci-tsecc(i)
        tsecc(i)=tsecci
       endif
  717  continue
       if(icomp(iident).eq.0) then
        icomp(iident)=1
        dent(iident)=dent(iident)+tsecci-tsecc(iident)
        tsecc(iident)=tsecci
       endif
      endif
      if(itemp.eq.ktemp.or.itex.ne.0) itemp=0
C
      write(35,2707)ite,iident,tday0,delttt,deltk
C-----------------------------------------
C    RENUMBER THE POINTS
 2707 format(1x,2i7,5e14.5)
       DO 790 I=IMIN,IMAX
        DO 1790 K=1,131
        if(k.gt.levhe) goto 1790
        qxhe(i,1,k)=qxhe(i,2,k)
        if(k.gt.levh) goto 1790
        qxh(i,1,k)=qxh(i,2,k)
 1790   QXB(I,1,K)=QXB(I,2,K)
  790  continue
C----------------------------------------
      JUP=1
      IF(JAP.EQ.JAPEX) JAP=1
      GOTO 5734
9100  FORMAT(1X,'*******************************************************
     &**********************')
 105  FORMAT(1X,'RADIUS OF SHOCK=',1PE10.3,' CM   INITIAL SHELL ',
     &'THICKNESS=',1PE10.3,' CM')
9130  FORMAT(1X,'INITIAL DENSITY',1PE12.5,'  POWER LAW INDEX ='
     &,0PF10.2)
 109  FORMAT(1X,'OUTER SHOCK TEMP.=',F10.2,' KEV.  OPTICAL DEPTH OF ',
     &'D:O =',1PE12.5)
 113  FORMAT(1X,'  AB(H)    AB(He)   AB(O)    AB(C)    AB(N)  ',
     &         '  AB(S)   AB(Ne)    AB(Fe)   AB(Si)   AB(NA)  ',
     &         ' AB(Mg)   AB(Al)   AB(Ar)   AB(Ca)')
  112 FORMAT(1X,14(1X,1PE8.2))
  110 FORMAT(1X,'TEMP. AND LUM. OF BLACKBODY FLUX',F12.2,' K',2X,
     &1PE12.4,' ERG/SEC')
  111 FORMAT(1X,'TEMP. AND LUM. OF SOFT X-RAY FLUX',F10.2,' KEV',2X,
     &1PE12.4,' ERG/SEC')
 5555 FORMAT(1X,10E11.5)
 7778 FORMAT(1X,'NUMBER OF IONIZING PHOTONS: S(13.6-54 EV)=',1PE13.6
     &,' S( >54 EV)=',1PE13.6)
  525 FORMAT(/)
  903 FORMAT(1X,'ENERGY BETWEEN 13.6-54 EV ',1PE12.5,' ABOVE 54 EV'
     &,E12.5,' RADIUS=',E13.6)
 905  FORMAT(1X,9E9.3)
  925 FORMAT(//)
 9420 FORMAT(1X,5E12.5)
  476 FORMAT(1X,E11.5,15F7.3)
 9430 FORMAT(1X,E11.5,14F7.3)
      GOTO 3882
 6746 PRINT *,'TOO MANY ITERATIONS IN RAD'
      GOTO 3882
 6900 PRINT *,'ERROR MET IN SUBROUTINE MATRIX'
      GOTO 3882
 6744 PRINT *,'ERROR MET IN NEWTON-RAPHSON ROUTINE'
 3882 CONTINUE
C      write(7,2229)
C      write(27,2229)
 2229 format(1x,'-9999.')
      RETURN
      END
C****************************************************************
C****************************************************************
      SUBROUTINE HYDIJ
C****************************************************************
C  Reads hydrogen A-values from file 'slaij_h_10', recombination
C  coefficients from 'slrecij_h_10'. (Used in HLOSS and HELOSS)
C****************************************************************
      IMPLICIT REAL*8(A-H,O-Z)                                          
      parameter (md=75,levhmx=55,levh=levhmx+1,nnhmax=10)
C
      common/hydaij/ahydij(levhmx,levhmx),ahydsm(levhmx)
      common/helaij/ahelij(levhmx,levhmx),ahelsm(levhmx)
      common/hrecij/rechij(25,levhmx),terech(25),rechsum(25)
      common/hjohns/cijjoh(9,levhmx,levhmx),njohn(286),mjohn(286),nmjo
      common/herecij/recheij(25,levhmx),tereche(25),rechesum(25)
C
      dimension atemp(9)
C
      open(17,file='slaij_h_10',STATUS='UNKNOWN')
      do i=1,levhmx
	 ahydsm(i)=0.0d0
	 ahelsm(i)=0.0d0
	 do j=1,levhmx
	    ahydij(i,j)=0.0d0
	    ahelij(i,j)=0.0d0
	 enddo
      enddo
      do iqq=1,1000
         read(17,*,err=999,end=999)n,m,l_n,l_m,i,j,aval
         ahydij(i,j)=aval
         ahelij(i,j)=1.6d1*aval
	 ahydsm(i)=ahydsm(i)+aval
	 ahelsm(i)=ahelsm(i)+1.6d1*aval
      enddo
  999 continue
      close(17)
C
      open(17,file='slrecij_h_10',STATUS='UNKNOWN')
      do i=1,25
         read(17,*)terech(i),(rechij(i,j),j=1,levhmx),rechsum(i)
	 tereche(i)=dlog10(4.d0*1.d1**terech(i))
	 rechesum(i)=dlog10(2.d0*1.d1**rechsum(i))
	 do j=1,levhmx
	    recheij(i,j)=dlog10(2.d0*1.d1**rechij(i,j))
	 enddo
      enddo
      close(17)
C
      open(17,file='slnm_h',STATUS='UNKNOWN')
      nmjo=286
      do iqq=1,nmjo
         read(17,*)njohn(iqq),mjohn(iqq),(atemp(ijte),ijte=1,9)
	 do i=1,9
            cijjoh(i,mjohn(iqq),njohn(iqq))=atemp(i)
	 enddo
      enddo
      close(17)
C
      RETURN
      END
C****************************************************************
C****************************************************************
      SUBROUTINE HLOSS                                                  
     &(te,deltt,corat4)
C****************************************************************
C
C  Calculates emission and (net collision) cooling rates for a
C  nn-level hydrogen atom + continuum. The principal levels are
C  assumed degenerate => an (nn*(nn+1))/2 + 1  configuration.
C  (max nn-value is 10 => 56 levels in total.)
C  All lines are treated as optically thin, except Lyman lines,
C  which are treated with escape probabilities.
C  Effective collison strenghts from Anderson et al. (J.Phys.B 
C  2000), for nl collisions between 1 < n < 5. The rates of 
C  Johnson used for other nl collisions.
C  Rates for dl = +-1 when dn = 0 from Brocklehurst 1973).
C  (Recombination is treated as a downward collision term)
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
      parameter (md=75,levhmx=55,levh=levhmx+1,nnhmax=10)
      parameter (mmatr=80)
C
      common/hydaij/ahydij(levhmx,levhmx),ahydsm(levhmx)
      common/hrecij/rechij(25,levhmx),terech(25),rechsum(25)
      common/hjohns/cijjoh(9,levhmx,levhmx),njohn(286),mjohn(286),nmjo
      common/hhelos/edens,pdens,he2dens,he3dens,ta12i,ta12o,f21,
     &f21two,f31,f32
      common/hlos1/f41,f42,f43,clymh,cbalh,cpaschh,cresth
      COMMON/SPOT/OTSP(7)
      COMMON/CHION/CIONH,HYCO,HEI,HEII,HEII32,HEIIBW                    
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      COMMON/CONSTS/PI,PISQRT,CLIGHT                                    
      COMMON/ABSH/ABS12(md),ABS13(md),ABSHE2(md),ABSHE3(md),
     &ABSO1(md),ABSHE4(MD),ABSHE5(MD),ABSHE6(MD),ABSHE7(MD),
     &ABSHE8(MD)
      COMMON/PHOTOS/TEBB,RPA15                                          
      COMMON/RADIE/R(md)                                                
      COMMON/LUMI/FSURF(-13:75)                                         
      COMMON/TEM/TEMPA(md)                                               
      COMMON/PHEXC/SINT(300)                                            
      COMMON/IND/I                                                      
      COMMON/PHY/DEN(md)                                                
      COMMON/ABUN/AB(15)                                                
      COMMON/ABU/XA(2,117)                                               
      common/hyddt/qxh(md,2,levh),hradm(levh,levh),c(levh,levh)
      common/recomb/al1h,al1he1,al1he2
      common/colexc/cc(levh,levh)
      common/aggarw/teagh(9),teaghe(10),agh(15,15,9),aghe(10,10,10)
      common/nontherm/zexc(2),inonth(117)
C                                                                       
      DIMENSION talyi(nnhmax),talyo(nnhmax),DN(levh),as(nnhmax),
     &ss(levhmx),chi(levhmx),om(levhmx,levhmx),epshh(nnhmax),
     &plinh(nnhmax),fracnonth(levh),teaghh(9)
C
      DATA ELCH/1.60219D-12/,SS/2.D0,2.D0,6.D0,2.D0,6.D0,10.D0,
     &2.d0,6.d0,10.d0,14.d0,2.d0,6.d0,10.d0,14.d0,18.d0,2.d0,6.d0,
     &10.d0,14.d0,18.d0,22.d0,2.d0,6.d0,10.d0,14.d0,18.d0,22.d0,
     &26.d0,2.d0,6.d0,10.d0,14.d0,18.d0,22.d0,26.d0,30.d0,
     &2.d0,6.d0,10.d0,14.d0,18.d0,22.d0,26.d0,30.d0,34.d0,
     &2.d0,6.d0,10.d0,14.d0,18.d0,22.d0,26.d0,30.d0,34.d0,38.d0/
C
      data chi/0.0d0,2*10.1985d0,3*12.0871d0,4*12.7481d0,5*
     &13.0541d0,6*13.2203d0,7*13.3205d0,8*13.3855d0,9*13.4301d0,
     &10*13.4620d0/
      DATA TEAGHH/2.5d3,5.d3,7.5d3,1.d4,1.5d4,2.d4,3.d4,4.d4,5.d4/
C
      DO IJ=1,levh
	 fracnonth(ij)=0.0d0
         DO II=1,levh
            A(II,IJ)=0.0D0                                                    
            hradm(ii,ij)=0.0d0
            C(II,IJ)=0.0D0
         ENDDO
      ENDDO
      TEV=TE/1.1605D4
      TTLOG=DLOG10(TE)
      T4=TE/1.D4
      tryd=te/1.5779d5
      fracnonth(2)=0.062d0
      fracnonth(3)=0.779d0
      fracnonth(4)=0.011d0
      fracnonth(5)=0.113d0
      fracnonth(6)=0.018d0
      fracnonth(8)=0.017d0
C
C -- Collisional ion. (Mihalas 1967, Ap. J. 149, 169.)
C
      CONST3=5.465D-11*DSQRT(TE)
      do nnn=1,nnhmax
         nnij=(nnn*(nnn-1))/2
	 evnn=13.542d0-13.595d0*(1.d0-1.d0/(dble(nnn))**2)
         call GAMMIH(1,nnn,evnn,TE,CGAMM)
         if(nnn.eq.1) CIONH=CONST3*CGAMM
	 do l_n=1,nnn
            c(nnij+l_n,levh)=const3*cgamm
	 enddo
      enddo
C -- To agree with Lotz..                                                
C     CIONH=1.36D0*CIONH*((T4/1.1605D0)**(-0.138))                      
C
C -- Collisional excitation --
C
      CCONS=8.63D-6/DSQRT(TE)                                     
       do 7870 iis=1,7
       if(te.ge.teagh(iis).and.te.lt.teagh(iis+1)) then
        iit=iis
       endif
 7870  continue
      if(te.lt.teagh(1)) iit=1
      if(te.ge.teagh(8)) iit=7
      cte=dlog10(te/teagh(iit))/dlog10(teagh(iit+1)/teagh(iit))
       do 7880 iis=1,8
       if(te.ge.teaghh(iis).and.te.lt.teaghh(iis+1)) then
        iitt=iis
       endif
 7880  continue
      if(te.lt.teaghh(1)) iitt=1
      if(te.ge.teaghh(9)) iitt=8
      ctee=dlog10(te/teaghh(iitt))/dlog10(teaghh(iitt+1)/
     &            teaghh(iitt))
C -- Anderson et al. (J.Phys.B, 2000) for 5 lowest n-levels.
      do 7871 ip=1,10
       do 7872 jp=ip+1,15
       if(agh(ip,jp,1).ne.0.0d0) then
        om(ip,jp)=agh(ip,jp,iit)*(agh(ip,jp,iit+1)/agh(ip,jp,iit))**cte
        c(jp,ip)=ccons*om(ip,jp)/ss(jp)
        c(ip,jp)=(ss(jp)/ss(ip))*c(jp,ip)*expfn((chi(jp)-chi(ip))/tev)
       endif
 7872  continue
 7871 continue
C -- Johnson's coll.exc. rates for higher levels
      do nmj=1,nmjo
         ip=mjohn(nmj)
         jp=njohn(nmj)
C	 if(ip.le.15.and.jp.le.15) goto 7873
	 if(jp.le.15) goto 7873
         om(ip,jp)=cijjoh(iitt,ip,jp)*
     &            (cijjoh(iitt+1,ip,jp)/cijjoh(iitt,ip,jp))**ctee
         c(jp,ip)=ccons*om(ip,jp)/ss(jp)
         c(ip,jp)=(ss(jp)/ss(ip))*c(jp,ip)*expfn((chi(jp)-chi(ip))/tev)
 7873    continue
      enddo
C
C -- Collisional rates within principal quantum states.
C
      do nnn=2,nnhmax
         nnij=(nnn*(nnn-1))/2
         do nnas=1,nnn
	    as(nnas)=ahydsm(nnij+nnas)
	 enddo
         sdf=brockl(nnn,as,edens,pdens,he2dens,he3dens,te,1)
         do nnas=1,nnn-1
	    nnijas=nnij+nnas
	    c(nnijas,nnijas+1)=cc(nnijas,nnijas+1)
	    c(nnijas+1,nnijas)=cc(nnijas+1,nnijas)
	 enddo
      enddo
C
C -- Line center optical depths in Lyman lines. --
C
      talyi(2)=ta12i
      talyo(2)=ta12o
      a31const=0.421875d0/ahydij(3,1)
      do ij=3,nnhmax
         nnij=(ij*(ij-1))/2
	 xnnn=(dble(ij))**6/((dble(ij))**2-1.d0)**3
	 tlnl2=a31const*xnnn*ahydij(nnij+2,1)
	 talyi(ij)=tlnl2*talyi(2)
	 talyo(ij)=tlnl2*talyo(2)
      enddo
C
C -- Escape from the shell --                                           
C                                                                       
      do iis=2,nnhmax
         call esc(talyi(iis),0.0d0,0.0d0,pesci,plini,pconti,
     &	          0.0d0)
         call esc(talyo(iis),0.0d0,0.0d0,pesco,plino,pconto,
     &	          0.0d0)
         epshh(iis)=0.5d0*(pesci+pesco)
         plinh(iis)=0.5d0*(plini+plino)
      enddo
C
C -- Recombination rates (OK for 10 K < T < 1e7 K.)
C    [Written as downward collisions c(levh,i)]
C
      do iis=1,24
         if(ttlog.ge.terech(iis).and.ttlog.lt.terech(iis+1)) 
     &	 then
            iit=iis
         endif
      enddo
      if(ttlog.lt.terech(1)) iit=1
      if(ttlog.ge.terech(25)) iit=24
      cte=(ttlog-terech(iit))/(terech(iit+1)-terech(iit))
      do iis=1,levhmx
	 c(levh,iis)=1.d1**(rechij(iit,iis)*(1.d0-cte)+
     &	 cte*rechij(iit+1,iis))
      enddo
      al1h=c(levh,1)
C
C -- Set up the matrix (the radiative part)                             
C
      DO J=1,levh
         hradm(j,j)=1.d0
      ENDDO
C
C   i = lower level, j = upper level  (phex18 = Ly-gamma photo exc.)
C
      phex18=0.0d0
      do nnn=2,nnhmax
         lmax_n=nnn
         do mmm=1,nnn-1
            lmax_m=mmm
            do l_n=1,lmax_n
               do l_m=1,lmax_m
                  if((iabs(l_n-l_m)).eq.1) then
                    j=((nnn-1)*nnn)/2+l_n
                    iii=((mmm-1)*mmm)/2+l_m
                    hradmij=-deltt*ahydij(j,iii)
		    if(iii.eq.1) hradmij=epshh(nnn)*hradmij
		    hradm(iii,j)=hradmij
		    hradm(j,j)=hradm(j,j)-hradmij
                  endif
               enddo
            enddo
         enddo
      enddo
C
      hradm(1,2)=hradm(1,2)-deltt*ahydij(2,1)
      hradm(2,2)=hradm(2,2)+deltt*ahydij(2,1)
C     hradm(1,1)=hradm(1,1)+deltt*phex18
C     hradm(8,1)=-deltt*phex18
C
C --  Add non-thermal excitation
C
      hradm(1,1)=hradm(1,1)+deltt*zexc(1)*(fracnonth(2)+
     &fracnonth(3)+fracnonth(4)+fracnonth(5)+fracnonth(6)+
     &fracnonth(8))
      hradm(2,1)=hradm(2,1)-deltt*zexc(1)*fracnonth(2)
      hradm(3,1)=hradm(3,1)-deltt*zexc(1)*fracnonth(3)
      hradm(4,1)=hradm(4,1)-deltt*zexc(1)*fracnonth(4)
      hradm(5,1)=hradm(5,1)-deltt*zexc(1)*fracnonth(5)
      hradm(6,1)=hradm(6,1)-deltt*zexc(1)*fracnonth(6)
      hradm(8,1)=hradm(8,1)-deltt*zexc(1)*fracnonth(8)
C
C --- Emission + recomb.rate (uses previous time step for level pops.)
C
      do io=1,levh
         dn(io)=qxh(i,1,io)
      enddo
      corat4=ab(1)*den(i)*c(levh,1)*edens*dn(levh)
      F21=EPSHH(2)*ELCH*DN(3)*AHYDIJ(3,1)*chi(3)/EDENS
      F31=EPSHH(3)*ELCH*DN(5)*AHYDIJ(5,1)*chi(5)/EDENS
      F41=EPSHH(4)*ELCH*DN(8)*AHYDIJ(8,1)*chi(8)/EDENS
      F21TWO=ELCH*DN(2)*AHYDIJ(2,1)*chi(2)/EDENS
      F32=ELCH*(chi(4)-chi(2))*(DN(4)*AHYDIJ(4,3)+DN(5)*AHYDIJ(5,2)
     &+DN(6)*AHYDIJ(6,3))/edens
      F42=ELCH*(chi(7)-chi(2))*(DN(7)*AHYDIJ(7,3)+DN(8)*AHYDIJ(8,2)
     &+DN(9)*AHYDIJ(9,3))/edens
      F43=ELCH*(chi(7)-chi(4))*(DN(7)*AHYDIJ(7,5)+DN(8)*(AHYDIJ(8,4)
     &+AHYDIJ(8,6))+DN(9)*AHYDIJ(9,5)+DN(10)*AHYDIJ(10,6))/edens
C
C -- Cooling rates due to collisional excitation (net rates).
C
      clymh=0.0d0
      do ij=2,levhmx
	 clymh=clymh+elch*chi(ij)*(dn(1)*c(1,ij)-dn(ij)*c(ij,1))
      enddo
      cbalh=0.0d0
      do ij=4,levhmx
	 cbalh=cbalh+elch*(chi(ij)-chi(2))*(dn(2)*c(2,ij)-dn(ij)*
     &	       c(ij,2))
	 cbalh=cbalh+elch*(chi(ij)-chi(3))*(dn(3)*c(3,ij)-dn(ij)*
     &	       c(ij,3))
      enddo
      cpaschh=0.0d0
      do ij=7,levhmx
	 cpaschh=cpaschh+elch*(chi(ij)-chi(4))*(dn(4)*c(4,ij)-
     &	 dn(ij)*c(ij,4))
	 cpaschh=cpaschh+elch*(chi(ij)-chi(5))*(dn(5)*c(5,ij)-
     &	 dn(ij)*c(ij,5))
	 cpaschh=cpaschh+elch*(chi(ij)-chi(6))*(dn(6)*c(6,ij)-
     &	 dn(ij)*c(ij,6))
      enddo
      cresth=0.0d0
      do nnn=4,nnhmax-1
         lmax_n=nnn
         do mmm=nnn+1,nnhmax
            lmax_m=mmm
            do l_n=1,lmax_n
               do l_m=1,lmax_m
                  ii=((nnn-1)*nnn)/2+l_n
                  ij=((mmm-1)*mmm)/2+l_m
                  cresth=cresth+elch*(chi(ij)-chi(ii))*
     &		         (dn(ii)*c(ii,ij)-dn(ij)*c(ij,ii))
               enddo
            enddo
         enddo
      enddo
      if(clymh.le.0.0d0) clymh=1.d-50
      if(cbalh.le.0.0d0) cbalh=1.d-50
      if(cpaschh.le.0.0d0) cpaschh=1.d-50
      if(cresth.le.0.0d0) cresth=1.d-50
C                                                                       
      RETURN                                                            
      END                                                               
C************************************************************
C************************************************************           
      SUBROUTINE HELOSS                                                 
     &(te,corat1,corat2,corat3,deltt,ite,teinn,corat4,corat5,
     &corat6,corat7,corat8,dt12c,dt13c,dt14c,dthe1c)
C************************************************************           
C
C  Calculates the coll. excitation cooling of a 72-level He atom.        
C  16 levels for He I, 55 levels for He II (all levels have
C  degenerate levels included) + He III.
C  He I data from Almog & Netzer (MNRAS, 238, 57 (1989)) and Verner
C  and Ferland (1996).
C  He II H-alpha is assumed optically thin, while L-lines are treated
C  with escape and destruction probabilities.        
C  Effective collision strengths for He II from Aggarwal et al., 
C  1991b. (Updated for 1s - 2 using Kisielius et al. (1995).)
C  Rates for collisional transitions between angular momentum
C  states from Brocklehurst (MNRAS, 153, 471 (1971)).
C  The line transfer of He II L-alpha takes the Bowen fluorescence
C  into account (skipped at the moment).
C  (updated March-96.)
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      parameter (md=75,levhemx=71,levhe=levhemx+1,nnhemax=10)
      parameter (mmatr=80,levhmx=55,levh=levhmx+1,levhe1=16)
C                                                                       
      common/hhelos/edens,pdens,he2dens,he3dens,ta12i,ta12o,f21,
     &f21two,f31,f32
      common/helos1/tao1i,tao1o,deno3,clymhe,cbalhe,cresthe,
     &he1cool,tahe1i,tahe2i,tahe3i,tahe4i,tahe1o,tahe2o,tahe3o,
     &tahe4o,tc12i,tc12o,tc13i,tc13o,tc14i,tc14o,tche1i,tche1o
      common/helaij/ahelij(levhmx,levhmx),ahelsm(levhmx)
      common/herecij/recheij(25,levhmx),tereche(25),rechesum(25)
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      COMMON/ABUN/AB(15)
      COMMON/HESTUFF/ALDHE,CHEO,CHE
      COMMON/ABC/AL(7),ALN(8),ALO(9),ALC(7),ALS(15),ALMG(11),
     &ALAL(12),COO(8),COC(6),CON(7)
      COMMON/CONSTS/PI,PISQRT,CLIGHT
      COMMON/EDAYY/TDAY
      COMMON/IND/I
      COMMON/ABSH/ABS12(md),ABS13(md),ABSHE2(md),ABSHE3(md),
     &ABSO1(md),ABSHE4(MD),ABSHE5(MD),ABSHE6(MD),ABSHE7(MD),
     &ABSHE8(MD)
      COMMON/TAUXX/TAXA(md,-13:75)
      COMMON/PHOTOS/TEBB,RPA15
      COMMON/RADIE/R(md)
      COMMON/LUMI/FSURF(-13:75)
      COMMON/YIELDA/YLA(md),YB(md),YO2(md),YA(md),IYIELD,KYIELD
      COMMON/YIELDB/PDEST(md)
      common/yieldc/ela(md),plinla(md),pconla(md),po(md)
      COMMON/TEM/TEMPA(md)
      COMMON/PHEXC/SINT(300)
      COMMON/BWLOSS/FOIII(md,8)
      common/hedt/qxhe(md,2,levhe),heradm(levhe,levhe),
     &cc(levhe,levhe)
      COMMON/PHY/DEN(md)
      COMMON/ABU/XA(2,117)
      COMMON/HEOPT/OPTHEL(10,md)
      common/recomb/al1h,al1he1,al1he2
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)
      COMMON/ION/XB(md,117)
      common/colexc/ccc(levh,levh)
      common/aggarw/teagh(9),teaghe(10),agh(15,15,9),aghe(10,10,10)
      common/nontherm/zexc(2),inonth(117)
C                                                                       
      DIMENSION DN(levhe),as(nnhemax),WBOW(8),WVEBOW(8),aherec(levhe1)
      dimension ome(levhe1,levhe1),ene(levhe1),g(levhe1),
     &aij(levhe1,levhe1),talyi(nnhemax),talyo(nnhemax),ss(levhmx),
     &chi(levhmx),om(levhmx,levhmx),epshhe(nnhemax),plinhe(nnhemax),
     &pconhe(nnhemax),eesche(nnhemax),fracnonth(levhe1)
C
      DATA ELCH/1.60219D-12/,SS/2.D0,2.D0,6.D0,2.D0,6.D0,10.D0,
     &2.d0,6.d0,10.d0,14.d0,2.d0,6.d0,10.d0,14.d0,18.d0,2.d0,6.d0,
     &10.d0,14.d0,18.d0,22.d0,2.d0,6.d0,10.d0,14.d0,18.d0,22.d0,
     &26.d0,2.d0,6.d0,10.d0,14.d0,18.d0,22.d0,26.d0,30.d0,
     &2.d0,6.d0,10.d0,14.d0,18.d0,22.d0,26.d0,30.d0,34.d0,
     &2.d0,6.d0,10.d0,14.d0,18.d0,22.d0,26.d0,30.d0,34.d0,38.d0/
C
      data chi/0.0d0,2*40.794d0,3*48.3484d0,4*50.9924d0,5*
     &52.2164d0,6*52.8812d0,7*53.282d0,8*53.542d0,9*53.7204d0,
     &10*53.848d0/
      DATA WVEBOW/303.621D0,3023.D0,3047.D0,3133.D0,3312.D0,            
     &3341.D0,3429.D0,3444.D0/                                          
      DATA WBOW/1.D0,0.04D0,0.114D0,0.66D0,0.057D0,0.096D0,             
     &0.068D0,0.20D0/                                                   
      DATA G/1.D0,3.D0,1.D0,9.D0,3.D0,3.D0,9.D0,15.D0,3.D0,9.D0,
     &15.D0,21.D0,3.30D2,1.53D3,1.4625D4,9.98474D5/
      DATA ENE/0.0D0,19.8201D0,20.6158D0,20.9642D0,21.2181D0,
     &22.7186D0,23.0072D0,23.0738D0,23.5941D0,23.7081D0,23.7363D0,
     &23.7373D0,24.2089D0,24.4332D0,24.545D0,24.5833D0/
C
      do ij=1,levhe1
	 fracnonth(ij)=0.0d0
         do ii=1,levhe1
            ome(ii,ij)=0.0d0
            aij(ii,ij)=0.0d0
         enddo 
      enddo
      DO IJ=1,levhe
         DO II=1,levhe
            A(II,IJ)=0.0D0
            heradm(ii,ij)=0.0d0
            cc(II,IJ)=0.0D0
         enddo
      enddo
      TEV=TE/1.1605D4
      T4=TE/1.D4
      T5000=2.D0*T4
      TTLOG=dlog10(te)
      fracnonth(2)=0.032d0
      fracnonth(3)=0.062d0
      fracnonth(4)=0.024d0
      fracnonth(5)=0.671d0
      fracnonth(6)=0.010d0
      fracnonth(7)=0.010d0
C
C -- Line center optical depths in He II Lyman lines. --
C
      talyi(2)=ta12i
      talyo(2)=ta12o
      a31const=0.421875d0/ahelij(3,1)
      do ij=3,nnhemax
         nnij=(ij*(ij-1))/2
	 xnnn=(dble(ij))**6/((dble(ij))**2-1.d0)**3
	 tlnl2=a31const*xnnn*ahelij(nnij+2,1)
	 talyi(ij)=tlnl2*talyi(2)
	 talyo(ij)=tlnl2*talyo(2)
      enddo
C
C -- Escape from the shell (skip Bowen fluorescence for now) --
C    (Include continuum absorption for the cooling
C    Net loss from line is escape + desctruction.)
C
      cllcons=1.d0/(abshe2(i)*den(i))
      do iis=2,nnhemax
	 if(iis.eq.2) then
            conopa=taxa(i,11)
	    ttcci=tc12i
	    ttcco=tc12o
	    dttcc=dt12c
	 elseif(iis.eq.3) then
            conopa=taxa(i,12)+den(i)*(ab(4)*si(8,13)*xa(2,8)+
     &	    ab(5)*si(20,13)*xa(2,20)+ab(6)*si(28,13)*xa(2,28))
	    ttcci=tc13i
	    ttcco=tc13o
	    dttcc=dt13c
	 else
            conopa=taxa(i,13)
	    ttcci=tc14i
	    ttcco=tc14o
	    dttcc=dt14c
	 endif
         clla=conopa*cllcons*talyi(iis)/talyi(2)
         call esc(talyi(iis),ttcci,clla,pesci,plini,pconti,dttcc)
         call esc(talyo(iis),ttcco,clla,pesco,plino,pconto,dttcc)
         eesche(iis)=0.5d0*(pesci+pesco)
         pconhe(iis)=0.5d0*(pconti+pconto)
         plinhe(iis)=0.5d0*(plini+plino)
         epshhe(iis)=pconhe(iis)+eesche(iis)
         if(iis.eq.2) ela(i)=eesche(iis)
         if(iis.eq.2) pconla(i)=pconhe(iis)
      enddo
C
C -- He I resonance lines --
C   (from level 3 to 1 is 2-photon.)
C
      cl1=taxa(i,5)/(abshe4(i)*den(i))
C      cl2=taxa(i,5)/(abshe5(i)*den(i))
      cl3=taxa(i,5)/(abshe6(i)*den(i))
      cl4=taxa(i,5)/(abshe7(i)*den(i))
c
      call esc(tahe1i,tche1i,cl1,pesci,plini,pconti,dthe1c)
      call esc(tahe1o,tche1o,cl1,pesco,plino,pconto,dthe1c)
      esc21=0.5d0*(pesci+pesco)
      plin21=0.5d0*(plini+plino)
      pcon21=0.5d0*(pconti+pconto)
c
C      call esc(tahe2i,tche1i,cl2,pesci,plini,pconti,dthe1c)
C      call esc(tahe2o,tche1o,cl2,pesco,plino,pconto,dthe1c)
C      esc31=0.5d0*(pesci+pesco)
C      plin31=0.5d0*(plini+plino)
C      pcon31=0.5d0*(pconti+pconto)
c
      call esc(tahe3i,tche1i,cl3,pesci,plini,pconti,dthe1c)
      call esc(tahe3o,tche1o,cl3,pesco,plino,pconto,dthe1c)
      esc41=0.5d0*(pesci+pesco)
      plin41=0.5d0*(plini+plino)
      pcon41=0.5d0*(pconti+pconto)
c
      call esc(tahe4i,tche1i,cl4,pesci,plini,pconti,dthe1c)
      call esc(tahe4o,tche1o,cl4,pesco,plino,pconto,dthe1c)
      esc51=0.5d0*(pesci+pesco)
      plin51=0.5d0*(plini+plino)
      pcon51=0.5d0*(pconti+pconto)
c
      ep21=esc21+pcon21
C     ep31=esc31+pcon31
      ep41=esc41+pcon41
      ep51=esc51+pcon51
C
C -- Collisional rates within principal quantum states of He II.
C
      do nnn=2,nnhemax
         nnij=(nnn*(nnn-1))/2
         do nnas=1,nnn
	    as(nnas)=ahelsm(nnij+nnas)
	 enddo
         sdf=brockl(nnn,as,edens,pdens,he2dens,he3dens,te,2)
         do nnas=1,nnn-1
	    nnijas=nnij+nnas
	    cc(nnijas+levhe1+1,nnijas+levhe1)=ccc(nnijas+1,nnijas)
	    cc(nnijas+levhe1,nnijas+levhe1+1)=ccc(nnijas,nnijas+1)
	 enddo
      enddo
C
C -- Collisional excitation of He II --
C
      CCONS=8.63D-6/DSQRT(TE)                                     
       do 7870 iis=1,9
       if(te.ge.teaghe(iis).and.te.lt.teaghe(iis+1)) then
        iit=iis
       endif
 7870  continue
      if(te.lt.teaghe(1)) iit=1
      if(te.ge.teaghe(10)) iit=9
      cte=dlog10(te/teaghe(iit))/dlog10(teaghe(iit+1)/teaghe(iit))
      do 7871 ip=1,8
       do 7872 jp=ip+1,10
       if(aghe(ip,jp,1).ne.0.0d0) then
        om(ip,jp)=aghe(ip,jp,iit)*(aghe(ip,jp,iit+1)/aghe(ip,jp,iit))
     &  **cte
        cc(jp+16,ip+16)=ccons*om(ip,jp)/ss(jp)
        cc(ip+16,jp+16)=(ss(jp)/ss(ip))*cc(jp+16,ip+16)*expfn((chi
     &  (jp)-chi(ip))
     &  /tev)
       endif
 7872  continue
 7871 continue
C
C -- For He I we take collision strengths from Almog & Netzer (MNRAS -89)
C    (T=1E4 K), except between the lowest 5 levels (Mendoza).
C
      ome(1,2)=7.36d-2*t4**.0975
      ome(1,3)=4.35d-2*t4**.469
      ome(1,4)=2.13d-2*t4**.798
      ome(1,5)=1.64d-2*t4**1.062
      ome(1,6)=1.80d-2
      ome(1,7)=8.40d-3
      ome(1,8)=2.60d-3
      ome(2,3)=dmax1(2.40d0,2.40d0*t4**.195)
      ome(2,4)=29.5d0*t4**.973
      ome(2,5)=1.03d0*t4**.332
      ome(2,6)=2.47d0
      ome(2,7)=1.83d0
      ome(2,8)=2.14d0
      ome(2,10)=7.2d-2
C -- (2,11) from CF
      ome(2,11)=0.261d0
      ome(2,13)=0.056d0
      ome(3,4)=1.66d0*t4**.178
      ome(3,5)=19.1d0*t4**.866
      ome(3,6)=0.59d0
      ome(3,7)=0.56d0
      ome(3,8)=0.35d0
      ome(4,5)=4.05d0*t4**.221
      ome(4,6)=5.93d0
      ome(4,7)=13.7d0
      ome(4,8)=15.4d0
      ome(4,9)=1.96d0
      ome(4,11)=3.80d0
      ome(4,13)=2.9d0
      ome(5,6)=1.16d0
      ome(5,7)=1.76d0
      ome(5,8)=1.52d0
      ome(6,7)=260.d0
      ome(6,8)=2.8d0
      ome(6,9)=0.4d0
      ome(6,10)=1.5d0
      ome(6,11)=0.2d0
      ome(6,12)=0.2d0
      ome(6,13)=0.6d0
      ome(7,8)=2.3d3
      ome(7,9)=3.3d2
      ome(7,10)=2.6d0
      ome(7,11)=1.5d2
      ome(7,13)=3.9d1
      ome(8,10)=38.d0
      ome(8,12)=7.5d2
      ome(8,13)=1.4d2
      ome(8,14)=0.1d0
      ome(9,10)=1.8d3
      ome(9,13)=6.2d1
      ome(9,14)=2.0d1
      ome(10,11)=1.9d4
      ome(10,13)=2.1d2
      ome(10,14)=2.0d1
      ome(11,12)=1.9d5
      ome(11,13)=3.1d2
      ome(11,14)=2.0d1
      ome(12,13)=4.4d2
      ome(12,14)=2.0d1
      ome(13,14)=3.0d5
      ome(13,15)=1.0d4
      ome(13,16)=2.0d3
      ome(14,15)=4.0d6
      ome(14,16)=1.0d5
      ome(15,16)=4.0d8
       do 1101 ii=1,15
        do 1102 jj=ii+1,16
        cc(jj,ii)=(ccons*ome(ii,jj))/g(jj)
        exiijj=expfn((ene(jj)-ene(ii))/tev)
        cc(ii,jj)=(cc(jj,ii)*g(jj)*exiijj)/g(ii)
 1102  continue
 1101 continue
C                                                                       
C  Ly-alpha transfer (Bowen fluorescence). 
C  Follow Kallman & McCray: Neglect continuum abs. for calculations 
C                           of escape probs. entering the BOWEN subr.
C                           (The observed spectrum.)
C                                                                       
C      conopa=taxa(i,11)
C      tao2i=(0.059/0.11)*tao1i
C      tao2o=(0.059/0.11)*tao1o
C      call esc(tao2i,0.0d0,0.0d0,pesci,plini,pconti,0.0d0)
C      call esc(tao2o,0.0d0,0.0d0,pesco,plino,pconto,0.0d0)
C      eo2=0.5d0*(pesci+pesco)
C      call esc(ta12i,0.0d0,0.0d0,pesci,plini,pconti,0.0d0)
C      call esc(ta12o,0.0d0,0.0d0,pesco,plino,pconto,0.0d0)
C      ela(i)=0.5d0*(pesci+pesco)
C     CALL BOWEN(EO2,TE,DENO3,DENHE2,CONOPA,YLA(I),YB(I),YO2(I),YA(I),
C    &ela(i))
      yla(i)=1.d0
      yb(i)=0.0d0
      yo2(i)=0.0d0
      ya(i)=0.0d0
C
C -- Recombination rates for He II (OK for 40 K < T < 2e7 K.)
C    [Written as downward collisions c(levh,i)]
C
      do iis=1,24
         if(ttlog.ge.tereche(iis).and.ttlog.lt.tereche(iis+1)) 
     &	 then
            iit=iis
         endif
      enddo
      if(ttlog.lt.tereche(1)) iit=1
      if(ttlog.ge.tereche(25)) iit=24
      cte=(ttlog-tereche(iit))/(tereche(iit+1)-tereche(iit))
      do iiss=levhe1+1,levhemx
	 iis=iiss-levhe1
	 cc(levhe,iiss)=1.d1**(recheij(iit,iis)*(1.d0-cte)+
     &	 cte*recheij(iit+1,iis))
      enddo
      al1he2=cc(levhe,levhe1+1)
C
C -- Collisional ionization
C
      cc(levhe1+1,levhe)=che
      cc(1,levhe1+1)=cheo
C
C -- Recombination rates for He I. (Verner & Ferland; Almog & Netzer)
C
      call HE1REC(te,al1he1,atothe1,aherec)
      cc(levhe1+1,1)=aldhe+aherec(1)
      do iher=2,levhe1
         cc(levhe1+1,iher)=aherec(iher)
      enddo
C                                                                       
C -- Set up the matrix (the radiative part)                             
C
      do 1000 j=1,levhe
 1000 heradm(j,j)=1.d0
C
      aij(2,1)=1.27d-4*ep21
      aij(3,1)=5.13d1
      aij(4,1)=1.6d2*ep41
      aij(5,1)=1.86d9*ep51
      aij(4,2)=0.1022d8
      aij(7,2)=0.09478d8
      aij(10,2)=0.0505d8
      aij(13,2)=4.287d5
      aij(14,2)=1.97d4
      aij(15,2)=8.64d2
      aij(16,2)=3.71d0
      aij(5,3)=0.01976d8
      aij(6,4)=0.2786d8
      aij(8,4)=0.7053d8
      aij(9,4)=0.106d8
      aij(11,4)=0.251d8
      aij(13,4)=1.069d6
      aij(14,4)=5.93d4
      aij(15,4)=2.6d3
      aij(16,4)=11.1d0
      aij(7,6)=0.0108d8
      aij(10,6)=0.00608d8
      aij(13,6)=2.6d4
      aij(14,6)=6.44d3
      aij(15,6)=2.71d2
      aij(16,6)=1.14d0
      aij(8,7)=1.28d4
      aij(9,7)=0.0652d8
      aij(11,7)=0.0668d8
      aij(13,7)=2.5d5
      aij(14,7)=1.93d4
      aij(15,7)=8.13d2
      aij(16,7)=3.44d0
      aij(10,8)=0.00597d8
      aij(12,8)=0.139d8
      aij(13,8)=5.9d5
      aij(14,8)=3.22d4
      aij(15,8)=1.35d3
      aij(16,8)=5.74d0
      aij(10,9)=0.00227d8
      aij(13,9)=2.1d5
      aij(14,9)=3.03d5
      aij(15,9)=1.2d2
      aij(16,9)=0.49d0
      aij(11,10)=4.15d3
      aij(13,10)=1.0d5
      aij(14,10)=9.11d3
      aij(15,10)=3.6d2
      aij(16,10)=1.49d0
      aij(12,11)=8.15d-2
      aij(13,11)=3.2d5
      aij(14,11)=1.51d4
      aij(15,11)=5.98d2
      aij(16,11)=2.48d0
      aij(13,12)=7.5d5
      aij(14,12)=2.12d4
      aij(15,12)=8.35d2
      aij(16,12)=3.47d0
      aij(14,13)=1.5d5
      aij(15,13)=4.32d3
      aij(16,13)=1.67d1
      aij(15,14)=6.81d3
      aij(16,14)=1.8d1
      aij(16,15)=3.69d1
       do 2011 ii=2,16
        do 2012 jj=1,ii-1
        heradm(ii,ii)=heradm(ii,ii)+deltt*aij(ii,jj)
        heradm(jj,ii)=-deltt*aij(ii,jj)
 2012   continue
 2011  continue
C
      phex13=0.0d0
C
C   i = lower level, j = upper level  (phex13 = Ly-beta photo exc.)
C
      do nnn=2,nnhemax
         lmax_n=nnn
         do mmm=1,nnn-1
            lmax_m=mmm
            do l_n=1,lmax_n
               do l_m=1,lmax_m
                  if((iabs(l_n-l_m)).eq.1) then
                    j=((nnn-1)*nnn)/2+l_n
                    iii=((mmm-1)*mmm)/2+l_m
                    heradmij=-deltt*ahelij(j,iii)
		    if(iii.eq.1) heradmij=epshhe(nnn)*heradmij
		    heradm(iii+levhe1,j+levhe1)=heradmij
		    heradm(j+levhe1,j+levhe1)=
     &                   heradm(j+levhe1,j+levhe1)-heradmij
                  endif
               enddo
            enddo
         enddo
      enddo
C
      levhe11=levhe1+1
      levhe12=levhe1+2
      heradm(levhe11,levhe12)=heradm(levhe11,levhe12)-
     &                        deltt*ahelij(2,1)
      heradm(levhe12,levhe12)=heradm(levhe12,levhe12)+
     &                        deltt*ahelij(2,1)
C     heradm(levhe11,levhe11)=heradm(levhe11,levhe11)+
C    &                        deltt*phex13
C     heradm(levhe11+4,levhe11)=-deltt*phex13
C
C --  Add non-thermal excitation
C
      heradm(1,1)=heradm(1,1)+deltt*zexc(2)*(fracnonth(2)+
     &fracnonth(3)+fracnonth(4)+fracnonth(5)+fracnonth(6)+
     &fracnonth(7))
      heradm(2,1)=heradm(2,1)-deltt*zexc(2)*fracnonth(2)
      heradm(3,1)=heradm(3,1)-deltt*zexc(2)*fracnonth(3)
      heradm(4,1)=heradm(4,1)-deltt*zexc(2)*fracnonth(4)
      heradm(5,1)=heradm(5,1)-deltt*zexc(2)*fracnonth(5)
      heradm(6,1)=heradm(6,1)-deltt*zexc(2)*fracnonth(6)
      heradm(7,1)=heradm(7,1)-deltt*zexc(2)*fracnonth(7)
C
      DO 1003 J=1,levhe
 1003 dn(j)=qxhe(i,1,j)                                                 
C
C -- He II Ly-alpha/beta and H-alpha emission.
C
      F21=ela(i)*ELCH*DN(levhe1+3)*ahelij(3,1)*chi(3)/EDENS
      F21TWO=ELCH*DN(levhe1+2)*ahelij(2,1)*chi(2)/EDENS
      F31=eesche(3)*ELCH*DN(levhe1+5)*ahelij(5,1)*chi(5)/EDENS
      F32=ELCH*(chi(4)-chi(3))*(DN(levhe1+4)*ahelij(4,3)+
     &DN(levhe1+5)*ahelij(5,2)+DN(levhe1+6)*ahelij(6,3))/EDENS
C     FYB=(F21*YB(I))/YLA(I)
C     FO2=(F21*YO2(I))/YLA(I)
      DO 2005 IBOW=1,8
C2005 FOIII(I,IBOW)=FYB*WBOW(IBOW)*(303.783/WVEBOW(IBOW))
 2005 FOIII(I,IBOW)=0.0d0
C
      ab2den=ab(2)*den(i)
      elab=(ab(2)*elch)/(12.57d0*edens)
C
C --- Continuum absorption rate (CORAT1 = He II Ly-alpha, CORAT2 = He II 
C     Ly-beta. Skip ionization by Ly-gamma and higher orders.)
C     (CORAT8 = He II 2-photon.)
C                                     (Absolute number rates.)
      corat1=0.0d0
      corat2=0.0d0
      corat8=0.0d0
      if(ite.eq.0.or.te.eq.teinn) goto 9006
         CORAT1=pconla(i)*ahelij(3,1)*DN(levhe1+3)*ab2den
         CORAT2=pconhe(3)*ahelij(5,1)*DN(levhe1+5)*ab2den
         CORAT8=ahelij(2,1)*DN(levhe1+2)*ab2den
 9006 continue
C -- CORAT3 = He I Ly-alpha
      corat3=ab2den*(aij(5,1)*dn(5)*pcon51+aij(4,1)*dn(4)*pcon41+
     &aij(2,1)*dn(2)*pcon21)
C -- CORAT5 = He I recombination to n=1.
      corat5=ab2den*(cc(levhe1+1,1)-aldhe)*edens*dn(levhe1+1)
C -- CORAT4 = He II recombination to n=2. (Adds to H I recombination.)
      corat4=corat4+ab2den*cc(levhe,levhe1+2)*edens*dn(levhe)
C -- CORAT6 = He II recombination to n=1.
      corat6=ab2den*cc(levhe,levhe1+1)*edens*dn(levhe)
C -- CORAT7 = He I 2-photon
      corat7=ab2den*aij(3,1)*dn(3)
C -- More line emission.
C -- He II 4686.
      opthel(1,i)=elab*(chi(4)-chi(3))*(dn(levhe1+7)*ahelij(7,5)+
     &dn(levhe1+8)*(ahelij(8,4)+ahelij(8,6))+dn(levhe1+9)*
     &ahelij(9,5)+dn(levhe1+10)*ahelij(10,6))
C -- He I optical emission lines.
      opthel(2,i)=(elab*aij(8,4)*dn(8)*(ene(8)-ene(4)))
      opthel(3,i)=(elab*aij(4,2)*dn(4)*(ene(4)-ene(2)))
      opthel(4,i)=esc51*(elab*aij(5,1)*dn(5)*(ene(5)-ene(1)))
      opthel(5,i)=(elab*aij(10,2)*dn(10)*(ene(10)-ene(2)))
      opthel(6,i)=(elab*aij(7,2)*dn(7)*(ene(7)-ene(2)))
      opthel(7,i)=(elab*aij(11,4)*dn(11)*(ene(11)-ene(4)))
      opthel(8,i)=(elab*aij(9,4)*dn(9)*(ene(9)-ene(4)))
      opthel(9,i)=(elab*aij(6,4)*dn(6)*(ene(6)-ene(4)))
C                                                                       
C -- Cooling rates due to collisional excitation (net rates).
C    (slightly overestimates the cooling since photons trapped in
C    the continuum provide lower cooling than escaping photons.)
C
      clymhe=0.0d0
      lev1=levhe1+1
      lev2=levhe1+2
      lev3=levhe1+3
      lev4=levhe1+4
      lev5=levhe1+5
      lev6=levhe1+6
      do ij=2,levhmx
	 levij=levhe1+ij
	 clymhe=clymhe+elch*chi(ij)*(dn(lev1)*cc(lev1,levij)-
     &	        dn(levij)*cc(levij,lev1))
      enddo
      cbalhe=0.0d0
      do ij=4,levhmx
	 levij=levhe1+ij
	 cbalhe=cbalhe+elch*(chi(ij)-chi(2))*(dn(lev2)*cc(lev2,levij)-
     &	 dn(levij)*cc(levij,lev2))
	 cbalhe=cbalhe+elch*(chi(ij)-chi(3))*(dn(lev3)*cc(lev3,levij)-
     &	 dn(levij)*cc(levij,lev3))
      enddo
      cpaschhe=0.0d0
      do ij=7,levhmx
	 levij=levhe1+ij
	 cpaschhe=cpaschhe+elch*(chi(ij)-chi(4))*(dn(lev4)*
     &	 cc(lev4,levij)-dn(levij)*cc(levij,lev4))
	 cpaschhe=cpaschhe+elch*(chi(ij)-chi(5))*(dn(lev5)*
     &	 cc(lev5,levij)-dn(levij)*cc(levij,lev5))
	 cpaschhe=cpaschhe+elch*(chi(ij)-chi(6))*(dn(lev6)*
     &	 cc(lev6,levij)-dn(levij)*cc(levij,lev6))
      enddo
      cresthe=cpaschhe
C
      do nnn=4,nnhemax-1
         lmax_n=nnn
         do mmm=nnn+1,nnhemax
            lmax_m=mmm
            do l_n=1,lmax_n
               do l_m=1,lmax_m
                  ii=((nnn-1)*nnn)/2+l_n
                  ij=((mmm-1)*mmm)/2+l_m
		  levii=levhe1+ii
		  levij=levhe1+ij
                  cresthe=cresthe+elch*(chi(ij)-chi(ii))*
     &		          (dn(levii)*cc(levii,levij)-dn(levij)*
     &                    cc(levij,levii))
               enddo
            enddo
         enddo
      enddo
      if(clymhe.le.0.0d0) clymhe=1.d-50
      if(cbalhe.le.0.0d0) cbalhe=1.d-50
      if(cpaschhe.le.0.0d0) cpaschhe=1.d-50
      if(cresthe.le.0.0d0) cresthe=1.d-50
C                                                                       
      he1cool=0.0d0
       do 2301 ii=1,levhe1-1
        do 2302 jj=ii+1,levhe1
        he1cool=he1cool+(ene(jj)-ene(ii))*elch*(dn(ii)*cc(ii,jj)-
     &  dn(jj)*cc(jj,ii))
 2302   continue
 2301  continue
      if(he1cool.le.0.0d0) he1cool=1.d-50
C
      RETURN                                                            
      END                                                               
C************************************************************           
C************************************************************
      SUBROUTINE GAMMIH(i,j,EEV,T,CG)
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      cg=0.0d0
      tev=t/1.1605d4
      te=dmin1(t,1.d5)
      te=dmax1(te,5.d3)
      ttlog=dlog10(te)
      if(i.eq.1) then
	if(j.eq.1) then
	  cg=-0.435d0+0.3d0*ttlog
	elseif(j.eq.2) then
	  cg=1.9987261d1-5.8906298d-5*te-2.8185937d4/te+
     &    5.444416d7/te**2
	elseif(j.eq.3) then
	  cg=1.3935312d3-1.6805859d2*ttlog-2.539d3/ttlog
	elseif(j.eq.4) then
	  cg=2.0684609d3-3.341582d2*ttlog-7.6440625d3/ttlog**2
	elseif(j.eq.5) then
	  cg=3.2174844d3-5.5882422d2*ttlog-6.86325d3/ttlog**2
	elseif(j.eq.6) then
	  cg=5.759125d3-1.5163125d3*ttlog+8.175d1*ttlog**2
	elseif(j.eq.7) then
	  cg=1.461475d4-4.828375d3*ttlog+3.9335938d2*ttlog**2
	elseif(j.eq.8) then
	  cg=2.827925d4-1.017275d4*ttlog+9.1967968d2*ttlog**2
	elseif(j.eq.9) then
	  cg=4.679925d4-1.76275d4*ttlog+1.6742031d3*ttlog**2
	elseif(j.eq.10) then
	  cg=-7.4073d4+6.8599375d3*ttlog+2.01698d5/ttlog
	endif
      endif
      cg=cg*expfn(eev/tev)
C
      RETURN
      END
C************************************************************
C************************************************************
      DOUBLE PRECISION FUNCTION BROCKL(N,A,EDENS,PDENS,HE2DENS,
     &he3dens,temp,ion)
C***********************************************************
C Collisional rates between sublevels of hydrogenic ions.
C See Brocklhurst 1971, MNRAS 153, 471.
C
C   ion = 1  (hydrogen),          ion = 2 (helium)
C   imu = 1  (coll.with protons), imu = 2 (coll.with He II)
C   imu = 3  (coll.with He III),  imu = 4 (coll.with electrons)
C
C roc=dmin1(constd,(10.95d0+dlog10(cons2/a(l+1)**2))) (not used)
C C.Kozma found (1996) that 10.95 should be 11.196
C***********************************************************
      IMPLICIT REAL*8(A-H,O-Z)
      parameter (md=75,levhmx=55,levh=levhmx+1)
C
      common/colexc/c(levh,levh)
C
      dimension a(10),reduc(4),zrat(4),amu(4),xfr(4),cc(0:9),
     &xl(0:9),w(0:9),d(0:9),cl(0:9,0:9),ch(0:9,0:9)
C
      mpome=1836.d0
C
C --- Avoid negative rates for low temperatures
      te=temp
c      if(ion.eq.1.and.temp.le.7.5d2) te=7.5d2
c      if(ion.eq.2.and.temp.le.3.d3) te=3.d3
C
      CONS=9.933D-6/DSQRT(TE)
c -- to correspond with Osterbrock -89 (p.94) at 1E4 K,
c -- multiply by 0.25
C      if(n.eq.2) cons=0.25d0*cons
      constd=1.181d0+dlog10(te/edens)
C
      if(ion.eq.1) then
       reduc(1)=0.5d0
       reduc(2)=0.8d0
       reduc(3)=0.8d0
       reduc(4)=1.d0/mpome
       zrat(1)=1.d0
       zrat(2)=1.d0
       zrat(3)=4.d0
       zrat(4)=1.d0
      elseif(ion.eq.2) then
       reduc(1)=0.8d0
       reduc(2)=2.d0
       reduc(3)=2.d0
       reduc(4)=1.d0/mpome
       zrat(1)=0.25d0
       zrat(2)=0.25d0
       zrat(3)=1.d0
       zrat(4)=0.25d0
      endif
       do 100 imu=1,4
       if(imu.eq.1) xfr(imu)=pdens/edens
       if(imu.eq.2) xfr(imu)=he2dens/edens
       if(imu.eq.3) xfr(imu)=he3dens/edens
       if(imu.eq.4) xfr(imu)=1.d0
  100  amu(imu)=mpome*reduc(imu)
      lmax=n-1
       do 99 l=0,lmax
       xl(l)=dble(l)
       w(l)=2.d0*(2.d0*xl(l)+1.d0)
   99  cc(l)=0.0d0
      xn=dble(n)
C
       do 101 imu=1,4
       cons1=cons*xfr(imu)*dsqrt(amu(imu))
       cons2=te/amu(imu)
       cons3=6.d0*zrat(imu)*xn**2
        do 102 l=0,lmax
        d(l)=cons3*(xn**2-xl(l)**2-xl(l)-1.d0)
        roc=dmin1(constd,(11.196d0+dlog10(cons2/a(l+1)**2)))
        cnl=dmax1(0.0d0,cons1*d(l)*(11.538d0+roc+dlog10(cons2/d(l))))
        cc(l)=cc(l)+cnl
  102   continue
  101  continue
       do 103 l=0,lmax-1
       if(l.eq.0) then
        cl(l,l+1)=cc(l)
       else
        cl(l,l+1)=cc(l)-cl(l,l-1)
       endif
       cl(l+1,l)=(w(l)/w(l+1))*cl(l,l+1)
  103  continue       
       do 104 l=lmax,1,-1
       if(l.eq.lmax) then
        ch(l,l-1)=cc(l)
       else
        ch(l,l-1)=cc(l)-ch(l,l+1)
       endif
       ch(l-1,l)=(w(l)/w(l-1))*ch(l,l-1)
  104  continue
       do 105 l=0,lmax-1
       k=(n-1)*n/2+1+l
       c(k,k+1)=0.5d0*(cl(l,l+1)+ch(l,l+1))
       c(k+1,k)=0.5d0*(cl(l+1,l)+ch(l+1,l))
  105  continue
      brockl=0.0d0
C
      RETURN
      END
C*****************************************************************      
C************************************************************           
      SUBROUTINE HE1REC(te,al1he1,atot,arec)
C************************************************************           
C Total recombination from Verner & Ferland ('96). Choice of
C levels, and level numbers from Almog & Netzer ('89).
C************************************************************           
      implicit real*8(a-h,o-z)
C
      parameter (levhe1=16)
C
      dimension arec(levhe1),a1s2(14),a31s2s(14),a31s2p(14),
     &a31s3s(14),a31s3p(14),a31s3d(14),a31s4s(14),a31s4d(14)
C
      data a1s2/8.59d-12,4.83d-12,2.72d-12,1.53d-12,8.60d-13,
     &4.84d-13,2.74d-13,1.56d-13,9.01d-14,5.29d-14,2.99d-14,
     &1.39d-14,4.95d-15,1.42d-15/
      data a31s2s/7.26d-13,4.08d-13,2.30d-13,1.29d-13,7.30d-14,
     &4.17d-14,2.44d-14,1.51d-14,1.00d-14,6.66d-15,3.75d-15,
     &1.50d-15,4.60d-16,1.20d-16/
      data a31s2p/3.70d-12,2.08d-12,1.17d-12,6.58d-13,3.70d-13,
     &2.07d-13,1.13d-13,5.74d-14,2.55d-14,9.49d-15,2.95d-15,
     &8.01d-16,1.97d-16,4.53d-17/
      data a31s3s/1.65d-13,9.29d-14,5.23d-14,2.95d-14,1.68d-14,
     &9.73d-15,5.88d-15,3.78d-15,2.59d-15,1.72d-15,9.73d-16,
     &3.93d-16,1.20d-16,3.13d-17/
      data a31s3p/1.27d-12,7.16d-13,4.03d-13,2.27d-13,1.29d-13,
     &7.35d-14,4.11d-14,2.09d-14,9.21d-15,3.38d-15,1.03d-15,
     &2.75d-16,6.65d-17,1.51d-17/
      data a31s3d/1.26d-12,7.09d-13,3.99d-13,2.25d-13,1.27d-13,
     &7.14d-14,3.72d-14,1.55d-14,4.92d-15,1.23d-15,2.58d-16,
     &4.88d-17,8.87d-18,1.59d-18/
      data a31s4s/6.20d-14,3.49d-14,1.97d-14,1.11d-14,6.41d-15,
     &3.83d-15,2.41d-15,1.58d-15,1.06d-15,6.89d-16,3.76d-16,
     &1.48d-16,4.46d-17,1.15d-17/
      data a31s4d/7.91d-13,4.45d-13,2.51d-13,1.42d-13,8.14d-14,
     &4.77d-14,2.61d-14,1.09d-14,3.38d-15,8.29d-16,1.72d-16,
     &3.26d-17,5.92d-18,1.96d-18/
      tlg=dlog10(te)
      t4=te*1.d-4
      do 7870 iis=1,13
	 tefix=0.5d0*dble(iis)
         if(tlg.ge.tefix.and.tlg.lt.(tefix+0.5d0)) then
	    iit=iis
	    goto 7871
	 endif
 7870 continue
 7871 continue
      if(tlg.lt.0.5d0) iit=1
      if(tlg.lt.0.5d0) tefix=0.5d0
      if(tlg.ge.7.d0) iit=13
      if(tlg.ge.7.d0) tefix=6.5d0
      qq=2.d0*(tlg-tefix)
      al1he1=1.d1**((dlog10(a1s2(iit)))*(1.d0-qq)+
     &		    (dlog10(a1s2(iit+1)))*qq)
      arec(1)=al1he1
      arec(2)=1.d1**((dlog10(a31s2s(iit)))*(1.d0-qq)+
     &              (dlog10(a31s2s(iit+1)))*qq)
      arec(4)=1.d1**((dlog10(a31s2p(iit)))*(1.d0-qq)+
     &              (dlog10(a31s2p(iit+1)))*qq)
      arec(6)=1.d1**((dlog10(a31s3s(iit)))*(1.d0-qq)+
     &              (dlog10(a31s3s(iit+1)))*qq)
      arec(7)=1.d1**((dlog10(a31s3p(iit)))*(1.d0-qq)+
     &              (dlog10(a31s3p(iit+1)))*qq)
      arec(8)=1.d1**((dlog10(a31s3d(iit)))*(1.d0-qq)+
     &              (dlog10(a31s3d(iit+1)))*qq)
      arec(9)=1.d1**((dlog10(a31s4s(iit)))*(1.d0-qq)+
     &              (dlog10(a31s4s(iit+1)))*qq)
      arec(11)=1.d1**((dlog10(a31s4d(iit)))*(1.d0-qq)+
     &              (dlog10(a31s4d(iit+1)))*qq)
C -- Remaining recombination rates for He I. (Almog & Netzer)
      arec(3)=1.92d-14/t4**.821
      arec(5)=4.35d-14/t4**.821
      arec(10)=9.50d-15/t4**.78
      arec(12)=4.16d-15/t4**1.114
      arec(13)=3.49d-14/t4**.915
      arec(14)=2.16d-14/t4**.750
      arec(15)=1.52d-14/t4**1.72
      arec(16)=7.91d-15/t4**1.27
      aknown=arec(1)+arec(2)+arec(4)+arec(6)+arec(7)+arec(8)+
     &arec(9)+arec(11)
      a1=(21.459d0*te)**.5d0
      a2=(2.1381d-7*te)**.5d0
      aa=a1*((1.d0+a1)**.2108d0)*((1.d0+a2)**1.7892d0)
      atot=9.356E-10/aa
      arestt=atot-aknown
      arest1=arec(3)+arec(5)+arec(10)+arec(12)
      arest2=arec(13)+arec(14)+arec(15)+arec(16)
      if(arest1.ge.arestt) then
         correc=arestt/arest1
         arec(3)=arec(3)*correc
         arec(5)=arec(5)*correc
         arec(10)=arec(10)*correc
         arec(12)=arec(12)*correc
         arec(13)=1.d-30
         arec(14)=1.d-30
         arec(15)=1.d-30
         arec(16)=1.d-30
      else
	 correc=(arestt-arest1)/arest2
         arec(13)=arec(13)*correc
         arec(14)=arec(14)*correc
         arec(15)=arec(15)*correc
         arec(16)=arec(16)*correc
      endif
C
      return
      end
C************************************************************           
c*********************************************************************   
      DOUBLE PRECISION FUNCTION ESCAPE(TAU)                             
C                                                                       
C     (Valid for Doppler profiles)                                      
C     Exact evaluation for TAU<1.E12                                    
C     Asymptotic expression for TAU>1.E12                               
C************************************************************           
C                                                                       
      IMPLICIT REAL*8(A-H,O-Z)                                          
C                                                                       
      COMMON/CONSTS/PI,PISQRT,CLIGHT                                    
C                                                                       
      DIMENSION X(11),F(11),G(11),T(11),XA(8),XB(8),TT(8)               
C                                                                       
      DATA XA/0.0D0,0.2D0,0.9D0,1.3D0,2.0D0,2.9D0,3.9D0,4.9D0/       
      DATA XB/3.3D0,3.4D0,3.5D0,3.6D0,3.9D0,4.5D0,5.0D0,6.0D0/          
      DATA TT/1.D0,1.3D0,1.6D0,2.D0,3.D0,5.D0,8.D0,12.D0/               
C                                                                       
      SUM=0.0D0                                                         
      if(tau.le.1.d-5) pesc=1.d0
      if(tau.le.1.d-5) goto 9999
      TTLOG=DLOG10(TAU)                                                 
      IF(TTLOG.LT.12.D0) GOTO 1000                                      
      PESC=1.D0/(2.D0*TAU*DSQRT(DLOG(TAU/PISQRT)))                      
      GOTO 9999                                                         
C                                                                       
 1000 CONTINUE                                                          
      IXLOOP=11                                                         
      ITLOOP=8                                                          
C                                                                       
      TTLOG=DLOG10(TAU)                                                 
      IF(TTLOG.GE.TT(1)) GOTO 40                                        
      XMIN=XA(1)                                                        
      XMAX=XB(1)                                                        
      GOTO 99                                                           
   40 CONTINUE                                                          
      IF(TTLOG.GE.TT(ITLOOP)) ISAVE=ITLOOP-1                            
      IF(TTLOG.GE.TT(ITLOOP)) GOTO 88                                   
      DO 50 I=1,ITLOOP-1                                                
      IF(TTLOG.LT.TT(I+1).AND.TTLOG.GE.TT(I)) GOTO 51                   
      GOTO 50                                                           
   51 ISAVE=I                                                           
      GOTO 88                                                           
   50 CONTINUE                                                          
   88 CC=(TTLOG-TT(ISAVE))/(TT(ISAVE+1)-TT(ISAVE))                      
      XMIN=XA(ISAVE)+(XA(ISAVE+1)-XA(ISAVE))*CC                         
      XMAX=XB(ISAVE)+(XB(ISAVE+1)-XB(ISAVE))*CC                         
   99 DX=(XMAX-XMIN)/DBLE(IXLOOP-1)                                     
C                                                                       
      DO 100 I=1,IXLOOP                                                 
      X(I)=XMIN+DX*DBLE(I-1)                                            
  100 CONTINUE                                                          
C                                                                       
      DO 200 I=1,IXLOOP                                                 
      F(I)=FIFUNC(X(I))                                                 
      G(I)=TAU*F(I)                                                     
  200 T(I)=F(I)*EXPINT(2,G(I))                                          
C                                                                       
      ISUM=(IXLOOP-1)/2                                                 
C                                                                       
      DO 350 I=1,ISUM                                                   
      SUM=SUM+(DX/3.D0)*(T(2*I-1)+T(2*I+1)+4.D0*T(2*I))                 
  350 CONTINUE                                                          
      PESC=2.D0*SUM                                                     
C                                                                       
 9999 ESCAPE=PESC                                                       
C                                                                       
      RETURN                                                            
      END                                                               
C                                                                       
C************************************************************           
C************************************************************           
C                                                                       
      DOUBLE PRECISION FUNCTION FIFUNC(X)                               
C                                                                       
      IMPLICIT REAL*8(A-H,O-Z)                                          
C                                                                       
      COMMON/CONSTS/PI,PISQRT,CLIGHT                                    
C                                                                       
      FIFUNC=EXPFN(X**2)/PISQRT                                       
C                                                                       
      RETURN                                                            
      END                                                               
C                                                                       
C************************************************************           
C************************************************************           
C                                                                       
      DOUBLE PRECISION FUNCTION EXPINT(N,X)                             
C                                                                       
C  COMPUTES THE N'TH EXPONENTIAL INTEGRAL OF X                          
C  INPUT - X,  INDEPENDENT VARIABLE (-100. .LE. X .LE. +100.)           
C         N,  ORDER OF DESIRED EXPONENTIAL INTEGRAL (1 .LE. N .LE. 8)   
C  OUTPUT - EXPINT,  THE DESIRED RESULT                                 
C           EX,  EXPF(-X)                                               
C  NOTE   RETURNS WITH E1(0)=0, (NOT INFINITY).                         
C  RUDOLF LOESER, 3 MAR 66                                              
C-----GENERAL COMPILATION OF 1 AUGUST 1967.                             
C                                                                       
      IMPLICIT REAL*8(A-H,O-Z)                                          
      DIMENSION TAB(20),XINT(7)                                         
      DATA XINT/1.,2.,3.,4.,5.,6.,7./                                   
      DATA TAB /.2707662555,.2131473101,.1746297218,.1477309984,        
     1.1280843565,.1131470205,.1014028126,.0919145454,.0840790292,      
     1.0774922515,.0718735405,.0670215610,.0627878642,.0590604044,      
     1.0557529077,.0527977953,.0501413386,.0477402600,.0455592945,      
     1.0435694088/                                                      
C                                                                       
      XSAVE=0.0D0                                                       
      U=X                                                               
      IF(U)603,602,603                                                  
  602 EX=1.                                                             
      IF(N-1)800,800,801                                                
  800 EXPINT=0.                                                         
      GOTO 777                                                          
  801 EXPINT=1./XINT(N-1)                                               
      GOTO 777                                                          
  603 IF(U-XSAVE)604,503,604                                            
  604 XSAVE=U                                                           
      XM=-U                                                             
      EMX=DEXP(XM)                                                      
C                                                                       
C  SELECT METHOD FOR COMPUTING EI(XM)                                   
C  COPIED FROM THE SHARE ROUTINE NUEXPI, WRITTEN BY J. W. COOLEY,       
C  COURANT INSTITUTE OF MATHEMATICAL SCIENCES, NEW YORK UNIVERSITY      
C                                                                       
      IF(XM-24.5)501,400,400                                            
  501 IF(XM-5.)502,300,300                                              
  502 IF(XM+1.)100,200,200                                              
  503 EISAVE=-ARG                                                       
      EXSAVE=EMX                                                        
C                                                                       
C  NOW RECURSE TO HIGHER ORDERS                                         
C                                                                       
      IF(N-1)507,507,505                                                
  505 DO 506 I=2,N                                                      
        EISAVE=(U*EISAVE-EXSAVE)/(-XINT(I-1))                           
  506 CONTINUE                                                          
  507 EXPINT=EISAVE                                                     
      EX=EXSAVE                                                         
  777 RETURN                                                            
C                                                                       
C  EI(XM) FOR XM .LT. -1.0                                              
C  HASTINGS POLYNOMIAL APPROXIMATION                                    
C                                                                       
  100 ARG=((((((U+8.573328740 )*U+18.05901697  )*U+8.634760893 )*U      
     *+.2677737343)/XM)*EMX)/((((U+9.573322345 )*U+25.63295615  )*U     
     *+21.09965308  )*U+3.958496923 )                                   
      GOTO 503                                                          
C     EI(XM) FOR -1. .LE. XM .LT. 5.0                                   
C     POWER SERIES EXPANSION ABOUT ZERO                                 
  200 ARG=DLOG(DABS(XM))                                                
      ARG=((((((((((((((((.41159050D-14*XM+.71745406D-13)*XM+.76404637D-
     *12)*XM+.11395905D-10)*XM+.17540077D-9)*XM+.23002666D-8)*XM+.275360
     *18D-7)*XM+.30588626D-6)*XM+.31003842D-5)*XM+.28346991D-4)*XM+.2314
     *8057D-3)*XM+.0016666574)*XM+.010416668)*XM+.055555572)*XM+.25)*XM+
     *.99999999)*XM+.57721566)+ARG                                      
      GOTO 503                                                          
C                                                                       
C  EI(XM) FOR 5.0 .LE. XM .LT. 24.5                                     
C  TABLE LOOK-UP AND INTERPOLATION                                      
C                                                                       
  300 I=XM+.5                                                           
      XZERO=I                                                           
      DELTA=XZERO-XM                                                    
      ARG=TAB(I-4)                                                      
      IF(DELTA)303,305,303                                              
  303 Y=ARG                                                             
      DELTAX=DELTA/XZERO                                                
      POWER=1./DELTAX                                                   
      DO 304 I=1,7                                                      
        POWER=POWER*DELTAX                                              
        Y=((Y-POWER/XZERO)*DELTA)/XINT(I)                               
        ARG=ARG+Y                                                       
        IF(DABS(Y/ARG)-1.D-8)305,304,304                                
  304 CONTINUE                                                          
  305 ARG=EMX*ARG                                                       
      GOTO 503                                                          
C     EI(XM) FOR 24.5 .LE. XM                                           
C     TRUNCATED CONTINUED FRACTION                                      
  400 ARG=((((XM-15.)*XM+58.)*XM-50.)*EMX)/((((XM-16.)*XM+72.)*XM-96.)  
     **XM+24.)                                                          
      GOTO 503                                                          
      END                                                               
C                                                                       
C************************************************************           
C************************************************************
      SUBROUTINE ESCACF(ENE,A21,ADAMP,G1,G2,TE,AMASS,TCORR,
     &COLTOTI,COLTOT,BE)
C********************************************************
C     Evaluate the escape probability (BE) for photons esca-
C     ping inwards and outwards. Take a mean value of these.
C     ENE in eV, AMASS in atomic mass units, TCORR correction
C     in opacity due to stimulated emission.
C********************************************************
      IMPLICIT REAL*8(A-H,O-Z)
      parameter (md=75)
C
      if(tcorr.le.0.0d0) tcorr=0.0d0
      vterm=1.285d4*dsqrt(te/amass)
      wl=1.2398d4/ene
      dnud=1.d8*vterm/wl
      f12=1.499d-16*g2*a21*wl**2./g1
      cap=1.4974d-2*f12*tcorr/dnud
      t0=cap*coltoti
      t0t=cap*(coltot-coltoti)
      if(t0.le.0.0d0) then
       be1=1.d0
      else
       CALL ESCAP(T0,WL,ADAMP,VTERM,BE1)
      endif
      if(t0t.le.0.0d0) then
       be2=1.d0
      else
       CALL ESCAP(T0T,WL,ADAMP,VTERM,BE2)
      endif
      BE=0.5d0*(BE1+BE2)
C
      RETURN
      END
C************************************************************
C************************************************************
      SUBROUTINE ESCAP(T0,WL,ADAMP,VTERM,BE)
C********************************************************
C     Escape probability (from Ferland, coded by C.F.)
C     (Does not apply to lines trapped by the continuum.)
C********************************************************
      IMPLICIT REAL*8(A-H,O-Z)
C
      IF(VTERM.LE.0.) VTERM=1.285E6        
      A=7.958E-10*WL*ADAMP/VTERM
      AT=A*T0
      IF(AT.GT.1.) THEN
        B=1.6 + 3./((2.*A)**.12*(1.+AT))
      ELSEIF(AT.GT.0.) THEN
        B=1.6 + 3./((2.*A)**.12*(1.+1./SQRT(AT)))
      ELSE
        B=1.6
      ENDIF
      IF(B.GT.5) THEN
        B=5.
      ENDIF
      BE = 1./(1. + B * T0)
C
      RETURN
      END
C************************************************************           
c*********************************************************************   
      SUBROUTINE ESC(taul,tauc,cl,pesc,plin,pcont,dtau)
c*********************************************************************   
c     Calculates probabilities that a line photon escapes (pesc),
c     is self-absorbed (plin), or is trapped in the continuum (pcont).
c     For plin/pcont, it is assumed that tauc/taul=cl, i.e.
c     local conditions govern plin/pcont. This routine is good
c     for lines with Voigt parameter  a < 1E-3. Both absorption
c     and emission profiles are assumed to be Voigt profiles.
c     The routine also works for zero continuum absorption.
c*********************************************************************   
      implicit real*8(a-h,o-z)
c
      dimension acl(9),fcl(9),tl(13),gtl(13)
c
      data acl/-10.,-7.,-5.,-4.,-3.,-2.,-1.,0.,1./
      data fcl/2.064,1.310,.8797,.8122,.7372,.6385,.5172,.4265,.3997/
      data tl/-1.,-0.5,0.,0.5,1.,1.5,2.,3.,4.,5.,7.,9.,11./
      data gtl/-.01768,-.05425,-.1677,-.4882,-1.1204,-1.7481,-2.3201,
     &         -3.4121,-4.4541,-5.3275,-6.0770,-7.9987,-93.876/
c
      if(cl.lt.1.d-10) then
       q=1.159d2*cl
      elseif(cl.ge.1.d-10.and.cl.lt.1.d1) then
       cllog=dlog10(cl)
        do 100 icl=1,8
        if(cllog.ge.acl(icl).and.cllog.lt.acl(icl+1)) then
         y=(cllog-acl(icl))/(acl(icl+1)-acl(icl))
         fqlog=(1.d0-y)*fcl(icl)+y*fcl(icl+1)
         q=cl*1.d1**fqlog
        endif
  100   continue
      elseif(cl.ge.1.d1) then
       q=2.51d0*cl
      endif
c
      if(taul.lt.0.1d0) then
       g=1.d0-0.39894d0*taul
      elseif(taul.ge.0.1d0.and.taul.lt.1.d11) then
       tlog=dlog10(taul)
        do 101 itl=1,12
        if(tlog.ge.tl(itl).and.tlog.lt.tl(itl+1)) then
         x=(tlog-tl(itl))/(tl(itl+1)-tl(itl))
         fglog=(1.d0-x)*gtl(itl)+x*gtl(itl+1)
         g=1.d1**fglog
        endif
  101   continue
      elseif(taul.ge.1.d11) then
       g=1.33d-94
      endif
c
      if(dtau.lt.1.d-2) then
       pesc=g*expfn(tauc)
      else
       pesc=(g*(1.d0-expfn(dtau))*expfn(tauc))/dtau
      endif
      ptrap=1.d0-pesc
      plin=ptrap/(1.d0+q)
      pcont=(q*ptrap)/(1.d0+q)
c
      return
      end
c*********************************************************************   
C*********************************************************************
      double precision function expfn(x)
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      if(x.gt.2.d2) then
       expfn=1.384d-87
      elseif(dabs(x).lt.1.d-15) then
       expfn=1.d0
      else
       expfn=dexp(-x)
      endif
C
      return
      end
C*********************************************************************
C*********************************************************************
      double precision function pexpfn(x)
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      if(x.gt.1.d-4) then
        pexpfn=(1.d0-expfn(x))/x
      else
        pexpfn=1.d0-x*0.5d0
      endif
C
      return
      end
C*********************************************************************
C*********************************************************************
      DOUBLE PRECISION FUNCTION RAD(TE,IMAX,CORAT1,CORAT2,ITE,          
     &DELTT,teinn,corat3,corat4,corat5,corat6,corat7,corat8,fudfed)
C************************************************************           
C                                                                       
C  THIS FUNCTION CALCULATES ALL RADIATIVE AND COLLISIONAL RATES         
C
C************************************************************
C************************************************************        
C     AB(1) = ABUNDANCE OF HYDROGEN
C     AB(2) =    D:O    OF HELIUM 
C     AB(3) =    D:O    OF OXYGEN
C     AB(4) =    D:O    OF CARBON 
C     AB(5) =    D:O    OF NITROGEN 
C     AB(6) =    D:O    OF SULPHUR
C     AB(7) =    D:O    OF NEON  
C     AB(8) =    D:O    OF IRON 
C     AB(9) =    D:O    OF SILICON
C     AB(10)=    D:O    OF SODIUM 
C     AB(11)=    D:O    OF MAGNESIUM
C     AB(12)=    D:O    OF ALUMINIUM
C     AB(13)=    D:O    OF ARGON
C     AB(14)=    D:O    OF CALCIUM
C                                                                       
C     ENUMERATION OF THE IONS:                                          
C                                                                       
C XA and XB
C      1 = H I      2 = HE I     3 = HE II    4 = O VI     5 = O VII    
C      6 = O VIII   7 = O V      8 = C III    9 = C IV    10 = C V      
C     11 = C VI    12 = C I     13 = C II    14 = O I     15 = O II     
C     16 = O III   17 = O IV    18 = N I     19 = N II    20 = N III    
C     21 = N IV    22 = N V     23 = N VI    24 = N VII   25 = S I     
C     26 = S II    27 = S III   28 = S IV    29 = S V     30 = S VI
C     31 = S VII   32 = S VIII  33 = S IX    34 = S X     35 = NE I    
C     36 = NE II   37 = NE III  38 = NE IV   39 = NE V    40 = NE VI
C     41 = NE VII  42 = NE VIII 43 = FE I    44 = FE II   45 = FE III
C     46 = FE IV   47 = FE V    48 = FE VI   49 = FE VII  50 = FE VIII
C     51 = FE IX   52 = FE X    53 = FE XI   54 = FE XII  55 = FE XIII
C     56 = FE XIV  57 = FE XV   58 = SI I    59 = SI II   60 = SI III
C     61 = SI IV   62 = SI V    63 = SI VI   64 = SI VII  65 = SI VIII
C     66 = SI IX   67 = SI X    68 = NA I    69 = NA II   70 = NA III
C     71 = NA IV   72 = NA V    73 = NA VI   74 = NA VII  75 = NA VIII
C     76 = NA IX   77 = MG I    78 = MG II   79 = MG III  80 = MG IV
C     81 = MG V    82 = MG VI   83 = MG VII  84 = MG VIII 85 = MG IX
C     86 = MG X    87 = AL I    88 = AL II   89 = AL III  90 = AL IV
C     91 = AL V    92 = AL VI   93 = AL VII  94 = AL VIII 95 = AL IX
C     96 = AL X    97 = AL XI   98 = AR I    99 = AR II   100 = AR III
C     101 = AR IV  102 = AR V   103 = AR VI  104 = AR VII  105 = AR VIII
C     106 = AR IX  107 = AR X   108 = CA I   109 = CA II   110 = CA III
C     111 = CA IV  112 = CA V   113 = CA VI  114 = CA VII  115 = CA VIII
C     116 = CA IX  117 = CA X
C QXB
C      1 = H I      2 = H II     3 = HE I     4 = HE II    5 = HE III
C      6 = C I      7 = C II     8 = C III    9 = C IV    10 = C V      
C     11 = C VI    12 = C VII   13 = O I     14 = O II    15 = O III
C     16 = O IV    17 = O V     18 = O VI    19 = O VII   20 = O VIII  
C     21 = O IX    22 = S I     23 = S II    24 = S III   25 = S IV
C     26 = S V     27 = S VI    28 = S VII   29 = S VIII  30 = S IX
C     31 = S X     32 = S XI    33 = N I     34 = N II    35 = N III
C     36 = N IV    37 = N V     38 = N VI    39 = N VII   40 = N VIII
C     41 = NE I    42 = NE II   43 = NE III  44 = NE IV   45 = NE V
C     46 = NE VI   47 = NE VII  48 = NE VIII 49 = NE IX   50 = FE I
C     51 = FE II   52 = FE III  53 = FE IV   54 = FE V    55 = FE VI
C     56 = FE VII  57 = FE VIII 58 = FE IX   59 = FE X    60 = FE XI
C     61 = FE XII  62 = FE XIII 63 = FE XIV  64 = FE XV   65 = FE XVI
C     66 = SI I    67 = SI II   68 = SI III  69 = SI IV   70 = SI V    
C     71 = SI VI   72 = SI VII  73 = SI VIII 74 = SI IX   75 = SI X
C     76 = SI XI   77 = NA I    78 = NA II   79 = NA III  80 = NA IV
C     81 = NA V    82 = NA VI   83 = NA VII  84 = NA VIII 85 = NA IX
C     86 = NA X    87 = MG I    88 = MG II   89 = MG III  90 = MG IV
C     91 = MG V    92 = MG VI   93 = MG VII  94 = MG VIII 95 = MG IX
C     96 = MG X    97 = MG XI   98 = AL I    99 = AL II   100 = AL III
C   101 = AL IV   102 = AL V    103 = AL VI   104 = AL VII   105 = AL VIII
C   106 = AL IX   107 = AL X    108 = AL XI   109 = AL XII   110 = AR I
C   111 = AR II   112 = AR III  113 = AR IV   114 = AR V     115 = AR VI
C   116 = AR VII  117 = AR VIII 118 = AR IX   119 = AR X     120 = AR XI
C   121 = CA I    122 = CA II   123 = CA III  124 = CA IV    125 = CA V
C   126 = CA VI   127 = CA VII  128 = CA VIII 129 = CA IX    130 = CA X
C   131 = CA XI
C
C************************************************************           
C                                                                       
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      parameter (md=75,levhmx=55,levh=levhmx+1,levhe=72,levhe1=16,
     &mmatr=80,nnhmax=10)
C
      common/hydaij/ahydij(levhmx,levhmx),ahydsm(levhmx)
      common/helaij/ahelij(levhmx,levhmx),ahelsm(levhmx)
      common/hrecij/rechij(25,levhmx),terech(25),rechsum(25)
      common/herecij/recheij(25,levhmx),tereche(25),rechesum(25)
      common/hhelos/edens,pdens,he2dens,he3dens,ta12i,ta12o,f21,
     &f21two,f31,f32
      common/hlos1/f41,f42,f43,clymh,cbalh,cpaschh,cresth
      common/helos1/tao1i,tao1o,deno3,clymhe,cbalhe,cresthe,
     &he1cool,tahe1i,tahe2i,tahe3i,tahe4i,tahe1o,tahe2o,tahe3o,
     &tahe4o,tc12i,tc12o,tc13i,tc13o,tc14i,tc14o,tche1i,tche1o
      COMMON/RLBESC/BESC
      common/heiila/heii21
      common/folines/redc(10),redn(10),redo(10),rede(10),reds(10),
     &forc(10),forn(10),foro(10),fore(11),fors(10)
      common/folines2/redsi(10),UVSI(md,4),forsi(10)
      common/folines3/redna(2),redmg(4),redar(4),redca(4),
     &forna(6),formg(11),forar(10),forca(9),foralu(2),fecol10,
     &forbfe1
      COMMON/FEIIC/OMFE2(16,16),AIJFE2(16,16),DLFE2(16,16) 
      common/FEIIC2/admpfe2(16,16),asmfe2(16)
      common/columns/columi(117),colum(117)
      common/column/colni(md,117),colno(md,117)
      common/hyddt/qxh(md,2,levh),hradm(levh,levh),hccoll(levh,levh)
      common/hedt/qxhe(md,2,levhe),heradm(levhe,levhe),
     &hecoll(levhe,levhe)
      common/arglin/farg(md,6)
      COMMON/FBFE7/FEADD(17)
      COMMON/BWLOSS/FOIII(md,8)                                         
      COMMON/MSTUFF/XAXA(mmatr),AM(mmatr,mmatr),EPS,NERR
      COMMON/TAUXX/TAXA(md,-13:75)                                      
      COMMON/ABSCNO/ABSEUV(md,99)                                       
      COMMON/CONSTS/PI,PISQRT,CLIGHT                                    
      COMMON/COLCNO/COLLC,COLLN,COLLO,COOLFR(175)                       
      COMMON/HEOPT/OPTHEL(10,md)                                        
      COMMON/CHION/CIONH,HYCO,HEI,HEII,HEII32,HEIIBW                    
      common/newhy/hy2,hy3,hyco2,hyco3,hytot
      COMMON/ABSH/ABS12(md),ABS13(md),ABSHE2(md),ABSHE3(md),ABSO1(md)   
     &,ABSHE4(MD),ABSHE5(MD),ABSHE6(MD),ABSHE7(MD),ABSHE8(MD)
      COMMON/ABSHN/ABSNV(md)
      COMMON/UVSUVS/UVHELI(md,1),UVCARB(md,6),UVOXYG(md,5),UVNITR(md,6) 
      COMMON/PHOTOS/TEBB,RPA15                                          
      COMMON/SPOT/OTSP(7)                                               
       COMMON/NLEV/NION,NHY,E00                                         
      COMMON/EDAYY/TDAY                                                 
      COMMON/SIMPL/ISIMP,IGAMM,GAHE,CR                                  
      COMMON/COL/RE(4),FF,HY,HE,C131,C142,C151,C161,C231,C241,COH,      
     &COHE,C351,C321,C331,C341,C222,C232,C221,C332,C441                 
      COMMON/EQUIV/W(30),CIN(30),FB(md,15),TWOP(md)                     
      COMMON/NUTSK/YI(7),YEX1(7)
      COMMON/HEA/CO,PH,PHEO,PHEI,PO(8),PC(6),PN(7),PMG,PSIL,PFE          
      COMMON/HEB/PHET,PCT,PNT,POXT,PMGT,PSIT,PFET,HEATT                 
      COMMON/HESTUFF/ALDHE,CHEO,CHE                                     
      COMMON/TEQQ/TEZ
      COMMON/REHEL/REHE21,REHE22,REC31,REN41                            
      COMMON/PHY/DEN(md)
      COMMON/IND/IK
      COMMON/PHQ/ZE(117),GE(117),ZK(117),GET(117)                        
      COMMON/REC/AL2(7)                                                 
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)       
      COMMON/ABC/AL(7),ALN(8),ALO(9),ALC(7),ALS(15),ALMG(11),
     &ALAL(12),COO(8),COC(6),CON(7)                                             
      common/abc2/alneo(9),alsul(11),cone(8),cosu(10),zneo(8),
     &zsul(10)
      common/abc3/alfe(16),cofe(15),zfe(15),alsi(11),cosi(10),
     &zsi(10)
      common/abc4/alna(10),alar(11),alca(11),cona(9),zna(9),
     &comg(10),zmg(10),coal(11),zal(11),coar(10),zar(10),coca(10),
     &zca(10)
      COMMON/ABD/ALF(7)                                                 
      COMMON/ELEC/DEL(md),MEL                                           
      COMMON/ABUN/AB(15)                                                
      COMMON/ABU/XA(2,117)                                               
      COMMON/TRES/EL(117),EK(117)
      common/abun2/abunda(117)
      common/absimp/xc(7),xn(8),xo(9),xne(9),xs(11),xfe(16),
     &xsi(11),xna(10),xmg(11),xal(12),xar(11),xca(11)
      COMMON/NHY/NH                                                     
      COMMON/PLUMP/XMIN,STE1,TOL1,TOL2,TOL13,KWRI                       
      COMMON/ITRIM/ITEX                                                 
      COMMON/HPOP/XN1,XN2,XN3                                           
      COMMON/BOLD/TOLD                                                  
      COMMON/HRAT/XEL,HYR,HEAT,COOL,RADR                                
      COMMON/REC1/RE1(7),RHE2B                                          
      COMMON/DIF/TAU(md,-13:75)                           
      COMMON/ZAETA/ZA(8),ZB(6),QXA1,ZC(7)                               
      COMMON/COOLVA/CEX(15),C311,C322,C111,C121,COCARB,                   
     &CONI,COOX
      COMMON/BOWHEL/BWHELI(md,2)                                        
      COMMON/YIELDA/YLA(md),YB(md),YO2(md),YA(md),IYIELD,KYIELD         
      COMMON/ION/XBB(md,117)
      common/yieldc/ela(md),plinla(md),pconla(md),poo(md)
      common/recomb/al1h,al1he1,al1he2
      common/radabs/rfix(md),rbound(md),dxabs(3,md,md)
      common/euvlin/w1eu(99),eeveu(99),alqeu(99),beqeu(99),tmeu(99),
     &ioeu(99),iityp(99),nline,jqeu(99),jweu(99)
      common/euvli2/rcont(79),leuv(79),iskipeu(79)
      common/fe4/faij(22,22),fome(22,22)
      common/twexps/twoexp(-13:75)
      COMMON/BOWENS/BORATE(md,117)
      common/robcum/fro21,fro41,frs21,frs41,frr21,frr41
      COMMON/NVLEV/x1n5(md),p21in(md),p21out(md)
      common/recnah/tenah(81),alnah(29,81)
C
      dimension chtrac(7),chtran(8),chtrao(9)                           
      dimension psu(10),pne(8),psi(10),pona(9),pomg(10),poal(11),
     &poar(10),poca(10)
      DIMENSION PF(15),GEA(117),FI1(7)
      DIMENSION TAEUVI(99),TAEUVO(99)                                   
      DIMENSION RLOFAC(99),RCOFAC(99),RLOESC(99)
      dimension tceuvi(99),tceuvo(99),cl(99),dtauc(99)
C     **************************************************************    
C 
      xh=ab(1)*xa(2,1)
      PI=3.141592D0                                                     
      PISQRT=1.77245385D0                                               
      CLIGHT=2.998D10                                                   
      CONS1=1.D0                                                        
      TELOG=DLOG10(TE)                                                  
      T4=TE/1.D4                                                        
      TEZ=TE                                                            
      TRYD=TE/1.578D5
      TEV=TE/1.1609D4                                                   
      sqrtte=dsqrt(te)
      E00=13.292
      HEI=0.d0
C
      DO 7622 KK=1,175                                               
 7622 COOLFR(KK)=0.
C                                                                       
      DO 824 I=1,16
      alfe(i)=0.0d0
      if(i.gt.15) goto 824
      cofe(i)=0.0d0
      zfe(i)=0.0d0
      if(i.gt.12) goto 824
      alal(i)=0.0d0
      if(i.gt.11) goto 824
      alsul(i)=0.0d0
      alsi(i)=0.0d0
      almg(i)=0.0d0
      alar(i)=0.0d0
      alca(i)=0.0d0
      coal(i)=0.0d0
      zal(i)=0.0d0
      fore(i)=0.0d0
      formg(i)=0.0d0
      if(i.gt.10) goto 824
      forar(i)=0.0d0
      forsi(i)=0.0d0
      alna(i)=0.0d0
      cosu(i)=0.0d0
      zsul(i)=0.0d0
      cosi(i)=0.0d0
      zsi(i)=0.0d0
      comg(i)=0.0d0
      zmg(i)=0.0d0
      coar(i)=0.0d0
      zar(i)=0.0d0
      coca(i)=0.0d0
      zca(i)=0.0d0
      if(i.gt.9) goto 824
      alo(i)=0.0d0
      alneo(i)=0.0d0
      cona(i)=0.0d0
      zna(i)=0.0d0
      foro(i)=0.0d0
      forca(i)=0.0d0
      if(i.gt.8) goto 824
      za(i)=0.0d0
      coo(i)=0.0d0
      zneo(i)=0.0d0
      cone(i)=0.0d0
      aln(i)=0.0d0
      if(i.gt.7) goto 824
      redo(i)=0.0d0
      reds(i)=0.0d0
      rede(i)=0.0d0
      al(i)=0.0d0
      alc(i)=0.0d0
      con(i)=0.0d0
      zc(i)=0.0d0
      al2(i)=0.0d0
      yex1(i)=0.0d0
      fi1(i)=0.0d0
      re1(i)=0.0d0
      if(i.gt.6) goto 824
      forna(i)=0.0d0
      forn(i)=0.0d0
      coc(i)=0.0d0
      zb(i)=0.0d0
      if(i.gt.5) goto 824
      redc(i)=0.0d0
      redn(i)=0.0d0
      if(i.gt.4) goto 824
      forc(i)=0.0d0
      redar(i)=0.0d0
      redca(i)=0.0d0
      redmg(i)=0.0d0
      re(i)=0.0d0
      if(i.gt.3) goto 824
      redsi(i)=0.0d0
      forsi(i)=0.0d0
      if(i.gt.2) goto 824
      redna(i)=0.0d0
      foralu(i)=0.0d0
  824 CONTINUE
c
      call trlate(1,ik)
C                                                                       
C     FIRST CALCULATE THE IONIZATION AND RECOMBINATION RATES            
C                                                                       
C     ***************************************************************   
C     *****                                                             
C     RECOMBINATION COEFF. FOR HYDROGEN LIKE IONS AND HELIUM            
C     SEATON (1959) AND GOULD AND TAKHUR (1966)                         
C     RATES ARE ADJUSTED TO AGREE WITH OSTERBROCK (74) FOR T=1E4        
C     (On-the-spot if OTSP.LE.1.)                                       
C     *****                                                             
C     ***************************************************************   
  915 continue                                                          
      YI(1)=1.5789D5/TE                                                 
      YI(2)=YI(1)                                                       
      YI(3)=4.*YI(1)                                                    
      YI(4)=64.*YI(1)                                                   
      YI(5)=36.*YI(1)                                                   
      YI(6)=49.*YI(1)                                                   
      YI(7)=196.*YI(1)                                                  
C                                                                       
C FI is the asymptotic TOTAL recombination term, accurate               
C to better than 3 % for T*Z**2 < 1E6.                                  
C  FI2 is the term for recombination to n=1 only.                       
C                                                                       
      DO 922 K1=1,6                                                     
      FI2=0.0d0
      FI3=0.0d0
      FI4=0.0d0
      Y=YI(K1)                                                          
      FI=(0.4288+0.5*DLOG(Y)+0.469/(Y**.33333))*0.99D0                  
C     DEXP. INTEGRAL APPROX. FOR RECOMB. DIRECTLY TO N=1                
C  Y < 1                                                                
      if(y.gt.1.) goto 926                                              
      FI1(K1)=-.5772-DLOG(Y)+Y-Y**2/4.+Y**3/18.-Y**4/96.+               
     &Y**5/600.                                                         
      FI2=Y*DEXP(Y)*FI1(K1)                                             
      goto 920                                                          
C  1 < Y < 20                                                           
  926 FI2=(Y**2.+2.3347*Y+.2506)/(Y**2.+3.3306*Y+1.6815)                
      IF(Y.GT.20.) GOTO 3114                                            
      FI3=-.17501+.54157/Y-2.4153/(Y*Y)+11.428/Y**3.-36.749/Y**4.       
     &+70.31/Y**5.-75.169/Y**6.+40.428/Y**7.-8.2368/Y**8.               
      GOTO 3112                                                         
C  20 < Y                                                               
 3114 FI3=-0.1728*(1.-8./(3.*Y)+70./(9.*Y*Y)-800./(27.*Y**3.)           
     &+11440./(81.*Y**4.))                                              
      FI4=-0.0496*(1.-1./Y)                                             
      FI3=FI3+FI4                                                       
 3112 CONTINUE                                                          
      YEX1(K1)=FI2                                                      
      FI2=FI2+FI3                                                       
      IF(Y.GT.100.) GOTO 907                                            
      FI1(K1)=FI2                                                       
      GOTO 910                                                          
  907 FI1(K1)=0.0                                                       
  910 CONTINUE                                                          
  920 IF(K1.EQ.2) GOTO 921                                              
      AL2(K1)=1.38D-16*DSQRT(TE)*Y*(FI-FI2)                             
      IF(K1.EQ.1) AL2(1)=AL2(1)*0.994D0*(TE/1.D4)**(0.03354)            
c     IF(OTSP(K1).LE.1.) GOTO 815                                       
      FI2=0.0D0                                                         
      AL(K1)=1.34D-16*DSQRT(TE)*Y*(FI-FI2)                              
      GOTO 8915                                                         
  815 AL(K1)=AL2(K1)                                                    
 8915 CONTINUE                                                          
      GOTO 922                                                          
C     HELIUM I                                                          
  921 AL2(2)=1.38D-16*DSQRT(TE)*Y*(FI-FI2)*(TE/1.D4)**0.1375            
      AL(2)=AL2(2)                                                      
c     IF(OTSP(2).LE.1.) GOTO 923                                        
      AL(2)=AL2(2)+1.33D-16*Y*FI2*DSQRT(TE)                             
  923 CONTINUE                                                          
C                                                                       
C     DIELECTRONIC RECOMBINATION (ALDROV. AND PEQU.)                    
C                                                                       
      ALDHE=0.                                                          
      IF(TE.LT.5000.) GOTO 945                                          
      ALA=1.+0.3*EXPFN(9.4D4/TE)                                        
      ALDHE=1.9D-3*EXPFN(4.7D5/TE)*ALA/(TE*DSQRT(TE))                   
      GOTO 946                                                          
  945 ALDHE=0.0                                                         
  946 CONTINUE                                                          
  922 CONTINUE                                                          
C      al1h=al(1)-al2(1)
C      al1he1=al(2)-al2(2)
C      al1he2=al(3)-al2(3)
C     ***************************************************************   
C     *****                                                             
C     COLL. IONIZATION RATES FOR HELIUM (LOTZ)                          
C     *****                                                             
C     ***************************************************************   
      IF(YI(2).GT.100.) GOTO 1379                                       
      CHEO=2.28D-11*DSQRT(TE)*EXPFN(1.8*YI(2))                          
      GOTO 1378                                                         
 1379 CHEO=0.0D0                                                        
 1378 CONTINUE                                                          
      IF(YI(3).GT.100.) GOTO 1380                                       
      CHE=4.39D-12*DSQRT(TE)*EXPFN(YI(3))                               
C -- Collisional rates revised 22/6-88 to agree better with Lotz.       
C    (at 10 eV)                                                         
      CHE=1.272D0*CHE                                                   
      GOTO 1385                                                         
 1380 CHE=0.0                                                           
 1385 CONTINUE                                                          
C     IF(ICOM.EQ.1) GOTO 1279                                           
C *********************************************************************   
C   RATES FOR METALS.
C                                                                       
C   For the dielectronic recombination rates, use Shull & Steenberg's   
C   fits (Ap.J. Suppl. 48, 95, 1982) to the results of Jacobs et al..
C   (Correct rates listed in Landini & Monsignori Fossi, including
C    some further fits made isoelectronically.) High density correc-
C   tions to Bely-Dubau et al. (1984) for H-like ions are;
C   Carbon = 1., Nitrogen = 1., Oxygen = 0.81, Neon = 0.7, 
C   Sodium = 1(?), Magnesium = 0.56, Aluminum = 1(?), Silicon = 0.6,
C   Sulphur = 0.6, Argon = 0.6, Calcium = 0.64, Iron = 0.59, 
C   Nickel = 0.6. Low temperature rates are from Nussbaumer and Storey's 
C   calcs, and the rates of Badnell & Co-workers as well as Romanik
C   (Ap.J 330,1022, 1988) are used.
C                                                                       
C   Direct rad.recomb. for C IV-V, N V-VI, O VI-VIII and Ne VIII-IX from
C   the compilation by Arnaud & Rothenflug, A&A Suppl. 60, 425, 1985.   
C   (Also checked against Landini & Monsignori Fossi, '90, Astr.Ap.Suppl.)
C   Rad. recomb. updated with Verner & Ferland (1995) for H-like, He-like,
C   Li-like and Na-like elements.
C   For some ions (see below) the full rates (rad.+diel.) are from
C   Nahar (priv.comm., '95).
C *********************************************************************   
C
C     ***************************************************************   
C     *****                                                             
C     OXYGEN 
C     *****                                                             
C     ***************************************************************   
C                                                                       
      BDCORR=0.81D0                                                     
      SUPR=1.                                                           
      IF(DEN(IK).GT.1.D9) SUPR=0.26                                     
C     IF(TELOG.GE.5.D0) CALL BADN1(TE,ALDNE7)
      call badn4(te,aldo2,aldo3,aldo4,aldo6)
      CALL BADN2(TE,ALDC3,ALDN4,ALDO5)
      call badn5(te,aldc2,aldn3)
C                                                                       
      call verferl(8.616d-10,0.7563d0,te,1.191d2,4.352d7,radrec)
      alo(9)=radrec
C
      CALL DIEL(6.23D-2,7.01D6,0.304D0,1.47D6,TE,ALD)                  
      ALD=ALD*SUPR*BDCORR                                              
      call verferl(4.897d-10,0.7048d0,te,1.906d2,4.093d7,radrec)
      ALO(8)=radrec+ALD
C
      BDCORR=1.D0                                                     
C
      CALL DIEL(1.06D-1,6.25D6,0.34D0,1.12D6,TE,ALD)                    
      ALD=ALD*SUPR*BDCORR                                              
      call verferl(2.053d-10,0.6019d0,te,4.772d2,1.711d7,radrec)
      ALO(7)=radrec+ALD
C
      ALD=ALDO6                                                         
      CALL DIELB(-2.8425D0,.2283D0,4.0407D1,-3.4956D0,1.7558D0,T4,ALDB) 
      ALD=(ALD+ALDB)*SUPR                                              
      ALO(6)=1.59D-11/T4**.759+ALD                                      
C                                                                       
      ALDB=0.0D0                                                        
      ALD=ALDO5                                                         
      ALD=(ALD+ALDB)*SUPR                                               
      CHTRAO(5)=2.52d-10*t4**.63*(1.d0+2.08d0*expfn(4.16d0*t4))*
     &XH/DEL(IK)+6.4d-10*ab(2)*xa(2,2)*(1.d0+2.d0*expfn(5.5d0*t4))/
     &del(ik)
      ALO(5)=9.6D-12/T4**.670+ALD+CHTRAO(5)                             
C
C      ald=aldo4
C      CALL DIELB(0.0D0,21.88D0,1.6273D1,-.7020D0,1.1899D0,T4,ALDB)      
C      ALD=(ALD+ALDB)*SUPR                                       
      CHTRAO(4)=3.98d-9*t4**.26*(1.d0+0.56d0*expfn(2.62d0*t4))*
     &XH/DEL(IK)+1.d-9*ab(2)*xa(2,2)*(1.d0+1.25d0*expfn(5.8d0*t4))/
     &del(ik)
      call totnah(3,te,alrec)
C      ALO(4)=5.1D-12/T4**.666+ALD+CHTRAO(4)                             
      ALO(4)=alrec+CHTRAO(4)                             
C
C Charge transfer with H from Honvault et al. (AA, 302, 931, '95)
C (OK from <500 K to >1E5 K.)
      ald=aldo3
      CALL DIELB(-3.6D-3,.7519D0,1.5252D0,-8.38D-2,.2769D0,T4,ALDB)     
      ALD=(ALD+ALDB)*SUPR                                       
C      CHTRAO(3)=(.28+.49*T4**.61)*1.D-9*XH/DEL(IK)+
      CHTRAO(3)=(dmax1(0.97d0*t4**(-.074),1.12d0*t4**.129,
     &0.99d0*t4**.3))*1.D-9*XH/DEL(IK)+
     &((1.d-10*ab(2)*xa(2,2))/del(ik))*(dmin1(5.7d0,2.d0*(t4**.95)))
      ALO(3)=2.0D-12/T4**.646+ALD+CHTRAO(3)                             
C
C -- Radiative recomb. from Chung et al. (Phys Rev A 43, 3433, 1991)
      ald=aldo2
      CALL DIELB(0.0D0,2.38D-2,6.59D-2,3.49D-2,.5334D0,T4,ALDB)         
      ALD=(ALD+ALDB)*SUPR                                       
      CHTRAO(2)=1.04D-9*(XH/DEL(IK))*(1.D0-0.61d0*expfn(9.73d0*t4))*
     &t4**.0315
      if(te.lt.4.d3) then
       alo2rr=5.15d-13*(t4/.4d0)**(-.845)
      else
       alo2rr=5.15d-13*(t4/.4d0)**(-.762)
      endif
      ALO(2)=alo2rr+ALD+CHTRAO(2)                             
C                                                                       
C     OXYGEN COLL. IONIZATION (SHULL&STEENBERG)                         
C                                                                       
      IF(TE.LT.3.D5) GOTO 801                                           
      COO(8)=2.89D-14*DSQRT(TE)*EXPFN(1.01D7/TE)/(1.+0.1*TE/1.01D7)     
      COO(7)=7.90D-14*DSQRT(TE)*EXPFN(8.57D6/TE)/(1.+0.1*TE/8.57D6)     
      GOTO 805                                                          
  801 COO(8)=0.0D0                                                      
      COO(7)=0.0D0                                                      
  805 IF(TE.LT.3.D4) GOTO 806                                           
      COO(6)=1.15D-12*DSQRT(TE)*EXPFN(1.60D6/TE)/(1.+0.1*TE/1.60D6)     
      COO(5)=3.33D-12*DSQRT(TE)*EXPFN(1.32D6/TE)/(1.+0.1*TE/1.32D6)     
C    (at 20 eV)                                                         
      COO(5)=1.12D0*COO(5)                                              
      GOTO 808                                                          
  806 DO 803 JH=5,6                                                     
  803 COO(JH)=0.0D0                                                     
  808 COO(4)=7.60D-12*DSQRT(TE)*EXPFN(8.99D5/TE)/(1.+0.1*TE/8.99D5)     
      COO(3)=1.67D-11*DSQRT(TE)*EXPFN(6.37D5/TE)/(1.+0.1*TE/6.37D5)     
      COO(2)=3.96D-11*DSQRT(TE)*EXPFN(4.07D5/TE)/(1.+0.1*TE/4.07D5)     
      COO(1)=1.09D-10*DSQRT(TE)*EXPFN(1.58D5/TE)/(1.+0.1*TE/1.58D5)     
C                                                                       
C     AUGER IONIZATION EQUATIONS                                        
C                                                                       
C     C-T IONIZATION OF O I  (Kingdon & Ferland '96)
C                                                                       
      CTION=1.09d-9*t4**(-.22)*(AB(1)-XH)*EXPFN(2.27D2/TE)*(1.D0-
     &0.97d0*EXPFN(3.77d0*T4))
      ZA(1)=ZE(14)+CTION*DEN(IK)                                 
      IF(ZA(1).LE.1.D-33) GOTO 1278                                     
      ZA(2)=ZE(15)
      ZA(3)=ZE(16)
      ZA(4)=ZE(17)
      ZA(5)=ZE(7)
      ZA(6)=ZE(4)
      ZA(7)=ZK(5)                         
      ZA(8)=ZK(6)                                                 
 1278 CONTINUE                                                          
C                                                                       
C     ***************************************************************   
C     *****                                                             
C     CARBON
C     *****                                                             
C     ***************************************************************   
C                                                                       
C     REC. COEFF.  FOR CARBON (ALD. & PEQ.)                             
C                                                                       
 9876 CONTINUE                                                          
      IF(DEN(IK).GT.1.D9) SUPR=0.17                                     
      ALC(1)=0.0D0                                                      
C
C      ald=aldc2
C      CALL DIELB(1.08D-2,-.1075D0,.281D0,-1.93D-2,-.1127D0,T4,ALDB)
C      ALD=(ALD+ALDB)*SUPR
C      ALC(2)=4.7D-13/T4**.624+ALD
      call totnah(17,te,alrec)
      ALC(2)=alrec
C --- artificially reduce this to simulate the diffuse rad.field < 13.6 eV
C     alc(2)=1.d-5*alc(2)
C
C      ALDB=0.0D0
C      ALD=ALDC3
C      ALD=(ALD+ALDB)*SUPR
      IF(T4.LT.5.) THEN
        CTRA1=1.67D-13*(T4**2.79)*(1.+304.72d0*EXPFN(4.07*T4))
      ELSE
        CTRA1=1.49D-11
      ENDIF
      CHTRAC(3)=CTRA1*XH/DEL(IK)
C      ALC(3)=2.3D-12/T4**.645+ALD+CHTRAC(3)
      call totnah(18,te,alrec)
      ALC(3)=alrec+CHTRAC(3)
C
C Charge exchange from Herrero et al. (J.Phys.B 21, 4607, '95)
C (Somewhat lower than Heil et al. ['81] at T>1E4 K)
C      CALL DIEL(1.62D-3,8.19D4,0.343D0,1.59D5,TE,ALD)
C      CALL DIELB(2.3196D0,1.0733D1,6.883D0,-.1824D0,.4101D0,T4,ALDB)
C      ALD=(ALD+ALDB)*SUPR
C      CHTRAC(4)=1.D-9*(1.49+2.09*T4**.39)*XH/DEL(IK) %Heil et al. ('81)
      CHTRAC(4)=3.45D-9*T4**.178*XH/DEL(IK)
C      ALC(4)=4.90D-12/T4**.803+ALD+CHTRAC(4)
      call totnah(19,te,alrec)
      ALC(4)=alrec+CHTRAC(4)
C
C -- Charge transfer a bit obsolete, but unimportant ...
C      CALL DIEL(4.78D-2,3.44D6,0.362D0,5.87D5,TE,ALD)
C      ALD=ALD*SUPR
      IF(T4.LT.1.) THEN
        CHTRA2=7.6D-10*(T4**1.4)
      ELSE
	CHTRA2=7.6D-10
      ENDIF
      CHTRAC(5)=CHTRA2*XH/DEL(IK)
C      call verferl(8.540d-11,0.5247d0,te,5.014d2,1.479d7,radrec)
C      ALC(5)=radrec+ALD+CHTRAC(5)
      call totnah(20,te,alrec)
      ALC(5)=alrec+CHTRAC(5)
C
C      CALL DIEL(3.22D-2,4.06D6,0.315D0,8.31D5,TE,ALD)
C      ALD=ALD*SUPR
C      call verferl(2.765d-10,0.6858d0,te,1.535d2,2.556d7,radrec)
      call totnah(21,te,alrec)
      ALC(6)=alrec
C
C      call verferl(6.556d-10,0.7567d0,te,6.523d1,2.446d7,radrec)
      call totnah(22,te,alrec)
      ALC(7)=alrec
C                                                                       
C      COLL. IONIZATION (LOTZ)                                          
C                                                                       
      COC(1)=1.44D-10*DSQRT(TE)*EXPFN(13.1/T4)                           
      COC(2)=4.0D-10*DSQRT(TE)*EXPFN(35.8/T4)                           
C    (considerable improvement around a few eVs)                        
      COC(2)=COC(2)*((T4/3.2494)**(-1.46))                              
      COC(3)=2.3D-11*DSQRT(TE)*EXPFN(55.6/T4)                           
C    (at 10 eV)                                                         
      COC(3)=0.894D0*COC(3)                                             
      COC(4)=6.4D-12*DSQRT(TE)*EXPFN(74.8/T4)                           
C    (at 14 eV)                                                         
      COC(4)=0.88D0*COC(4)                                              
      COC(5)=0.                                                         
      IF(T4.LT.10.) GOTO 891                                            
      COC(5)=3.5D-13*DSQRT(TE)*EXPFN(454./T4)                           
  891 COC(6)=0.                                                         
      IF(T4.LT.10.) GOTO 892                                            
      COC(6)=1.1D-13*DSQRT(TE)*EXPFN(568./T4)                           
  892 CONTINUE                                                          
C                                                                       
C     AUGER IONIZATION EQUATIONS                                        
C                                                                       
      AUG=0.                                                            
      ZB(1)=ZE(12)
      IF(ZB(1).LE.1.D-33) GOTO 1279                                     
      ZB(2)=ZE(13)
      ZB(3)=ZE(8)
      ZB(4)=ZE(9)
      ZB(5)=ZK(10)
      ZB(6)=ZK(11)
 1279 CONTINUE                                                          
C                                                                       
C     ***************************************************************   
C     *****                                                             
C     NITROGEN
C     *****                                                             
C     ***************************************************************   
C                                                                       
C     NITROGEN COLL. ION. RATES (LOTZ)                                  
C                                                                       
      CON(1)=6.52D-11*DSQRT(TE)*EXPFN(16.8/T4)                          
      CON(2)=4.33D-11*DSQRT(TE)*EXPFN(34.3/T4)                          
      CON(3)=1.25D-11*DSQRT(TE)*EXPFN(55./T4)                           
C    (around 7-14 eV)                                                   
      CON(3)=1.374D0*CON(3)*(T4/11.605)**0.189                          
      CON(4)=0.                                                         
      IF(T4.LT.2.) GOTO 885                                             
      CON(4)=8.96D-12*DSQRT(TE)*EXPFN(89.9/T4)                          
C    (at 14 eV)                                                         
      CON(4)=0.888D0*CON(4)                                             
  885 CON(5)=0.                                                         
      IF(T4.LT.2.) GOTO 886                                             
      CON(5)=2.83D-12*DSQRT(TE)*EXPFN(113.6/T4)                         
C    (at 14 eV)                                                         
      CON(5)=0.908D0*CON(5)                                             
  886 CON(6)=0.                                                         
      CON(7)=0.                                                         
      IF(T4.LT.10.) GOTO 823                                            
      CON(6)=1.44D-13*DSQRT(TE)*EXPFN(641./T4)/(1.+0.1*T4/641.)         
      CON(7)=4.93D-14*DSQRT(TE)*EXPFN(774./T4)/(1.+0.1*T4/774.)         
 823  CONTINUE                                                          
C                                                                       
C     NITROGEN RECOMB. COEFF (ALD.&PEQ, S & S)                          
C                                                                       
      ALN(1)=0.
C
      IF(DEN(IK).GT.1.D9) SUPR=0.20
C      CALL DIEL(2.98D-3,2.20D5,0.0D0,0.0D0,TE,ALD)
C      CALL DIELB(0.D0,.631D0,.199D0,-1.97D-2,.4398D0,T4,ALDB)
C      ALD=(ALD+ALDB)*SUPR
      IF(T4.LT.5.) THEN
	CHTRA3=1.01D-12*(T4**(-.29))*(1.-0.92*EXPFN(8.38*T4))
      ELSE
        CHTRA3=6.5D-13
      ENDIF
      CHTRAN(2)=CHTRA3*XH/DEL(IK)
C      ALN(2)=4.1D-13/T4**.608+ALD+CHTRAN(2)
      call totnah(23,te,alrec)
      ALN(2)=alrec+CHTRAN(2)
C
C  H charge transfer: Herrero et al. (J.Phys.B 28, 711 (1995)).
      chtrcn=0.49d0+0.073d0*t4
      if(t4.gt.20.d0) chtrcn=1.95d0
C (old) chtrcn=0.57d0+0.29d0*t4**.46
C      ald=aldn3
C      CALL DIELB(3.2D-2,-.6624D0,4.3191D0,3.D-4,.5946D0,T4,ALDB)        
C      ALD=(ALD+ALDB)*SUPR                                              
      CHTRAN(3)=chtrcn*1.D-9*XH/DEL(IK)+
     &(3.3d-10*ab(2)*xa(2,2)/del(ik))*(t4**.29)*(1.d0+1.3d0*
     &expfn(4.5d0*t4))
C      ALN(3)=2.2D-12/T4**.639+ALD+CHTRAN(3)                             
      call totnah(24,te,alrec)
      ALN(3)=alrec+CHTRAN(3)
C
      ALDB=0.0D0                                                        
C      ALD=ALDN4
C      ALD=(ALD+ALDB)*SUPR
      CHTRA4=4.54d-9*t4**.57*(1.d0-.65d0*expfn(0.89d0*t4))
      CHTRAN(4)=CHTRA4*XH/DEL(IK)+1.5d-10*ab(2)*xa(2,2)/del(ik)
C -- Artificially increase diel.rec. from N IV --> N III by factor 'dicorr'
C      dicorr=1.5d0
C      ALN(4)=5.0D-12/T4**.676+dicorr*ALD+CHTRAN(4)
      call totnah(25,te,alrec)
      ALN(4)=alrec+CHTRAN(4)
C
C - Charge transfer from Zygelman et al. (Phys Rev A 46, 3846, 1992)
C     aldb=0.0d0
C     call romn5(tryd,ald)
C      CALL DIEL(2.62d-3,1.02d5,0.243d0,1.25d5,te,ald)
C      CALL DIELB(.4134d0,-4.6319d0,2.5917d1,-2.229d0,.236d0,t4,aldb)
C      ALD=(ALD+ALDB)*SUPR
      CHTRA5=8.5D-9*(T4/3.d0)**.414
      CHTRAN(5)=CHTRA5*XH/DEL(IK)+
     &1.7d-9*ab(2)*xa(2,2)*(1.d0+2.5d0*expfn(3.7d0*t4))/del(ik)
C -- Artificially decrease diel.rec. from N V --> N IV by factor 'dicorr'
C      ALN(5)=9.40D-12/T4**.765+ALD+CHTRAN(5)
      call totnah(26,te,alrec)
C      ALN(5)=alrec*0.5d0+CHTRAN(5)
      ALN(5)=alrec+CHTRAN(5)
C
C      CALL DIEL(7.50D-2,4.75D6,0.35D0,8.35D5,TE,ALD)
C      ALD=ALD*SUPR
C      call verferl(1.169d-10,0.5470d0,te,6.793d2,1.650d7,radrec)
C      ALN(6)=radrec+ALD
      call totnah(27,te,alrec)
      ALN(6)=alrec
C
C      CALL DIEL(4.61D-2,5.44D6,0.309D0,1.14D6,TE,ALD)
C      ALD=ALD*SUPR
C      call verferl(3.910d-10,0.6988d0,te,1.611d2,3.271d7,radrec)
C      ALN(7)=radrec+ALD
      call totnah(28,te,alrec)
      ALN(7)=alrec
C
C      call verferl(7.586d-10,0.7563d0,te,9.015d1,3.338d7,radrec)
C      ALN(8)=radrec
      call totnah(29,te,alrec)
      ALN(8)=alrec
C                                                                       
C     AUGER IONIZATION EQUATIONS                                        
C                                                                      
      CTION=4.55d-12*t4**(-.29)*(AB(1)-XH)*EXPFN(1.086d4/TE)*(1.D0-
     &0.92d0*EXPFN(8.38d0*T4))
      ZC(1)=ZE(18)+CTION*DEN(IK)
      IF(ZC(1).LE.1.D-33) GOTO 1276                                     
      ZC(2)=ZE(19)
      ZC(3)=ZE(20)
      ZC(4)=ZE(21)
      ZC(5)=ZE(22)
      ZC(6)=ZK(23)
      ZC(7)=ZK(24)
 1276 CONTINUE                                                          
C     ***************************************************************   
C     *****                                                             
C     COLLISIONAL IONIZATION RATES INCLUDING AUTOIONIZATION
C     (Arnaud & Rothenflug)
C     *****                                                             
C     ***************************************************************   
C -Hydrogen-
C      cionh=arf(22.8d0,-12.d0,1.9d0,-22.6d0,13.6d0,tev)
C -Helium-
C      cheo=arf(17.8d0,-11.d0,7.d0,-23.2d0,24.6d0,tev)
C      che=arf(14.4d0,-5.6d0,1.9d0,-13.3d0,54.4d0,tev)
C -Carbon-
C      coc(1)=arf(6.d0,-16.d0,12.d0,-15.1d0,11.3d0,tev)
C      coc(1)=coc(1)+arf(24.3d0,-7.8d0,2.5d0,-24.d0,16.6d0,tev)
C      coc(2)=arf(16.d0,-9.d0,2.5d0,-10.5d0,24.4d0,tev)
C      coc(2)=coc(2)+arf(23.7d0,-7.6d0,2.5d0,-21.7d0,30.9d0,tev)
C      coc(3)=arf(23.2d0,-7.4d0,2.5d0,-19.4d0,47.9d0,tev)
C      coc(3)=coc(3)+arf(20.d0,-5.6d0,4.1d0,-18.d0,325.d0,tev)
C      coc(4)=arf(8.2d0,-2.7d0,1.4d0,-6.6d0,64.5d0,tev)
C      coc(4)=coc(4)+arf(20.d0,-5.6d0,4.1d0,-18.d0,343.d0,tev)
C      coc(5)=arf(20.4d0,-6.1d0,4.5d0,-18.d0,392.d0,tev)
C      coc(6)=arf(12.2d0,-3.9d0,1.9d0,-10.3d0,490.d0,tev)
C -Nitrogen-
C      con(1)=arf(19.5d0,-30.5d0,15.d0,-29.d0,14.5d0,tev)
C      con(1)=con(1)+arf(19.d0,-4.5d0,2.8d0,-20.2d0,20.3d0,tev)
C      con(2)=arf(21.d0,-9.d0,5.3d0,-22.5d0,29.6d0,tev)
C      con(2)=con(2)+arf(18.5d0,-4.3d0,2.8d0,-18.d0,36.7d0,tev)
C      con(3)=arf(16.d0,-7.5d0,2.3d0,-10.d0,47.4d0,tev)
C      con(3)=con(3)+arf(18.1d0,-4.d0,2.8d0,-15.8d0,55.8d0,tev)
C      con(4)=arf(17.6d0,-3.8d0,2.8d0,-13.6d0,77.5d0,tev)
C      con(4)=con(4)+arf(20.5d0,-5.8d0,4.1d0,-18.d0,471.d0,tev)
C      con(5)=arf(10.5d0,-3.3d0,1.4d0,-7.7d0,97.9d0,tev)
C      con(5)=con(5)+arf(20.5d0,-5.8d0,4.1d0,-18.d0,493.d0,tev)
C      con(6)=arf(20.8d0,-6.3d0,4.4d0,-18.2d0,552.d0,tev)
C      con(7)=arf(12.3d0,-4.d0,1.9d0,-10.3d0,667.d0,tev)
C -Oxygen-
C      coo(1)=arf(9.5d0,-17.5d0,12.5d0,-19.5d0,13.6d0,tev)
C      coo(1)=coo(1)+arf(18.2d0,-4.d0,2.8d0,-20.2d0,28.5d0,tev)
C      coo(2)=arf(25.d0,-8.d0,8.4d0,-29.5d0,35.1d0,tev)
C      coo(2)=coo(2)+arf(17.8d0,-3.8d0,2.9d0,-18.1d0,42.6d0,tev)
C      coo(3)=arf(25.d0,-7.d0,5.d0,-18.d0,54.9d0,tev)
C      coo(3)=coo(3)+arf(17.3d0,-3.5d0,2.9d0,-16.1d0,63.8d0,tev)
C      coo(4)=arf(15.d0,-5.d0,2.2d0,-10.5d0,77.4d0,tev)
C      coo(4)=coo(4)+arf(16.8d0,-3.3d0,2.8d0,-14.1d0,87.6d0,tev)
C      coo(5)=arf(16.4d0,-3.d0,2.9d0,-12.d0,114.d0,tev)
C      coo(5)=coo(5)+arf(20.8d0,-6.d0,4.1d0,-18.d0,644.d0,tev)
C      coo(6)=arf(10.4d0,-3.3d0,1.4d0,-7.4d0,138.d0,tev)
C      coo(6)=coo(6)+arf(20.8d0,-6.d0,4.1d0,-18.d0,670.d0,tev)
C      coo(7)=arf(21.2d0,-6.5d0,4.3d0,-18.4d0,739.d0,tev)
C      coo(8)=arf(12.3d0,-4.d0,1.9d0,-10.3d0,871.d0,tev)
C                                                                       
C     ***************************************************************   
C     *****                                                             
C     NEON
C     *****                                                             
C     ***************************************************************   
C     ***************************************************************   
      cone(1)=arf(40.d0,-42.d0,18.d0,-56.d0,21.6d0,tev)
      cone(1)=cone(1)+arf(19.d0,-4.9d0,2.8d0,-22.d0,48.5d0,tev)
      cone(2)=arf(37.d0,-33.d0,15.5d0,-46.d0,41.1d0,tev)
      cone(2)=cone(2)+arf(18.6d0,-4.6d0,2.8d0,-20.2d0,66.4d0,tev)
      cone(3)=arf(33.d0,-17.5d0,11.2d0,-33.d0,63.5d0,tev)
      cone(3)=cone(3)+arf(18.2d0,-4.4d0,2.8d0,-18.4d0,86.2d0,tev)
      cone(4)=arf(34.d0,-10.d0,7.5d0,-25.d0,97.1d0,tev)
      cone(4)=cone(4)+arf(17.8d0,-4.d0,2.8d0,-16.7d0,108.d0,tev)
      cone(5)=arf(25.5d0,-8.5d0,4.5d0,-16.8d0,126.d0,tev)
      cone(5)=cone(5)+arf(17.4d0,-3.8d0,2.8d0,-14.9d0,139.d0,tev)
      cone(6)=arf(14.5d0,-4.6d0,1.9d0,-8.5d0,158.d0,tev)
      cone(6)=cone(6)+arf(16.9d0,-3.4d0,2.8d0,-13.2d0,172.d0,tev)
      cone(7)=arf(16.5d0,-3.1d0,2.8d0,-11.4d0,207.d0,tev)
      cone(7)=cone(7)+arf(21.5d0,-6.4d0,4.1d0,-18.d0,644.d0,tev)
      cone(8)=arf(10.1d0,-3.1d0,1.4d0,-7.1d0,239.d0,tev)
      cone(8)=cone(8)+arf(21.5d0,-6.4d0,4.1d0,-18.d0,1107.d0,tev)
      cone(8)=cone(8)+arauli(10,tev)
C                                                                       
      ZNEO(1)=ZE(35)
      IF(ZNEO(1).LE.1.D-33) GOTO 4576                                   
      ZNEO(2)=ZE(36)
      ZNEO(3)=ZE(37)
      ZNEO(4)=ZE(38)
      ZNEO(5)=ZE(39)
      ZNEO(6)=ZE(40)
      ZNEO(7)=ZE(41)
      ZNEO(8)=ZE(42)
 4576 CONTINUE                                                          
C                                                                       
C     NEON RECOMB. COEFF (ALD.&PEQ, S & S)                          
C                                                                       
      BDCORR=1.D0
      SUPR=1.D0
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
      ALNEO(1)=0.                                                       
C
      CALL DIEL(9.77D-4,3.11D5,7.30D-2,2.06D5,TE,ALD)                
      ALDB=0.0D0
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALNEO(2)=2.2D-13/T4**.759+ALD                             
C                                                                       
      CALL DIEL(2.65D-3,2.84D5,0.242D0,3.07D5,TE,ALD)                  
      CALL DIELB(.0129D0,-.1779D0,.9353D0,-.0682D0,.4516D0,T4,ALDB)
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALNEO(3)=1.5D-12/T4**.693+ALD                             
C
      CALL DIEL(3.69D-3,2.24D5,1.01D0,2.94D5,TE,ALD)                   
      CALL DIELB(3.6781D0,14.1481D0,17.1175D0,-.5017D0,.2313D0,T4,      
     &ALDB)
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
C     C-T BHD 80                                                        
      CHCONS=14.73D-9*(T4**.0452)*(1.D0-0.84D0*EXPFN(0.31D0*T4))
      IF(T4.GT.5.D0) CHCONS=1.295D-8
      CHTRNE=XH*CHCONS/DEL(IK)
      ALNEO(4)=4.4D-12/T4**.675+ALD+CHTRNE                             
C
      CALL DIEL(1.18D-2,2.70D5,.391D0,5.50D5,TE,ALD)                   
      CALL DIELB(-.0254D0,5.5365D0,1.70727D1,-0.7225D0,.1702D0,T4,    
     &ALDB)
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
C     C-T BD 80                                                        
      CHCONS=6.47D-9*(T4**.54)*(1.D0+3.59D0*EXPFN(5.22D0*T4))
      IF(T4.GT.3.D0) CHCONS=1.169D-8
      CHTRNE=XH*CHCONS/DEL(IK)+
     &(1.7d-9*ab(2)*xa(2,2)/del(ik))*(t4**.52)*(1.d0+3.3d0*
     &expfn(5.3d0*t4))
      ALNEO(5)=9.1D-12/T4**.668+ALD+CHTRNE                             
C
C      CALL DIEL(2.44D-2,3.09D5,2.52D0,9.91D5,TE,ALD)                    
C      CALL DIELB(-.0141D0,33.8479D0,43.1608D0,-1.6072D0,.1942D0,T4,    
C     &ALDB)
C      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
C      ALNEO(6)=1.50D-11/T4**.684+ALD                                   
      call totnah(5,te,alrec)
      alneo(6)=alrec
C
C -- Badnell has done this too....
      CALL DIEL(3.02D-2,2.83D5,0.445D0,1.73D6,TE,ALD)                   
      CALL DIELB(19.928D0,235.0536D0,152.5096D0,9.1413D0,0.1282D0,
     &T4,ALDB)
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALNEO(7)=2.30D-11/T4**.704+ALD                                   
C
      CALL DIEL(6.10D-3,1.68D5,0.254D0,6.13D5,TE,ALD)                   
      CALL DIELB(5.4751D0,203.9751D0,86.9016D0,-7.4568D0,2.5145D0,
     &T4,ALDB)
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALNEO(8)=3.46D-11/T4**.742+ALD                                   
C
      CALL DIEL(2.52D-1,1.40D7,0.304D0,1.80D6,TE,ALD)                   
      ALD=ALD*BDCORR*SUPR                                       
      call verferl(3.200d-10,0.6198d0,te,6.329d2,2.616d7,radrec)
      ALNEO(9)=radrec+ALD                                   
C
C     ***************************************************************   
C     *****                                                             
C     SULPHUR
C     *****                                                             
C     ***************************************************************   
      cosu(1)=arf(6.d0,-22.d0,20.d0,-20.d0,10.4d0,tev)
      cosu(1)=cosu(1)+arf(51.3d0,-33.2d0,1.4d0,-40.2d0,20.2d0,tev)
      cosu(1)=cosu(1)+arau(16,16,tev)
      cosu(2)=arf(98.7d0,-65.4d0,1.9d0,-72.3d0,23.4d0,tev)
      cosu(2)=cosu(2)+arf(52.5d0,-34.5d0,1.4d0,-40.5d0,30.7d0,tev)
      cosu(2)=cosu(2)+arau(16,15,tev)
      cosu(3)=arf(74.5d0,-49.4d0,1.3d0,-54.6d0,35.d0,tev)
      cosu(3)=cosu(3)+arf(53.8d0,-35.8d0,1.4d0,-40.7d0,43.8d0,tev)
      cosu(3)=cosu(3)+arau(16,14,tev)
      cosu(4)=arf(50.4d0,-33.4d0,0.6d0,-36.9d0,47.3d0,tev)
      cosu(4)=cosu(4)+arf(55.1d0,-37.2d0,1.4d0,-41.d0,57.6d0,tev)
      cosu(4)=cosu(4)+arau(16,13,tev)
      cosu(5)=arf(19.8d0,-5.7d0,1.6d0,-11.9d0,72.7d0,tev)
      cosu(5)=cosu(5)+arf(73.2d0,-27.d0,15.8d0,-61.1d0,239.d0,tev)
      cosu(5)=cosu(5)+arf(23.1d0,-8.d0,3.3d0,-19.5d0,288.2d0,tev)
      cosu(5)=cosu(5)+arau(16,12,tev)
      cosu(6)=arf(9.d0,-2.8d0,0.7d0,-5.4d0,88.1d0,tev)
      cosu(6)=cosu(6)+arf(73.2d0,-27.d0,15.8d0,-61.1d0,257.d0,tev)
      cosu(6)=cosu(6)+arf(23.1d0,-8.d0,3.3d0,-19.5d0,309.7d0,tev)
      cosu(6)=cosu(6)+arau(16,11,tev)
      cosu(7)=arf(72.d0,-24.1d0,14.2d0,-50.d0,281.d0,tev)
      cosu(7)=cosu(7)+arf(19.6d0,-6.8d0,2.8d0,-17.5d0,343.d0,tev)
      cosu(8)=arf(60.8d0,-20.2d0,10.9d0,-41.7d0,328.d0,tev)
      cosu(8)=cosu(8)+arf(19.3d0,-6.3d0,2.8d0,-16.6d0,384.d0,tev)
      cosu(9)=arf(49.5d0,-16.3d0,8.d0,-33.4d0,379.d0,tev)
      cosu(9)=cosu(9)+arf(19.1d0,-5.9d0,2.8d0,-15.6d0,426.d0,tev)
      cosu(10)=arf(38.3d0,-12.4d0,5.5d0,-25.1d0,447.d0,tev)
      cosu(10)=cosu(10)+arf(18.8d0,-5.5d0,2.8d0,-14.7d0,469.d0,tev)
C
C     AUGER IONIZATION EQUATIONS                                        
C                                                                       
C     C-T IONIZATION OF S I (BUTLER AND DALGARNO 80)                    
C     (Arnauld and Rothenflug, 85)                                      
C                                                                       
      CTION=1.D-9*(AB(1)-XH)
      ZSUL(1)=ZE(25)+CTION*DEN(IK)                               
      IF(ZSUL(1).LE.1.D-33) GOTO 5576                                   
      ZSUL(2)=ZE(26)                                             
      ZSUL(3)=ZE(27)                                             
      ZSUL(4)=ZE(28)                                             
      ZSUL(5)=ZE(29)                                             
      ZSUL(6)=ZE(30)                                             
      ZSUL(7)=ZE(31)                                             
      ZSUL(8)=ZE(32)                                             
      ZSUL(9)=ZE(33)                                             
      ZSUL(10)=ZE(34)                                            
 5576 CONTINUE                                                          
C                                                                       
C     SULPHUR RECOMB. COEFF (ALD.&PEQ, S & S)                          
C -- Take high-T part of diel.rec. from Badnell. Compare this to results
C    for O (Badnell & Pindzola) at 5E4 K. Use scale factors to scale low-
C    T results by N & S for O to estimate low-T diel.rec. for S.
C    Assume that the shape of S VI -> S V recombination follows that of
C    S V rec. for low-T.
C
      CALL BADN3(TE,alds2,alds3,alds4,alds5,alds6)
      BDCORR=1.D0
      SUPR=1.D0
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
      ALSUL(1)=0.                                                       
C
      CALL DIELB(0.0D0,0.0238D0,0.0659D0,0.0349D0,0.5334D0,T4,ALDB)
      aldssi2=alds2+1.84d0*aldb
      ALSUL(2)=4.1D-13/T4**.630+ALDSSI2*SUPR
C --- artificially reduce this to simulate the diffuse rad.field < 13.6 eV
C     alsul(2)=1.d-5*alsul(2)
C                                                                       
C      CALL DIELB(-.0036D0,.7519D0,1.5252D0,-.0838D0,.2769D0,T4,ALDB)
C      aldssi3=alds3+6.10d0*aldb
C      ALSUL(3)=1.8D-12/T4**.686+ALDSSI3*SUPR
      call totnah(13,te,alrec)
      alsul(3)=alrec
C
c     C-T BHD 80                                                        
      CHCONS=2.29D-9*t4**.0402*(1.D0+1.59D0*EXPFN(6.06D0*T4))
      IF(T4.GT.3.D0) CHCONS=2.3D-9
      CHTRSU=XH*CHCONS/DEL(IK)+
     &(1.1d-9*ab(2)*xa(2,2)/del(ik))*(t4**.56)
C      CALL DIELB(0.0D0,21.879D0,16.273D0,-.702D0,1.1899D0,T4,ALDB)
C      aldssi4=alds4+7.67d0*aldb
C      ALSUL(4)=2.7D-12/T4**.745+ALDSSI4*SUPR+CHTRSU
      call totnah(14,te,alrec)
      alsul(4)=alrec+CHTRSU
C
C     C-T BD 80                                                        
      CHCONS=6.44D-9*t4**.13*(1.D0+2.69D0*EXPFN(5.69D0*T4))
      IF(T4.GT.3.D0) CHCONS=7.0D-9
      CHTRSU=XH*CHCONS/DEL(IK)
      CALL DIELB(.0061D0,.2269D0,32.1419D0,1.9939D0,-.0646D0,T4,ALDB)
      aldssi5=alds5+2.14d0*aldb
      ALSUL(5)=5.7D-12/T4**.755+ALDSSI5*SUPR+CHTRSU                             
C
      CALL DIELB(.0061D0,.2269D0,32.1419D0,1.9939D0,-.0646D0,T4,ALDB)
      aldssi6=alds6+3.27d0*aldb
      ALSUL(6)=1.2D-11/T4**.701+ALDSSI6*SUPR
C
      ALDB=0.0D0
C
C - Romanik, 1988 for diel.rec. for S VII -> S VI
      ald=(6.59d-11*expfn(8.8d0/tryd)+3.82d-10*expfn(12.d0/tryd)+
     &1.03d-9*expfn(14.5/tryd))/tryd**1.5
       ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      call verferl(3.502d-10,0.6266d0,te,1.532d2,1.755d7,radrec)
      ALSUL(7)=radrec+ALD                                   
C
      CALL DIEL(1.34D-2,6.90D5,1.04D0,2.15D6,TE,ALD)
       ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALSUL(8)=2.70D-11/T4**.733+ALD                                   
C
      CALL DIEL(2.38D-2,5.84D5,1.12D0,2.59D6,TE,ALD)                   
       ALD=ALD*BDCORR*SUPR                                       
      ALSUL(9)=4.00D-11/T4**.696+ALD                                   
C
      CALL DIEL(3.19D-2,5.17D5,1.40D0,2.91D6,TE,ALD)                   
       ALD=ALD*BDCORR*SUPR
      ALSUL(10)=5.50D-11/T4**.711+ALD                                   
C
      CALL DIEL(7.13D-2,6.66D5,1.00D0,2.32D6,TE,ALD)                   
       ALD=ALD*BDCORR*SUPR
      ALSUL(11)=7.40D-11/T4**.716+ALD                                   
C
C     ***************************************************************   
C     *****                                                             
C     IRON
C     *****                                                             
C     ***************************************************************   
      cofe(1)=arf(31.9d0,-15.d0,0.32d0,-28.1d0,7.9d0,tev)
      cofe(1)=cofe(1)+arf(15.0d0,-16.7d0,7.d0,-12.7d0,9.d0,tev)
      cofe(1)=cofe(1)+arf(115.d0,-72.4d0,9.57d0,-107.d0,59.d0,tev)
      cofe(2)=arf(17.4d0,-3.27d0,0.16d0,-10.2d0,16.18d0,tev)
      cofe(2)=cofe(2)+arf(30.1d0,-38.8d0,18.6d0,-45.7d0,24.83d0,tev)
      cofe(2)=cofe(2)+arf(115.d0,-72.4d0,9.57d0,-107.d0,83.37d0,tev)
      cofe(3)=arf(84.8d0,-67.6d0,21.d0,-84.1d0,34.75d0,tev)
      cofe(3)=cofe(3)+arf(87.7d0,-49.6d0,16.4d0,-84.8d0,93.28d0,tev)
      cofe(3)=cofe(3)+arf(25.9d0,-11.7d0,2.32d0,-23.9d0,131.9d0,tev)
      cofe(4)=arf(77.4d0,-43.9d0,19.6d0,-81.9d0,53.74d0,tev)
      cofe(4)=cofe(4)+arf(79.1d0,-30.d0,8.38d0,-74.6d0,111.9d0,tev)
      cofe(4)=cofe(4)+arf(16.7d0,-3.44d0,2.32d0,-15.1d0,151.d0,tev)
      cofe(4)=cofe(4)+arau(26,22,tev)
      cofe(5)=arf(48.1d0,-20.4d0,16.2d0,-48.4d0,75.15d0,tev)
      cofe(5)=cofe(5)+arf(66.8d0,-18.9d0,9.29d0,-60.5d0,132.7d0,tev)
      cofe(5)=cofe(5)+arf(13.4d0,-0.41d0,2.33d0,-10.9d0,172.4d0,tev)
      cofe(5)=cofe(5)+arau(26,21,tev)
      cofe(6)=arf(36.9d0,-11.8d0,10.3d0,-31.5d0,98.69d0,tev)
      cofe(6)=cofe(6)+arf(67.0d0,-18.6d0,9.43d0,-56.5d0,155.5d0,tev)
      cofe(6)=cofe(6)+arf(12.7d0,-.086d0,2.42d0,-8.97d0,195.8d0,tev)
      cofe(7)=arf(14.6d0,-4.36d0,5.98d0,-10.5d0,124.2d0,tev)
      cofe(7)=cofe(7)+arf(67.9d0,-20.6d0,9.82d0,-53.7d0,180.d0,tev)
      cofe(7)=cofe(7)+arf(15.6d0,-2.29d0,2.3d0,-10.6d0,220.9d0,tev)
      cofe(8)=arf(14.3d0,-4.44d0,2.45d0,-9.53d0,151.7d0,tev)
      cofe(8)=cofe(8)+arf(69.9d0,-23.7d0,9.5d0,-51.7d0,213.d0,tev)
      cofe(8)=cofe(8)+arf(19.2d0,-5.7d0,2.3d0,-12.7d0,249.d0,tev)
      cofe(9)=arf(69.9d0,-23.7d0,9.5d0,-51.7d0,235.d0,tev)
      cofe(9)=cofe(9)+arf(19.2d0,-5.7d0,2.3d0,-12.7d0,271.d0,tev)
      cofe(10)=arf(57.7d0,-18.6d0,7.8d0,-40.3d0,262.d0,tev)
      cofe(10)=cofe(10)+arf(21.d0,-7.1d0,2.3d0,-14.1d0,297.d0,tev)
      cofe(11)=arf(45.6d0,-13.9d0,6.2d0,-30.d0,290.d0,tev)
      cofe(11)=cofe(11)+arf(22.8d0,-8.4d0,2.3d0,-15.4d0,324.d0,tev)
      cofe(11)=cofe(11)+arau(26,16,tev)
      cofe(12)=arf(33.4d0,-9.7d0,4.6d0,-20.8d0,331.d0,tev)
      cofe(12)=cofe(12)+arf(24.6d0,-9.8d0,2.3d0,-16.8d0,356.d0,tev)
      cofe(12)=cofe(12)+arau(26,15,tev)
      cofe(13)=arf(21.3d0,-5.9d0,3.d0,-12.6d0,361.d0,tev)
      cofe(13)=cofe(13)+arf(26.4d0,-11.2d0,2.3d0,-18.1d0,388.d0,tev)
      cofe(13)=cofe(13)+arau(26,14,tev)
      cofe(14)=arf(9.1d0,-2.6d0,1.4d0,-5.6d0,392.d0,tev)
      cofe(14)=cofe(14)+arf(28.2d0,-12.5d0,2.3d0,-19.4d0,421.d0,tev)
      cofe(14)=cofe(14)+arau(26,13,tev)
      cofe(15)=arf(19.8d0,-5.7d0,2.1d0,-11.9d0,457.d0,tev)
      cofe(15)=cofe(15)+arf(78.9d0,-27.d0,10.6d0,-52.8d0,1185.d0,tev)
      cofe(15)=cofe(15)+arf(23.5d0,-8.4d0,3.3d0,-17.8d0,1254.3d0,tev)
      cofe(15)=cofe(15)+arau(26,12,tev)
C     ***************************************************************   
C     *****                                                             
C
      do 6776 i=1,15
      zfe(i)=ze(i+42)
 6776 continue
C -- Charge transfer ionization of Fe II to Fe III (Kingdon & Ferland '96)
      CTION=2.1D-9*(t4**.0772)*(AB(1)-XH)*expfn(3.0054d4/te)*
     &(1.d0-0.41d0*expfn(7.31d0*t4))
      ZFE(2)=ZFE(2)+CTION*DEN(IK)                               
Cc                                                                      
Cc    IRON RECOMBINATION COEFFS. (S & S)
C   Rad.rec. for recombining Fe XV-XVI and diel.rec. from Arnaud & Raymond 
C    ApJ, 398, 394 (1992)
Cc-- (No low temp. diel. recombination data exists !) --
C
      BDCORR=1.D0
      SUPR=1.D0
      ALDB=0.0D0
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
      ALFE(1)=0.                                                       
C
      ald=(2.2d-4*expfn(5.12d0/tev)+1.d-4*expfn(1.29d1/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(2)=1.42D-13/T4**.891+ALD                             
C --- artificially reduce this to simulate the diffuse rad.field < 13.6 eV
C     alfe(2)=1.d-5*alfe(2)
Cc                                                                      
C - Charge transfer recomb. from Fe III from Neufeld & Dalgarno (cf. above)
      ald=(2.3d-3*expfn(1.67d1/tev)+2.7d-3*expfn(3.14d1/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      CHCONS=1.26D-9*t4**.0772*(1.D0-0.41D0*EXPFN(7.31D0*T4))
      CHTRFE=XH*CHCONS/DEL(IK)
      ALFE(3)=1.02D-12/T4**.843+ALD+CHTRFE
C
C - alrec from Nahar ('96)
C      ald=(1.5d-2*expfn(2.86d1/tev)+4.7d-3*expfn(5.21d1/tev))/TE**1.5
      if(t4.gt.0.1d0) then
        CHCONS=3.42D-9*t4**.51*(1.D0-2.06D0*EXPFN(8.99D0*T4))
      else
	CHCONS=0.0d0
      endif
      CHTRFE=XH*CHCONS/DEL(IK)
C      ALD=BDCORR*SUPR*ALD
      call totnah(16,te,alrec)
      ALFE(4)=alrec+ALD+CHTRFE
C
      ald=(3.8d-2*expfn(3.73d1/tev)+1.6d-2*expfn(6.74d1/tev))/TE**1.5
      CHCONS=14.6D-9*t4**.0357*(1.D0-0.92D0*EXPFN(0.37D0*T4))
      CHTRFE=XH*CHCONS/DEL(IK)
      ALD=BDCORR*SUPR*ALD
      ALFE(5)=7.80D-12/T4**.682+ALD+CHTRFE
C
      ald=(8.d-2*expfn(5.42d1/tev)+2.4d-2*expfn(1.d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(6)=1.51D-11/T4**.699+ALD
C
      ald=(9.2d-2*expfn(4.55d1/tev)+4.1d-2*expfn(3.6d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(7)=2.62D-11/T4**.728+ALD
C
      ald=(1.6d-1*expfn(6.67d1/tev)+3.6d-2*expfn(1.23d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(8)=4.12D-11/T4**.759+ALD
C
      ald=(1.8d-1*expfn(6.61d1/tev)+7.d-2*expfn(1.29d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(9)=6.05D-11/T4**.79+ALD
C
      ald=(1.4d-1*expfn(2.16d1/tev)+2.6d-1*expfn(1.36d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(10)=8.13D-11/T4**.81+ALD
C
      ald=(1.d-1*expfn(2.22d1/tev)+2.8d-1*expfn(1.44d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(11)=1.09D-10/T4**.829+ALD
C
      ald=(2.25d-1*expfn(5.96d1/tev)+2.31d-1*expfn(3.62d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(12)=1.33D-10/T4**.828+ALD
C
      ald=(2.4d-1*expfn(7.5d1/tev)+1.7d-1*expfn(2.05d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(13)=1.64D-10/T4**.834+ALD
C
      ald=(2.6d-1*expfn(3.63d1/tev)+1.6d-1*expfn(1.93d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(14)=2.00D-10/T4**.836+ALD
C
      ald=(1.9d-1*expfn(3.94d1/tev)+9.d-2*expfn(1.98d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(15)=1.46D-10/(T4**(-.597d0-.0522d0*dlog10(t4)))+ALD
C
      ald=(1.2d-1*expfn(2.46d1/tev)+1.2d-1*expfn(2.48d2/tev)+6.d-1*
     &expfn(5.6d2/tev))/TE**1.5
      ALD=BDCORR*SUPR*ALD
      ALFE(16)=1.68D-10/(T4**(-.602d0-.0507d0*dlog10(t4)))+ALD
C                                                                       
C     ***************************************************************   
C     *****                                                             
C     SILICON
C     *****                                                             
C     ***************************************************************   
      cosi(1)=arf(74.5d0,-49.4d0,1.3d0,-54.6d0,8.1d0,tev)
      cosi(1)=cosi(1)+arf(53.8d0,-35.8d0,1.4d0,-40.7d0,13.5d0,tev)
      cosi(1)=cosi(1)+arau(14,14,tev)
      cosi(2)=arf(50.4d0,-33.4d0,0.6d0,-36.9d0,16.3d0,tev)
      cosi(2)=cosi(2)+arf(55.1d0,-37.2d0,1.4d0,-41.d0,22.9d0,tev)
      cosi(2)=cosi(2)+arau(14,13,tev)
      cosi(3)=arf(19.8d0,-5.7d0,1.3d0,-11.9d0,33.5d0,tev)
      cosi(3)=cosi(3)+arf(66.7d0,-24.8d0,18.7d0,-65.d0,133.d0,tev)
      cosi(3)=cosi(3)+arf(22.d0,-7.2d0,3.3d0,-20.9d0,176.6d0,tev)
      cosi(3)=cosi(3)+arau(14,12,tev)
      cosi(4)=arf(9.d0,-3.d0,0.6d0,-5.8d0,45.1d0,tev)
      cosi(4)=cosi(4)+arf(66.7d0,-24.8d0,18.7d0,-65.d0,148.d0,tev)
      cosi(4)=cosi(4)+arf(22.d0,-7.2d0,3.3d0,-20.9d0,193.5d0,tev)
      cosi(4)=cosi(4)+arau(14,11,tev)
      cosi(5)=arf(72.d0,-24.1d0,17.4d0,-50.d0,167.d0,tev)
      cosi(5)=cosi(5)+arf(19.6d0,-6.2d0,2.8d0,-19.d0,217.d0,tev)
      cosi(6)=arf(60.8d0,-20.2d0,13.2d0,-41.7d0,205.d0,tev)
      cosi(6)=cosi(6)+arf(19.3d0,-5.8d0,2.8d0,-17.8d0,250.d0,tev)
      cosi(7)=arf(49.5d0,-16.3d0,9.6d0,-33.4d0,246.d0,tev)
      cosi(7)=cosi(7)+arf(19.d0,-5.4d0,2.8d0,-16.6d0,285.d0,tev)
      cosi(8)=arf(38.3d0,-12.4d0,6.4d0,-25.1d0,303.d0,tev)
      cosi(8)=cosi(8)+arf(18.6d0,-5.1d0,2.8d0,-15.4d0,321.d0,tev)
      cosi(9)=arf(27.d0,-8.5d0,3.8d0,-16.8d0,351.d0,tev)
      cosi(9)=cosi(9)+arf(18.3d0,-4.7d0,2.8d0,-14.1d0,371.d0,tev)
      cosi(10)=arf(14.d0,-4.6d0,1.6d0,-8.5d0,401.d0,tev)
      cosi(10)=cosi(10)+arf(18.d0,-4.3d0,2.8d0,-12.9d0,423.d0,tev)
C                                                                       
C     C-T IONIZATION OF Si I and Si II with H II, and Si III with He II
C     (Arnauld and Rothenflug, 85; Si II Kingdon & Ferland '96)
C     (Si I Kimura et al. '96 (to be publ. in ApJ).)
C                                                                       
      CTION=1.45d-9*(t4**.985)*(AB(1)-XH)
      ZSI(1)=ZE(58)+CTION*DEN(IK)                               
      CTION=0.41D-9*(t4**.24)*(expfn(3.178d4/te))*(AB(1)-XH)*
     &(1.d0+3.17d0*expfn(4.18d-3*t4))
      ZSI(2)=ZE(59)+CTION*DEN(IK)
      CTION=1.15D-9*(t4**.44)*(expfn(8.88d0/tev))*ab(2)*xa(2,3)
      ZSI(3)=ZE(60)+CTION*DEN(IK)
      ZSI(4)=ZE(61)                                             
      ZSI(5)=ZE(62)                                             
      ZSI(6)=ZE(63)                                             
      ZSI(7)=ZE(64)                                             
      ZSI(8)=ZE(65)                                             
      ZSI(9)=ZE(66)                                             
      ZSI(10)=ZE(67)                                            
C                                                                       
C     SILICON RECOMB. COEFF (ALD.&PEQ, S & S)                          
C   Diel. from N & S + S & S. Romanik for S V -> S IV
C
      BDCORR=1.D0
      SUPR=1.D0
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
      ALSI(1)=0.                                                       
C
C      CALL DIELB(-.0219d0,.4364d0,.0684d0,-.0032d0,.1342d0,T4,ALDB)
C      CALL DIEL(1.1D-3,7.7D4,0.0d0,0.0d0,TE,ALD)
C      ALD=ALD+ALDB
C      ALSI(2)=5.9D-13/T4**.601+ALD*SUPR
C --- artificially reduce this to simulate the diffuse rad.field < 13.6 eV
C     alsi(2)=1.d-5*alsi(2)
      call totnah(11,te,alrec)
      alsi(2)=alrec
C                                                                       
      CHCONS=1.23d-9*t4**.24*(1.d0+3.17d0*dexp(.00418*t4))
      chtrsi=xh*chcons/del(ik) 
C      CALL DIELB(3.2163d0,-12.0571d0,16.2118d0,-.5886d0,.5613d0,T4,ALDB)
C      CALL DIEL(5.87D-3,9.63D4,.753d0,6.46d4,TE,ALD)
      ALD=ALD+ALDB
C      ALSI(3)=1.0D-12/T4**.786+ALD*SUPR+chtrsi
      call totnah(12,te,alrec)
      alsi(3)=alrec+chtrsi
C
      CHCONS=4.9d-10*(t4**(-.0874))*(1.d0-0.36d0*expfn(0.79d0*t4))
      CHTRSI=XH*CHCONS/DEL(IK)+
     &(9.5d-10*ab(2)*xa(2,2)/del(ik))*(t4**.75)
      CALL DIELB(.1203d0,-2.6900d0,19.1943d0,-.1479d0,.1118d0,T4,ALDB)
      CALL DIEL(.503d0,8.75D4,.188d0,4.71d4,TE,ALD)
      ALD=ALD+ALDB
      ALSI(4)=3.7D-12/T4**.693+ALD*SUPR+CHTRSI
C
      CHCONS=7.58d-9*(t4**.37)*(1.d0+1.06d0*expfn(4.09d0*t4))
      CHTRSI=XH*CHCONS/DEL(IK)+1.2d-9*ab(2)*xa(2,2)/del(ik)
C - Romanik, 1988 for Si V -> Si IV (No low temp. acc. to N & S)
      ald=(2.4d-11*expfn(6.d0/tryd)+8.68d-11*expfn(7.53d0/tryd)+
     &1.39d-10*expfn(9.04d0/tryd))/tryd**1.5
      call verferl(5.942d-11,0.3930d0,te,8.962d2,1.213d7,radrec)
      ALSI(5)=radrec+ALD*SUPR+CHTRSI
C
C Low temp. probably important (N & S), but not available.
      CALL DIEL(8.86d-3,1.14d6,0.0d0,0.0d0,TE,ALD)
      ALSI(6)=1.2D-11/T4**.735+ALD*SUPR
C
      ALDB=0.0D0
C
      CALL DIEL(1.68D-2,4.85D5,1.80D0,1.03D6,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALSI(7)=2.11D-11/T4**.716+ALD                                   
C
      CALL DIEL(2.49D-2,4.15D5,1.88D0,1.91D6,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALSI(8)=3.00D-11/T4**.702+ALD                                   
C
      CALL DIEL(3.13D-2,3.66D5,2.01D0,2.11D6,TE,ALD)                   
      ALD=ALD*BDCORR*SUPR                                       
      ALSI(9)=4.30D-11/T4**.688+ALD                                   
C
C      CALL DIEL(4.25D-1,3.63D5,1.22D0,2.14D6,TE,ALD)                   
C      ALD=ALD*BDCORR*SUPR
C      ALSI(10)=5.80D-11/T4**.703+ALD                                   
      call totnah(9,te,alrec)
      alsi(10)=alrec
C
C - Should really include Romanik, 1988 for Si XI -> Si X (but elaborate)
      CALL DIEL(6.18D-2,3.88D5,0.303D0,1.12D6,TE,ALD)                   
      ALD=ALD*BDCORR*SUPR
      ALSI(11)=7.70D-11/T4**.714+ALD                                   
C                                                                       
C     ***************************************************************   
C     *****                                                             
C     SODIUM
C     *****                                                             
C     ***************************************************************   
      cona(1)=arf(16.d0,-1.d0,0.2d0,-13.5d0,5.1d0,tev)
      cona(1)=cona(1)+arf(63.9d0,-27.d0,33.d0,-80.d0,34.d0,tev)
      cona(1)=cona(1)+arau(11,11,tev)
      cona(2)=arf(40.d0,-28.d0,19.4d0,-60.d0,47.3d0,tev)
      cona(2)=cona(2)+arf(19.2d0,-5.3d0,2.8d0,-21.2d0,80.1d0,tev)
      cona(3)=arf(50.1d0,-20.2d0,14.8d0,-41.7d0,71.7d0,tev)
      cona(3)=cona(3)+arf(18.8d0,-5.d0,2.8d0,-19.6d0,102.d0,tev)
      cona(4)=arf(43.3d0,-16.3d0,10.7d0,-33.4d0,141.d0,tev)
      cona(4)=cona(4)+arf(18.4d0,-4.7d0,2.8d0,-18.d0,126.d0,tev)
      cona(5)=arf(35.1d0,-12.4d0,7.2d0,-25.1d0,138.d0,tev)
      cona(5)=cona(5)+arf(18.d0,-4.3d0,2.8d0,-16.3d0,151.d0,tev)
      cona(6)=arf(25.5d0,-8.5d0,4.2d0,-16.8d0,172.d0,tev)
      cona(6)=cona(6)+arf(17.6d0,-4.d0,2.8d0,-14.7d0,186.d0,tev)
      cona(7)=arf(14.5d0,-4.6d0,1.8d0,-8.5d0,208.d0,tev)
      cona(7)=cona(7)+arf(17.2d0,-3.7d0,2.8d0,-13.1d0,224.d0,tev)
      cona(8)=arf(16.8d0,-3.4d0,2.8d0,-11.4d0,264.d0,tev)
      cona(8)=cona(8)+arf(21.7d0,-6.5d0,4.1d0,-18.d0,1328.d0,tev)
      cona(9)=arf(10.d0,-3.d0,1.4d0,-6.9d0,300.d0,tev)
      cona(9)=cona(9)+arf(21.7d0,-6.5d0,4.1d0,-18.d0,1366.d0,tev)
      cona(9)=cona(9)+arauli(11,tev)
C
      do iziz=1,9
      ZNA(iziz)=ZE(67+iziz)
      enddo
C --- for charge transfer ionization we assume that Na is in the 3s ground 
C     state. (Croft & Dickinson, J.Phys.B, 29, 57 [1996])
      t4c=2.d0
      if(t4.gt.0.1d0.and.t4.lt.2.d0) t4c=t4
      CTION=2.61d-12*(t4c**4.67d0)*(expfn(2.02d0/t4c))*(AB(1)-XH)
      ZNA(1)=ZE(68)+CTION*DEN(IK)                               
C
      ALNA(1)=0.0d0
C
      BDCORR=1.D0
      SUPR=1.D0
      ALDB=0.0D0
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
C
      CALL DIEL(2.07d-3,4.566D5,2.999D-2,1.932d6,TE,ALD)    
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      call verferl(5.641d-12,0.1749d0,te,3.077d2,2.617d6,radrec)
      ALNA(2)=radrec+ALD
C --- artificially reduce this to simulate the diffuse rad.field < 13.6 eV
C     alna(2)=1.d-5*alna(2)
C                                                                       
      CALL DIEL(2.237D-3,3.819D5,.1536D0,3.944D5,TE,ALD)                  
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALNA(3)=8.775d-13/T4**.7464+ALD                      
C
      CALL DIEL(4.529D-3,3.259D5,.3923D0,4.918D5,TE,ALD)            
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      CHCONS=1.33d-9*(t4**1.15)*(1.d0+1.2d0*expfn(.32d0*t4))
      CHTRNA=XH*CHCONS/DEL(IK)
      ALNA(4)=3.399D-12/T4**.7054+ALD+CHTRNA
C
      CALL DIEL(6.571D-3,2.711D5,.9028D0,5.476D5,TE,ALD)   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      CHCONS=1.01d-10*(t4**1.34)*(1.d0+10.05d0*expfn(6.41d0*t4))
      CHTRNA=XH*CHCONS/DEL(IK)
      ALNA(5)=7.849D-12/T4**.6952+ALD+CHTRNA
C
      CALL DIEL(2.087D-2,3.6D5,.3705D0,7.315D5,TE,ALD)        
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALNA(6)=1.447D-11/T4**.6814+ALD                   
C
C      CALL DIEL(2.976D-2,3.463D5,1.175D0,8.552d5,TE,ALD) 
C      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
C      ALNA(7)=2.191D-11/T4**.6875+ALD                 
      call totnah(6,te,alrec)
      alna(7)=alrec
C
      CALL DIEL(3.54d-2,3.097D5,0.322D0,8.129D5,TE,ALD)            
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALNA(8)=3.253D-11/T4**.7075+ALD                             
C
      CALL DIEL(7.352d-3,1.883d5,0.2842D0,9.716d5,TE,ALD)       
      ALD=ALD*BDCORR*SUPR                                       
      ALNA(9)=4.03D-11/T4**.7873+ALD                
C
      CALL DIEL(2.52D-1,1.40D7,0.3095D0,2.18D6,TE,ALD)      
      ALD=ALD*BDCORR*SUPR                                       
      call verferl(3.873d-10,0.6295d0,te,7.000d2,2.989d7,radrec)
      ALNA(10)=radrec+ALD           
C
C     ***************************************************************   
C     *****                                                             
C     MAGNESIUM
C     *****                                                             
C     ***************************************************************   
      comg(1)=arf(18.d0,-1.d0,0.6d0,-4.d0,7.6d0,tev)
      comg(1)=comg(1)+arf(37.7d0,-30.d0,24.8d0,-62.d0,54.d0,tev)
      comg(1)=comg(1)+arf(17.6d0,-5.2d0,3.3d0,-19.d0,92.2d0,tev)
      comg(1)=comg(1)+arau(12,12,tev)
      comg(2)=arf(9.d0,-3.6d0,0.3d0,-5.4d0,15.d0,tev)
      comg(2)=comg(2)+arf(37.7d0,-30.d0,24.8d0,-62.d0,65.d0,tev)
      comg(2)=comg(2)+arf(17.6d0,-5.2d0,3.3d0,-19.d0,104.5d0,tev)
      comg(2)=comg(2)+arau(12,11,tev)
      comg(3)=arf(55.5d0,-24.1d0,18.7d0,-65.d0,80.1d0,tev)
      comg(3)=comg(3)+arf(19.3d0,-5.6d0,2.8d0,-20.5d0,119.d0,tev)
      comg(4)=arf(50.1d0,-20.2d0,14.2d0,-41.7d0,109.d0,tev)
      comg(4)=comg(4)+arf(19.d0,-5.3d0,2.8d0,-19.d0,144.d0,tev)
      comg(5)=arf(43.3d0,-16.3d0,10.3d0,-33.4d0,141.d0,tev)
      comg(5)=comg(5)+arf(18.6d0,-4.9d0,2.8d0,-17.5d0,172.d0,tev)
      comg(6)=arf(35.1d0,-12.4d0,6.9d0,-25.1d0,187.d0,tev)
      comg(6)=comg(6)+arf(18.2d0,-4.6d0,2.8d0,-16.d0,201.d0,tev)
      comg(7)=arf(25.5d0,-8.5d0,4.1d0,-16.8d0,225.d0,tev)
      comg(7)=comg(7)+arf(18.d0,-4.3d0,2.8d0,-14.5d0,241.d0,tev)
      comg(8)=arf(14.5d0,-4.6d0,1.8d0,-8.5d0,266.d0,tev)
      comg(8)=comg(8)+arf(17.5d0,-4.d0,2.8d0,-13.d0,283.d0,tev)
      comg(9)=arf(17.1d0,-3.6d0,2.7d0,-11.5d0,328.d0,tev)
      comg(9)=comg(9)+arf(22.d0,-6.7d0,4.1d0,-18.d0,1611.d0,tev)
      comg(10)=arf(10.d0,-3.d0,1.4d0,-6.8d0,367.d0,tev)
      comg(10)=comg(10)+arf(22.d0,-6.7d0,4.1d0,-18.d0,1653.d0,tev)
      comg(10)=comg(10)+arauli(12,tev)
C                                                                       
      do iziz=1,10
      ZMG(iziz)=ZE(76+iziz)
      enddo
C  Kingdon & Ferland ('96)
      CTION=9.76d-12*(t4**3.14d0)*(AB(1)-XH)*(1.D0+55.54d0*
     &EXPFN(1.12d0*T4))
      zmg(1)=zmg(1)+den(ik)*ction
C
      ALMG(1)=0.                                                       
C
      BDCORR=1.D0
      SUPR=1.D0
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
      CALL DIEL(4.49D-4,5.01D4,2.1D-2,2.81D4,TE,ALD)                
      CALL DIELB(1.2044D0,-4.6836D0,7.662D0,-.593D0,1.626D0,T4,ALDB)
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALMG(2)=1.4D-13/T4**.855+ALD                             
C --- artificially reduce this to simulate the diffuse rad.field < 13.6 eV
C     almg(2)=1.d-5*almg(2)
C                                                                       
C - Romanik, 1988 for Mg III -> Mg II (No low temp. acc. to N & S)
      ALDB=0.0D0
      ald=(5.02d-13*expfn(2.79d0/tryd)+4.17d-11*expfn(3.88d0/tryd))
     &/tryd**1.5
C     CALL DIEL(1.95D-3,6.06D5,0.074D0,1.44D6,TE,ALD)                  
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      call verferl(1.920d-11,0.3028d0,te,4.849d2,5.890d6,radrec)
      ALMG(3)=radrec+ALD                             
C
      t4c=3.d0
      if(t4.le.0.1d0) t4c=0.1d0
      if(t4.gt.0.1d0.and.t4.lt.3.d0) t4c=t4
      CHCONS=6.49d-9*(t4c**.53)*(1.d0+2.82d0*expfn(7.63d0*t4c))
      CHTRH=XH*CHCONS/DEL(IK)+(7.5d-10*ab(2)*xa(2,2)*(1.d0+
     &                      1.25*expfn(5.8d0*t4c)))/del(ik)
      CALL DIEL(5.12D-3,4.69D5,0.323D0,7.55D5,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALMG(4)=3.50D-12/T4**.734+ALD+CHTRH
C
      CHCONS=6.36d-9*(t4c**.55)*(1.d0+3.86d0*expfn(5.19d0*t4c))
      CHTRH=XH*CHCONS/DEL(IK)+(2.2d-9*ab(2)*xa(2,2)*(t4c**.33)*
     &                 (1.d0+1.25*expfn(5.8d0*t4c)))/del(ik)
      CALL DIEL(7.74D-3,3.74D5,.636D0,7.88D5,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALMG(5)=7.7D-12/T4**.718+ALD+CHTRH
C
      CALL DIEL(1.17D-2,3.28D5,0.807D0,1.02D6,TE,ALD)                    
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALMG(6)=1.40D-11/T4**.716+ALD                                   
C
      CALL DIEL(3.69D-2,4.80D5,0.351D0,9.73D5,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALMG(7)=2.30D-11/T4**.695+ALD                                   
C
C      CALL DIEL(3.63D-2,3.88D5,0.548D0,7.38D5,TE,ALD)                   
C      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
C      ALMG(8)=3.20D-11/T4**.691+ALD                                   
      call totnah(7,te,alrec)
      almg(8)=alrec
C
C - Romanik, 1988 for Mg IX -> Mg VIII
C - Badnell has done this too....
      ald=(5.59d-12*expfn(0.292d0/tryd)+3.89d-12*expfn(0.502d0/tryd)
     &   +2.38d-11*expfn(0.742d0/tryd)+2.08d-11*expfn(1.31d0/tryd)
     &   +6.96d-10*expfn(2.38d0/tryd)+1.19d-10*expfn(10.3d0/tryd)
     &   +5.98d-10*expfn(13.d0/tryd))
     &   /tryd**1.5
C     CALL DIEL(4.15D-2,3.39D5,0.233D0,3.82D5,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALMG(9)=4.6D-11/T4**.711+ALD                                   
C
C - Romanik, 1988 for Mg X -> Mg IX
      ald=(9.40d-12*expfn(0.217d0/tryd)+6.36d-12*expfn(0.48d0/tryd)
     &   +9.22d-12*expfn(0.738d0/tryd)+1.16d-11*expfn(0.999d0/tryd)
     &   +2.11d-10*expfn(1.43d0/tryd)+3.65d-11*expfn(5.81d0/tryd)
     &   +6.72d-11*expfn(10.5d0/tryd)+5.07d-11*expfn(13.8d0/tryd))
     &   /tryd**1.5
C     CALL DIEL(8.86D-3,2.11D5,0.318D0,1.54D6,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR 
      ALMG(10)=5.8D-11/T4**.804+ALD                                   
C
C - Romanik, 1988 for Mg XI -> Mg X
      ald=(6.30d-10*expfn(74.4d0/tryd)+2.42d-9*expfn(92.9d0/tryd))
     &/tryd**1.5
C     CALL DIEL(2.52D-1,1.40D7,0.315D0,2.64D6,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      call verferl(4.284d-10,0.6287d0,te,8.748d2,3.586d7,radrec)
      ALMG(11)=radrec+ALD                                   
C                                                                       
C     ***************************************************************   
C     *****                                                             
C     ALUMINIUM
C     *****                                                             
C     ***************************************************************   
      coal(1)=arf(47.d0,-26.d0,0.6d0,-39.d0,6.d0,tev)
      coal(1)=coal(1)+arf(55.1d0,-37.2d0,1.4d0,-41.d0,10.6d0,tev)
      coal(1)=coal(1)+arau(13,13,tev)
      coal(2)=arf(17.d0,-6.d0,1.d0,-8.d0,18.8d0,tev)
      coal(2)=coal(2)+arf(31.3d0,-22.7d0,21.d0,-44.1d0,90.d0,tev)
      coal(2)=coal(2)+arf(12.1d0,-3.5d0,3.3d0,-13.1d0,131.d0,tev)
      coal(2)=coal(2)+arau(13,12,tev)
      coal(3)=arf(6.3d0,-2.4d0,0.5d0,-4.1d0,28.4d0,tev)
      coal(3)=coal(3)+arf(31.3d0,-22.7d0,21.d0,-44.1d0,103.d0,tev)
      coal(3)=coal(3)+arf(12.1d0,-3.5d0,3.3d0,-13.1d0,145.6d0,tev)
      coal(3)=coal(3)+arau(13,11,tev)
      coal(4)=arf(72.d0,-24.1d0,18.d0,-50.d0,120.d0,tev)
      coal(4)=coal(4)+arf(19.5d0,-5.9d0,2.8d0,-19.8d0,164.d0,tev)
      coal(5)=arf(60.8d0,-20.2d0,13.7d0,-41.7d0,154.d0,tev)
      coal(5)=coal(5)+arf(19.1d0,-5.5d0,2.8d0,-18.4d0,194.d0,tev)
      coal(6)=arf(49.5d0,-16.3d0,9.9d0,-33.4d0,190.d0,tev)
      coal(6)=coal(6)+arf(18.9d0,-5.2d0,2.8d0,-17.1d0,225.d0,tev)
      coal(7)=arf(38.3d0,-12.4d0,6.7d0,-25.1d0,241.d0,tev)
      coal(7)=coal(7)+arf(18.4d0,-4.8d0,2.8d0,-15.7d0,258.d0,tev)
      coal(8)=arf(27.d0,-8.5d0,3.9d0,-16.8d0,285.d0,tev)
      coal(8)=coal(8)+arf(18.2d0,-4.5d0,2.8d0,-14.3d0,302.d0,tev)
      coal(9)=arf(14.d0,-4.6d0,1.7d0,-8.5d0,330.d0,tev)
      coal(9)=coal(9)+arf(17.9d0,-4.1d0,2.8d0,-13.d0,350.d0,tev)
      coal(10)=arf(17.4d0,-3.8d0,2.7d0,-11.6d0,399.d0,tev)
      coal(10)=coal(10)+arf(22.2d0,-6.8d0,4.1d0,-18.d0,1921.d0,tev)
      coal(11)=arf(9.9d0,-3.d0,1.4d0,-6.7d0,442.d0,tev)
      coal(11)=coal(11)+arf(22.2d0,-6.8d0,4.1d0,-18.d0,1967.d0,tev)
      coal(11)=coal(11)+arauli(13,tev)
C                                                                       
      do iziz=1,11
      ZAL(iziz)=ZE(86+iziz)
      enddo
C
      ALAL(1)=0.                                                       
C
      BDCORR=1.D0
      SUPR=1.D0
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
      CALL DIEL(2.547D-3,7.311D4,0.4018D0,5.794D4,TE,ALD)                
      CALL DIELB(0.0219D0,-0.4528D0,2.5427D0,-.1678D0,0.2276D0,T4,
     &ALDB)
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAL(2)=3.98D-13/T4**.8019+ALD                             
C --- artificially reduce this to simulate the diffuse rad.field < 13.6 eV
C     alal(2)=1.d-5*alal(2)
C                                                                       
      CALL DIEL(1.503D-3,6.621D4,0.06283D0,3.638D4,TE,ALD)                  
      CALL DIELB(0.7086D0,-3.1083D0,7.0422D0,0.5998D0,.4194D0,T4,
     &ALDB)
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAL(3)=7.197D-13/T4**.7697+ALD                             
C
      CHCONS=7.11d-14*(t4c**4.12)*(1.d0+1.72d4*expfn(22.24d0*t4c))
      CHTRH=XH*CHCONS/DEL(IK)
      ALDB=0.0D0
      CALL DIEL(3.254D-3,7.977D5,0.1825D0,1.072D6,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      call verferl(3.753d-11,0.3585d0,te,6.848d2,9.035d6,radrec)
      ALAL(4)=radrec+ALD+CHTRH
C
      CHCONS=7.52d-10*(t4c**.77)*(1.d0+6.24d0*expfn(5.67d0*t4c))
      CHTRH=XH*CHCONS/DEL(IK)
      CALL DIEL(6.735D-3,7.312D5,5.683D-16,8.689D-13,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAL(5)=6.481D-12/T4**.7345+ALD+CHTRH
C
      CALL DIEL(1.14D-2,4.295D5,1.07D0,9.009D5,TE,ALD)                    
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAL(6)=1.275D-11/T4**.717+ALD                                   
C
      CALL DIEL(1.707D-2,3.689D5,1.232D0,1.396D6,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAL(7)=2.049D-11/T4**.709+ALD                                   
C
      CALL DIEL(3.398D-2,4.191D5,.8399D0,1.433D6,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAL(8)=3.145D-11/T4**.6915+ALD                                   
C
C      CALL DIEL(3.928D-2,3.753D5,0.8177D0,1.257D6,TE,ALD)                   
C      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
C      ALAL(9)=4.308D-11/T4**.697+ALD                                   
      call totnah(8,te,alrec)
      alal(9)=alrec
C
      CALL DIEL(5.064D-2,3.627D5,0.2657D0,6.541D5,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAL(10)=5.951D-11/T4**.7125+ALD                                   
C
      CALL DIEL(1.106D-2,2.301D5,0.672D0,2.46D6,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAL(11)=8.343D-11/T4**.8291+ALD                                   
C
      CALL DIEL(2.871D-1,1.622D7,0.3105D0,3.083D6,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      call verferl(4.881d-10,0.6326d0,te,9.941d2,4.085d7,radrec)
      ALAL(12)=radrec+ALD                                   
C                                                                       
C     ***************************************************************   
C     *****                                                             
C     ARGON
C     *****                                                             
C     ***************************************************************   
      coar(1)=arf(171.d0,-78.d0,3.8d0,-169.d0,15.8d0,tev)
      coar(1)=coar(1)+arf(48.7d0,-30.5d0,1.4d0,-39.7d0,29.2d0,tev)
      coar(2)=arf(147.d0,-97.4d0,3.2d0,-107.7d0,27.6d0,tev)
      coar(2)=coar(2)+arf(50.d0,-31.8d0,1.4d0,-40.d0,41.7d0,tev)
      coar(3)=arf(122.8d0,-81.4d0,2.6d0,-90.d0,40.9d0,tev)
      coar(3)=coar(3)+arf(51.3d0,-33.2d0,1.4d0,-40.2d0,55.5d0,tev)
      coar(3)=coar(3)+arau(18,16,tev)
      coar(4)=arf(98.7d0,-65.4d0,1.9d0,-72.3d0,59.7d0,tev)
      coar(4)=coar(4)+arf(52.5d0,-34.5d0,1.4d0,-40.5d0,70.4d0,tev)
      coar(4)=coar(4)+arau(18,15,tev)
      coar(5)=arf(74.5d0,-49.4d0,1.3d0,-54.6d0,75.2d0,tev)
      coar(5)=coar(5)+arf(53.8d0,-35.8d0,1.4d0,-40.7d0,87.6d0,tev)
      coar(5)=coar(5)+arau(18,14,tev)
      coar(6)=arf(50.4d0,-33.4d0,0.6d0,-36.9d0,91.2d0,tev)
      coar(6)=coar(6)+arf(55.1d0,-37.2d0,1.4d0,-41.d0,105.d0,tev)
      coar(6)=coar(6)+arau(18,13,tev)
      coar(7)=arf(19.8d0,-5.7d0,1.9d0,-11.9d0,125.d0,tev)
      coar(7)=coar(7)+arf(74.8d0,-27.d0,14.1d0,-58.6d0,373.d0,tev)
      coar(7)=coar(7)+arf(23.4d0,-8.3d0,3.3d0,-18.5d0,427.d0,tev)
      coar(7)=coar(7)+arau(18,12,tev)
      coar(8)=arf(9.d0,-2.7d0,0.8d0,-5.4d0,143.d0,tev)
      coar(8)=coar(8)+arf(74.8d0,-27.d0,14.1d0,-58.6d0,394.d0,tev)
      coar(8)=coar(8)+arf(23.4d0,-8.3d0,3.3d0,-18.5d0,453.1d0,tev)
      coar(8)=coar(8)+arau(18,11,tev)
      coar(9)=arf(72.d0,-24.1d0,11.9d0,-50.d0,423.d0,tev)
      coar(9)=coar(9)+arf(19.6d0,-7.3d0,2.8d0,-16.d0,498.d0,tev)
      coar(10)=arf(60.8d0,-20.2d0,9.3d0,-41.7d0,479.d0,tev)
      coar(10)=coar(10)+arf(19.4d0,-6.8d0,2.8d0,-15.3d0,545.d0,tev)
C                                                                       
      do iziz=1,10
      ZAR(iziz)=ZE(97+iziz)
      enddo
      CTION=1.1D-10*(expfn(3.04d0/tev))*ab(2)*xa(2,3)
      ZAR(2)=ZAR(2)+CTION*DEN(IK)
C
      ALAR(1)=0.                                                       
C
      BDCORR=1.D0
      SUPR=1.D0
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
      ALDB=0.0D0
C
      CALL DIEL(1.0D-3,3.20D5,5.0D-3,3.10D5,TE,ALD)                
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAR(2)=3.77D-13/T4**.651+ALD                             
C                                                                       
      t4c=3.d0
      if(t4.gt.0.1d0.and.t4.lt.3.d0) t4c=t4
      CHTRH=(1.3d-10*ab(2)*xa(2,2))/del(ik)
      CALL DIEL(1.10D-2,2.90D5,0.045D0,5.50D5,TE,ALD)                  
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAR(3)=1.95D-12/T4**.752+ALD+CHTRH
C
      CHCONS=4.57d-9*(t4c**.27)*(1.d0-0.18d0*expfn(1.57d0*t4c))
      CHTRH=XH*CHCONS/DEL(IK)
      CALL DIEL(3.40D-2,2.39D5,0.057D0,6.00D5,TE,ALD)              
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAR(4)=3.23D-12/T4**.869+ALD+CHTRH
C
      CHCONS=6.37d-9*(t4c**.85)*(1.d0+10.21d0*expfn(6.22d0*t4c))
      CHTRH=XH*CHCONS/DEL(IK)+(1.d-9*ab(2)*xa(2,2)*
     &                                (t4c**(-.3)))/del(ik)
      CALL DIEL(6.85D-2,2.56D5,0.087D0,3.81D5,TE,ALD)              
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAR(5)=6.03D-12/T4**.812+ALD+CHTRH
C
      CALL DIEL(9.00D-2,2.50D5,0.0769D0,3.30D5,TE,ALD)               
      ALD=(ALD*BDCORR+ALDB)*SUPR
      ALAR(6)=9.12D-12/T4**.811+ALD
c
      CALL DIEL(6.35D-2,2.10D5,0.14D0,2.15D5,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAR(7)=1.58D-11/T4**.793+ALD                                   
C
      CALL DIEL(2.60D-2,1.80D5,0.12D0,2.15D5,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALAR(8)=2.69D-11/T4**.744+ALD                                   
C
C - Romanik, 1988 for diel.rec. for Ar IX -> Ar VIII
      ald=(1.19d-10*expfn(11.7d0/tryd)+1.78d-10*expfn(14.5d0/tryd)+
     &2.55d-9*expfn(19.8d0/tryd))/tryd**1.5
C     CALL DIEL(1.70D-2,2.70D6,0.10D0,3.30D6,TE,ALD)                   
      ALD=ALD*BDCORR*SUPR                                       
      call verferl(2.862d-10,0.5621d0,te,7.002d2,4.885d7,radrec)
      ALAR(9)=radrec+ALD                                   
C
      CALL DIEL(2.10D-2,8.30D5,1.92D0,3.50D6,TE,ALD)                   
      ALD=ALD*BDCORR*SUPR
      ALAR(10)=4.90D-11/T4**.801+ALD                                   
C
      CALL DIEL(3.50D-2,6.95D5,1.66D0,3.60D6,TE,ALD)                   
      ALD=ALD*BDCORR*SUPR
      ALAR(11)=6.92D-11/T4**.811+ALD                                   
C                                                                       
C     ***************************************************************   
C     *****                                                             
C     CALCIUM
C     *****                                                             
C     ***************************************************************   
      coca(1)=arf(2.5d0,-2.5d0,8.d0,-5.5d0,6.1d0,tev)
      coca(1)=coca(1)+arf(74.3d0,-24.2d0,7.d0,-63.d0,28.d0,tev)
      coca(1)=coca(1)+arf(17.6d0,-3.8d0,1.9d0,-13.8d0,40.3d0,tev)
      coca(1)=coca(1)+arau(20,20,tev)
      coca(2)=arf(7.9d0,-2.d0,0.2d0,-6.d0,11.9d0,tev)
      coca(2)=coca(2)+arf(74.3d0,-24.2d0,7.d0,-68.d0,37.d0,tev)
      coca(2)=coca(2)+arf(17.6d0,-3.8d0,1.9d0,-13.8d0,45.2d0,tev)
      coca(2)=coca(2)+arau(20,19,tev)
      coca(3)=arf(74.3d0,-24.3d0,7.d0,-63.d0,51.2d0,tev)
      coca(3)=coca(3)+arf(17.6d0,-3.8d0,1.9d0,-13.8d0,70.1d0,tev)
      coca(4)=arf(55.8d0,-15.8d0,6.4d0,-44.5d0,67.3d0,tev)
      coca(4)=coca(4)+arf(16.2d0,-3.2d0,1.8d0,-11.6d0,86.4d0,tev)
      coca(5)=arf(47.1d0,-14.5d0,4.8d0,-35.5d0,84.5d0,tev)
      coca(5)=coca(5)+arf(18.9d0,-5.1d0,1.6d0,-13.2d0,104.d0,tev)
      coca(5)=coca(5)+arau(20,16,tev)
      coca(6)=arf(40.9d0,-13.6d0,3.4d0,-30.1d0,109.d0,tev)
      coca(6)=coca(6)+arf(20.4d0,-6.3d0,2.1d0,-13.8d0,123.d0,tev)
      coca(6)=coca(6)+arau(20,15,tev)
      coca(7)=arf(22.9d0,-7.4d0,2.8d0,-15.9d0,128.d0,tev)
      coca(7)=coca(7)+arf(21.9d0,-7.7d0,1.9d0,-14.9d0,144.d0,tev)
      coca(7)=coca(7)+arau(20,14,tev)
      coca(8)=arf(11.1d0,-3.4d0,1.3d0,-7.3d0,148.d0,tev)
      coca(8)=coca(8)+arf(22.7d0,-8.6d0,1.9d0,-15.5d0,165.d0,tev)
      coca(8)=coca(8)+arau(20,13,tev)
      coca(9)=arf(19.8d0,-5.7d0,1.8d0,-11.9d0,189.d0,tev)
      coca(9)=coca(9)+arf(76.1d0,-27.d0,12.8d0,-56.6d0,534.d0,tev)
      coca(9)=coca(9)+arf(23.5d0,-8.4d0,3.3d0,-17.8d0,593.1d0,tev)
      coca(9)=coca(9)+arau(20,12,tev)
      coca(10)=arf(9.d0,-2.6d0,0.9d0,-5.4d0,211.d0,tev)
      coca(10)=coca(10)+arf(76.1d0,-27.d0,12.8d0,-56.6d0,559.d0,tev)
      coca(10)=coca(10)+arf(23.5d0,-8.4d0,3.3d0,-17.8d0,623.7d0,tev)
      coca(10)=coca(10)+arau(20,11,tev)
C                                                                       
      do iziz=1,10
      ZCA(iziz)=ZE(107+iziz)
      enddo
C
      ALCA(1)=0.                                                       
C
      BDCORR=1.0D0
      SUPR=1.D0
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
      ALDB=0.0D0
C
      CALL DIEL(3.28D-4,3.46D4,0.0907D0,1.64D4,TE,ALD)                
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALCA(2)=1.12D-13/T4**.900+ALD                             
C --- artificially reduce this to simulate the diffuse rad.field < 13.6 eV
C     alca(2)=1.d-5*alca(2)
C                                                                       
      CALL DIEL(5.84D-2,3.85D5,0.11D0,2.45D5,TE,ALD)                  
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALCA(3)=6.78D-13/T4**.800+ALD                             
C --- artificially reduce this to simulate the diffuse rad.field < 13.6 eV
C     alca(3)=1.d-5*alca(3)
C
      CHCONS=3.17d-11*(t4c**2.12)*(1.d0+12.06d0*expfn(0.4d0*t4c))
      CHTRH=XH*CHCONS/DEL(IK)
      CALL DIEL(1.12D-1,4.08D5,0.0174D0,4.27D5,TE,ALD)              
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALCA(4)=3.96D-12/T4**.700+ALD+CHTRH
C
      CHCONS=2.68d-9*(t4c**.69)*(1.d0-0.68d0*expfn(4.47d0*t4c))
      CHTRH=XH*CHCONS/DEL(IK)
      CALL DIEL(1.32D-1,3.82D5,0.132D0,6.92D5,TE,ALD)              
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALCA(5)=7.08D-12/T4**.780+ALD+CHTRH
C
      CALL DIEL(1.33D-1,3.53D5,0.114D0,8.78D5,TE,ALD)               
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALCA(6)=1.07D-11/T4**.840+ALD                                   
C
      CALL DIEL(1.26D-1,3.19D5,0.162D0,7.43D5,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALCA(7)=1.80D-11/T4**.82+ALD                                   
C
      CALL DIEL(1.39D-1,3.22D5,0.0878D0,6.99D5,TE,ALD)                   
      ALD=(ALD*BDCORR+ALDB)*SUPR                                       
      ALCA(8)=2.40D-11/T4**.820+ALD                                   
C
      CALL DIEL(9.55D-2,2.47D5,0.263D0,4.43D5,TE,ALD)                   
      ALD=ALD*BDCORR*SUPR                                       
      ALCA(9)=3.76D-11/T4**.810+ALD                                   
C
      CALL DIEL(4.02D-2,2.29D5,0.0627D0,2.81D5,TE,ALD)                   
      ALD=ALD*BDCORR*SUPR
      ALCA(10)=5.04D-11/T4**.780+ALD                                   
C
C - Romanik, 1988 for diel.rec. for Ca XI -> Ca X
      ald=(6.37d-10*expfn(16.7d0/tryd)+1.21d-9*expfn(23.d0/tryd)+
     &3.92d-9*expfn(27.7d0/tryd))/tryd**1.5
C     CALL DIEL(4.19D-2,3.73D6,0.0616D0,5.84D6,TE,ALD)                   
      ALD=ALD*BDCORR*SUPR
      call verferl(5.273d-10,0.6281d0,te,5.329d2,3.188d7,radrec)
      ALCA(11)=radrec+ALD                                   
C
C  ***************************************************************   
Cc--- Hydrogen net ionization due to charge transfer reactions.         
Cc    (add HCHEXC to CIONH, the coll. ionization).                      
C !!! Be careful with this !!!                                          
c     HCHEXC=(AB(3)*(CHTRAO(2)*XA(2,15)+CHTRAO(3)*XA(2,16)+             
c    &CHTRAO(4)*XA(2,17)+CHTRAO(5)*XA(2,7))+AB(4)*(CHTRAC(3)*           
c    &XA(2,8)+CHTRAC(4)*XA(2,9)+CHTRAC(5)*XA(2,10))+AB(5)*              
c    &(CHTRAN(2)*XA(2,19)+CHTRAN(3)*XA(2,20)+CHTRAN(4)*XA(2,21)         
c    &+CHTRAN(5)*XA(2,22)))/XH                                          
c    &-(CTION*AB(3)*XA(2,14))/(DEL(IK)*(AB(1)-XH))                      
C  ***************************************************************   
C  *****                                                             
C     RECOMBINATION COOLING (CASE A) FOR HYDROGEN AND HELIUM            
C     RATES ARE ADJUSTED TO AGREE WITH OSTERBROCKS VALUES FOR           
C     T<2.E4*Z**2                                                       
C  *****                                                             
C  ***************************************************************   
      DO 925  K3=1,3                                                    
      re12=0.0d0
      re11=0.0d0
      Y=YI(K3)                                                          
      IF(Y.GT.0.309) GOTO 930                                           
C                                                                       
C     GOULD AND THAKUR (md)                                             
C                                                                       
      FJ1=1.202*Y+.542*Y**2.+1.037*Y**2.*DLOG(Y)-.438*Y**3.             
      FJ=FJ1-1.008*Y**3.*DLOG(Y)-.465*Y**4.+(.501*Y**4.)*DLOG(Y)        
      GOTO 948                                                          
C                                                                       
C     GOULD AND THAKUR (30)                                             
C                                                                       
  930 FJ=.5*(.735+DLOG(Y)+1./(3.*Y))                                    
  948 IF(Y.GT.100.) GOTO 927                                            
      IF(Y.GT.5.) RF1=1.-2./Y+6/Y**2.                                   
      IF(Y.LT.5.) RF1=Y*(1.-YEX1(K3))                                   
      RE12=1.468D-32*Y*TE**1.5*RF1                                      
      RE11=FJ                                                           
      GOTO 928                                                          
  927 RE11=FJ                                                           
  928 CONTINUE                                                          
      REA=1.615D-32*TE**1.5*Y*RE11                                      
      IF(K3.NE.1) GOTO 924                                              
      IF(OTSP(K3).LT.1.) REA=REA-RE12                                   
      RE(K3)=AB(1)*REA*(1.-XA(2,K3))                                    
      RE1(K3)=AB(1)*RE12*(1.-XA(2,K3))                                  
      RHE2B=2.*AB(2)*RE12*(1.-XA(2,3)-XA(2,2))                          
      GOTO 925                                                          
  924 IF(K3.NE.2) GOTO 942                                              
C                                                                       
C     CORRECT HE I FOR NONHYDROGENIC GROUND STATE (SEE GOULD AND TAKHUR)
C                                                                       
      RE(2)=AB(2)*XA(2,3)*(1.615D-32*TE**1.5*Y*FJ-RE12)                 
      RE1(2)=2.04D-27*AB(2)*XA(2,3)*DSQRT(TE)                           
      IF(OTSP(2).GT.1.) RE(2)=RE(2)+RE1(2)                              
      GOTO 925                                                          
 942  RE1(3)=AB(2)*RE12*(1.-XA(2,3)-XA(2,2))                            
      RE(3)=AB(2)*REA*(1.-XA(2,3)-XA(2,2))                              
      IF(OTSP(3).LT.1.) RE(3)=RE(3)-RE1(3)                              
  925 CONTINUE                                                          
C
      XHEIII=DMAX1(0.0d0,(1.D0-XA(2,2)-XA(2,3)))
C
C     ***************************************************************   
C     *****                                                             
C     FREE-FREE COOLING,(COX&TUCKER AP.J. 157:1157) FOR T>1+7           
C     AND SPITZER TABLE 3.3 FOR GAUNT FACTOR                            
C     *****                                                             
C     ***************************************************************   
      IF(TE.GT.1.D7) GOTO 8436                                          
      FF=0.                                                             
      DO 8437 KK=1,3                                                    
      ZZ=1.                                                             
      IF(KK.EQ.3) ZZ=4.                                                 
      GAUNT=-1.08+0.925*DLOG10(TE/ZZ)-0.085*(DLOG10(TE/ZZ))**2.         
      IF(KK.EQ.3) ABX=AB(2)*XHEIII
      IF(KK.EQ.2) ABX=AB(2)*XA(2,3)                                     
      IF(KK.EQ.1) ABX=AB(1)*(1.-XA(2,1))                                
8437  FF=FF+1.426D-27*ZZ*ABX*GAUNT*DSQRT(TE)                            
      GOTO 8438                                                         
8436  FF=1.426D-27*(1.23*AB(1)+4.*1.3587*AB(2))*DSQRT(TE)               
8438  CONTINUE                                                          
      COOLFR(1)=DEL(IK)*RE(1)
      COOLFR(2)=DEL(IK)*RE(2)                                           
      COOLFR(3)=DEL(IK)*RE(3)                                           
      COOLFR(4)=1.d-50
      COOLFR(5)=1.d-50
      COOLFR(6)=DEL(IK)*FF                                              
C     ***************************************************************   
C     *****                                                             
C     LINE COOLING RATES AND EMITTED FLUX TO INFINITY ( W ).            
C     *****                                                             
C     ***************************************************************   
      pdens=(1.d0-xa(2,1))*ab(1)*den(ik)                                
      he1dens=xa(2,2)*ab(2)*den(ik)
      he2dens=xa(2,3)*ab(2)*den(ik)
      he3dens=(1.d0-xa(2,2)-xa(2,3))*ab(2)*den(ik)                       
      EDENS=DEL(IK)*DEN(IK)                                             
      hdens=xa(2,1)*ab(1)*den(ik)
C
      abs12(ik)=1.044d-11/sqrtte
      ta12ih=colni(ik,1)*abs12(ik)
      ta12oh=colno(ik,1)*abs12(ik)
C He II
      abshe2(ik)=5.215d-12/sqrtte
      abshe3(ik)=8.362d-13/sqrtte
      abshe8(ik)=2.908d-14/sqrtte
C He I
      abshe4(ik)=9.889d-25/sqrtte
      abshe5(ik)=0.0d0
      abshe6(ik)=3.158d-18/sqrtte
      abshe7(ik)=1.142d-11/sqrtte
      abso1(ik)=2.757d-12/sqrtte
      ta12ie=colni(ik,3)*abshe2(ik)
      tao1i=colni(ik,16)*abso1(ik)
      tahe1i=colni(ik,2)*abshe4(ik)
      tahe2i=colni(ik,2)*abshe5(ik)
      tahe3i=colni(ik,2)*abshe6(ik)
      tahe4i=colni(ik,2)*abshe7(ik)
      ta12oe=colno(ik,3)*abshe2(ik)
      tao1o=colno(ik,16)*abso1(ik)
      tahe1o=colno(ik,2)*abshe4(ik)
      tahe2o=colno(ik,2)*abshe5(ik)
      tahe3o=colno(ik,2)*abshe6(ik)
      tahe4o=colno(ik,2)*abshe7(ik)
      tc12i=0.0d0
      tc13i=0.0d0
      tc14i=0.0d0
      tche1i=0.0d0
      tc12o=0.0d0
      tc13o=0.0d0
      tc14o=0.0d0
      tche1o=0.0d0
      dt12c=0.0d0
      dt13c=0.0d0
      dt14c=0.0d0
      dthe1c=0.0d0
       do 7961 j=1,ik
       if(j.ne.ik) then
        tc12i=tc12i+dxabs(1,ik,j)*taxa(j,11)
        tc13i=tc13i+dxabs(1,ik,j)*(taxa(j,12)+den(j)*(ab(4)*si(8,13)*
     &  xa(2,8)+ab(5)*si(20,13)*xa(2,20)+ab(6)*si(28,13)*xa(2,28)))
        tc14i=tc14i+dxabs(1,ik,j)*taxa(j,13)
        tche1i=tche1i+dxabs(1,ik,j)*taxa(j,5)
       else
        dt12c=dxabs(1,ik,j)*taxa(j,11)
        dt13c=dxabs(1,ik,j)*(taxa(j,12)+den(j)*(ab(4)*si(8,13)*
     &  xa(2,8)+ab(5)*si(20,13)*xa(2,20)+ab(6)*si(28,13)*xa(2,28)))
        dt14c=dxabs(1,ik,j)*taxa(j,13)
        dthe1c=dxabs(1,ik,j)*taxa(j,5)
       endif
 7961  continue
       do 7962 j=ik,imax
       if(j.ne.ik) then
        tc12o=tc12o+dxabs(3,ik,j)*taxa(j,11)
        tc13o=tc13o+dxabs(3,ik,j)*(taxa(j,12)+den(j)*(ab(4)*si(8,13)*
     &  xa(2,8)+ab(5)*si(20,13)*xa(2,20)+ab(6)*si(28,13)*xa(2,28)))
        tc14o=tc14o+dxabs(3,ik,j)*taxa(j,13)
        tche1o=tche1o+dxabs(3,ik,j)*taxa(j,5)
       else
        dt12c=dt12c+dxabs(3,ik,j)*taxa(j,11)
        dt13c=dt13c+dxabs(3,ik,j)*(taxa(j,12)+den(j)*(ab(4)*si(8,13)*
     &  xa(2,8)+ab(5)*si(20,13)*xa(2,20)+ab(6)*si(28,13)*xa(2,28)))
        dt14c=dt14c+dxabs(3,ik,j)*taxa(j,13)
        dthe1c=dthe1c+dxabs(3,ik,j)*taxa(j,5)
       endif
 7962  continue
C     ***************************************************************   
C                                                                       
C -- Collisional excitation of H --                                     
C                                                                       
      Z=AB(1)                                                   
      ta12i=ta12ih
      ta12o=ta12oh
      CALL HLOSS                                                        
     &(te,deltt,corat4)
      hy=z*f21
      hy2=z*f31
      hy3=z*f41
      hyco=z*f32
      hyco2=z*f42
      hyco3=z*f43
C -- Add Pickering lines (Case B) to H-alpha and H-beta (Osterbrock -89).
      hyco=hyco+(2.d-25*he3dens)/(den(ik)*t4**1.092)
      hyco2=hyco2+(6.99d-26*he3dens)/(den(ik)*t4**1.007)
      hytot=hy+hy2+hy3+hyco+hyco2+hyco3
      coolfr(7)=del(ik)*z*clymh
      coolfr(8)=del(ik)*z*cbalh
      coolfr(78)=del(ik)*z*cpaschh
      coolfr(9)=del(ik)*z*cresth
      hcool1=z*clymh
      hcool2=z*cbalh
      hcool3=z*cpaschh
      hcool4=z*cresth
      twop(ik)=(z*f21two)/12.57d0
C                                                                       
C -- Collisional excitation of He I and He II --                       
C                                                                       
      Z=AB(2)
      DENO3=DEN(IK)*AB(3)*XA(2,16)
      ta12i=ta12ie
      ta12o=ta12oe
      CALL HELOSS
     &(te,corat1,corat2,corat3,deltt,ite,teinn,corat4,corat5,
     &corat6,corat7,corat8,dt12c,dt13c,dt14c,dthe1c)
C -- Total line emission (for the cooling) --
      HEII=Z*(F21+F31+f21two)
C     HEIIBW=Z*(FYB+FO2)                                                
      HEII32=Z*F32                                                      
      heii21=(z*f21)/12.57d0
      coolfr(10)=del(ik)*z*clymhe
C     COOLFR(11)=DEL(IK)*HEIIBW
      coolfr(11)=1.d-50
      coolfr(12)=del(ik)*z*cbalhe
C                                                                       
      hecool=(coolfr(10)+coolfr(12))/del(ik)+(cresthe+he1cool)*z
C -- Observationally important line emission --
C -- Emission in He II 1640 (H-alpha)  --
      UVHELI(IK,1)=HEII32/12.57D0
C -- Emission in O III-lines (for ident., see DATA in HELOSS).          
      DO 9494 IBOW=1,8
C 9494 FOIII(IK,IBOW)=(FOIII(IK,IBOW)*Z)/12.57D0
 9494 FOIII(IK,IBOW)=1.d-50
C                                                                       
C Line center opacities (fi(X=0)=1, normalized by pisqrt later).
C The lines are H Ly-a, He II Ly-a, He II Ly-b, O III 303.799,
C N V 1238 and H Ly-b.
C For Fe-lines we put f-values that are uncertain equal to 0.20.
C
      dentei=den(ik)/dsqrt(te)
      if(x1n5(ik).le.1.d-15) x1n5(ik)=1.d-15
      ABSNV(IK)=2.2406D-11*AB(5)*XA(2,22)*DENTEI*x1n5(ik)
      ABS13(IK)=1.673D-12*AB(1)*XA(2,1)*DENTEI
      ABSEUV(IK,1)=3.745D-12*AB(3)*XA(2,4)*DENTEI
      ABSEUV(IK,2)=9.852D-12*AB(3)*XA(2,17)*DENTEI
      ABSEUV(IK,3)=1.086D-11*AB(3)*XA(2,16)*DENTEI
      ABSEUV(IK,4)=7.972D-12*AB(3)*XA(2,16)*DENTEI
      ABSEUV(IK,5)=1.74D-11*AB(3)*XA(2,17)*DENTEI
      ABSEUV(IK,6)=5.034D-12*AB(3)*XA(2,17)*DENTEI
      ABSEUV(IK,7)=2.757D-11*AB(3)*XA(2,7)*DENTEI
      ABSEUV(IK,8)=2.385D-11*AB(5)*XA(2,20)*DENTEI
      ABSEUV(IK,9)=1.046D-11*AB(3)*XA(2,16)*DENTEI
      ABSEUV(IK,10)=3.784D-11*AB(5)*XA(2,21)*DENTEI
      ABSEUV(IK,11)=9.783D-12*AB(3)*XA(2,17)*DENTEI
      ABSEUV(IK,12)=2.962D-11*AB(3)*XA(2,15)*DENTEI
      ABSEUV(IK,13)=1.034D-11*AB(3)*XA(2,16)*DENTEI
      ABSEUV(IK,14)=2.825D-12*AB(4)*XA(2,13)*DENTEI
      ABSEUV(IK,15)=3.364D-12*AB(4)*XA(2,13)*DENTEI
      ABSEUV(IK,16)=1.315D-12*AB(6)*XA(2,33)*DENTEI
      ABSEUV(IK,17)=1.774D-12*AB(6)*XA(2,34)*DENTEI
      ABSEUV(IK,18)=4.662D-12*AB(7)*XA(2,39)*DENTEI
      ABSEUV(IK,19)=9.318D-13*AB(7)*XA(2,40)*DENTEI
      ABSEUV(IK,20)=1.692D-11*AB(7)*XA(2,41)*DENTEI
      ABSEUV(IK,21)=6.264D-12*AB(7)*XA(2,39)*DENTEI
      ABSEUV(IK,22)=7.293D-12*AB(7)*XA(2,40)*DENTEI
      ABSEUV(IK,23)=5.828D-12*AB(7)*XA(2,39)*DENTEI
      ABSEUV(IK,24)=4.383D-11*AB(6)*XA(2,28)*DENTEI
      ABSEUV(IK,25)=6.101D-12*AB(7)*XA(2,42)*DENTEI
      ABSEUV(IK,26)=1.343D-10*AB(6)*XA(2,29)*DENTEI
      ABSEUV(IK,27)=1.60D-11*AB(8)*XA(2,49)*DENTEI
      ABSEUV(IK,28)=1.27D-11*AB(8)*XA(2,48)*DENTEI
      ABSEUV(IK,29)=1.77D-11*AB(8)*XA(2,49)*DENTEI
      ABSEUV(IK,30)=1.89D-11*AB(8)*XA(2,50)*DENTEI
      ABSEUV(IK,31)=1.46D-11*AB(8)*XA(2,50)*DENTEI
      ABSEUV(IK,32)=9.87D-11*AB(8)*XA(2,51)*DENTEI
      ABSEUV(IK,33)=2.93D-11*AB(8)*XA(2,52)*DENTEI
      ABSEUV(IK,34)=2.95D-11*AB(8)*XA(2,49)*DENTEI
      ABSEUV(IK,35)=2.50D-11*AB(8)*XA(2,52)*DENTEI
      ABSEUV(IK,36)=0.753D-11*AB(8)*XA(2,53)*DENTEI
      ABSEUV(IK,37)=1.51D-11*AB(8)*XA(2,50)*DENTEI
      ABSEUV(IK,38)=0.461D-11*AB(8)*XA(2,52)*DENTEI
      ABSEUV(IK,39)=0.292D-11*AB(8)*XA(2,53)*DENTEI
      ABSEUV(IK,40)=0.0352D-11*AB(8)*XA(2,54)*DENTEI
      ABSEUV(IK,41)=3.63D-11*AB(8)*XA(2,54)*DENTEI
      ABSEUV(IK,42)=1.15D-11*AB(8)*XA(2,54)*DENTEI
      ABSEUV(IK,43)=1.92D-11*AB(8)*XA(2,54)*DENTEI
      ABSEUV(IK,44)=3.86D-11*AB(8)*XA(2,54)*DENTEI
      ABSEUV(IK,45)=0.337D-11*AB(8)*XA(2,55)*DENTEI
      ABSEUV(IK,46)=4.47D-11*AB(8)*XA(2,55)*DENTEI
      ABSEUV(IK,47)=0.468D-11*AB(8)*XA(2,56)*DENTEI
      ABSEUV(IK,48)=1.07D-11*AB(8)*XA(2,55)*DENTEI
      ABSEUV(IK,49)=0.657D-11*AB(8)*XA(2,55)*DENTEI
      ABSEUV(IK,50)=1.26D-11*AB(8)*XA(2,56)*DENTEI
      ABSEUV(IK,51)=0.175D-11*AB(8)*XA(2,56)*DENTEI
      ABSEUV(IK,52)=0.298D-11*AB(8)*XA(2,55)*DENTEI
      ABSEUV(IK,53)=0.0911D-11*AB(8)*XA(2,56)*DENTEI
      ABSEUV(IK,54)=0.190D-11*AB(8)*XA(2,52)*DENTEI
      ABSEUV(IK,55)=0.235D-11*AB(8)*XA(2,53)*DENTEI
      ABSEUV(IK,56)=0.529D-11*AB(8)*XA(2,54)*DENTEI
      ABSEUV(IK,57)=0.264D-11*AB(8)*XA(2,55)*DENTEI
      ABSEUV(IK,58)=2.48D-12*AB(9)*XA(2,66)*DENTEI
      ABSEUV(IK,59)=2.97D-12*AB(9)*XA(2,63)*DENTEI
      ABSEUV(IK,60)=5.42D-12*AB(9)*XA(2,64)*DENTEI
      ABSEUV(IK,61)=1.61D-12*AB(9)*XA(2,66)*DENTEI
      ABSEUV(IK,62)=3.48D-12*AB(9)*XA(2,65)*DENTEI
      ABSEUV(IK,63)=1.74D-12*AB(9)*XA(2,66)*DENTEI
      ABSEUV(IK,64)=3.68D-12*AB(6)*XA(2,34)*DENTEI
      ABSEUV(IK,65)=4.43D-12*AB(9)*XA(2,34)*DENTEI
      ABSEUV(IK,66)=6.06D-12*AB(9)*XA(2,66)*DENTEI
      ABSEUV(IK,67)=1.10D-12*AB(9)*XA(2,66)*DENTEI
      ABSEUV(IK,68)=2.47D-12*AB(8)*XA(2,56)*DENTEI
      ABSEUV(IK,69)=4.48D-12*AB(9)*XA(2,65)*DENTEI
      ABSEUV(IK,70)=1.30D-12*AB(8)*XA(2,48)*DENTEI
      ABSEUV(IK,71)=3.39D-12*AB(11)*XA(2,83)*DENTEI
      ABSEUV(IK,72)=6.73D-12*AB(11)*XA(2,84)*DENTEI
      ABSEUV(IK,73)=8.50D-12*AB(11)*XA(2,81)*DENTEI
      ABSEUV(IK,74)=4.41D-12*AB(11)*XA(2,83)*DENTEI
      ABSEUV(IK,75)=1.26D-11*AB(11)*XA(2,85)*DENTEI
      ABSEUV(IK,76)=1.16D-11*AB(11)*XA(2,82)*DENTEI
      ABSEUV(IK,77)=4.74D-12*AB(11)*XA(2,83)*DENTEI
      ABSEUV(IK,78)=4.14D-12*AB(11)*XA(2,84)*DENTEI
      ABSEUV(IK,79)=8.41D-12*AB(11)*XA(2,86)*DENTEI
C
C---------------------------------------------------------------
C  Forbidden, semi-forbidden and resonance lines with E > 912 A.
C---------------------------------------------------------------
C                                                                       
C   ***** Carbon lines *****                                            
C
C     C I  609.8 mu, 370.4 mu, 9853 A, 4623 A, 2966-68 A, 8729 A
C
      Z=AB(4)*XA(2,12)                                                  
      call cforb(te,1,edens,z,columi(12),colum(12),redc(1),redc(2),
     &forc(1),forc(2),forc(3),forc(4),coolc1)
C                                                                       
C     C II 1037  (G & S)                             
C                                                                       
      Z=AB(4)*XA(2,13)                                                  
      OM=2.6D0*T4**0.23
      CALL RLOSS(1036.8D0,6.D0,2.D0,2.27d9,OM,edens,Z,TE,12.d0,
     &columi(13),colum(13),cc1037)
C                                                                       
C     C II 1335  (Lennon et al., Ap.J. -85)                             
C                                                                       
      OM=5.38D0*T4**0.153                                               
      CALL RLOSS(1335.3D0,6.D0,10.D0,2.85D8,OM,edens,Z,TE,12.d0,
     &columi(13),colum(13),C221)
      UVCARB(IK,1)=C221/12.57d0 
C
C     C II 157.7 mu, 2326 A, 454.6 mu, 353.4 mu
C
      call bforb(te,1,edens,hdens,z,columi(13),colum(13),redc(3),C222,
     &redc(4),redc(5),coolc)
      UVCARB(IK,2)=C222                                         
      coolc2=c221+coolc+cc1037
C
C     C III 977 A, 1907-09 A
C
      Z=AB(4)*XA(2,8)                                                   
      call beforb(te,1,edens,z,columi(8),colum(8),c1909,c1907,c977,
     &c1070,c2297,coolc3)
      UVCARB(IK,3)=C977
      UVCARB(IK,4)=C1907+C1909
C                                                                       
C     DIEL. REC. TO C III 1909 (STOREY 81),WITH SUPRESSION 0.17         
C     (Nussb. & Storey -83)                                             
C                                                                       
      Z=AB(4)*XA(2,9)                                                   
      if((ab(4)*xa(2,8)).ge.1.d-10) then
       pbesc=besc
      else
       pbesc=0.d0
      endif
      IF(DEN(IK).GT.1.D9) SUPR=0.17                                     
      EXXX=0.409D0/T4                                                   
      ALDB=0.                                                           
      IF(EXXX.GT.100.) GOTO 7283                                        
      ALDB=1.D-12*(1.879D0/T4+5.0085D0+4.553D0*T4-0.1453D0*T4           
     &*T4)*EXPFN(0.409D0/T4)/T4**1.5                                    
      IF(ALDB.LE.0.0D0) ALDB=0.0D0                                      
7283  CONTINUE                                                          
      REC31=Z*SUPR*1.041D-11*ALDB                                       
      UVCARB(IK,6)=(pbesc*REC31)/12.57D0                                        
C                                                                       
C     C IV 1549    (Cochrane & McWhirter '82; changed to Burke '92, including
C                   correction factor 0.8 by Savin et al. '95)
C                                                                       
C      OM=8.26D0*T4**0.06
      OM=6.65D0*T4**0.069
      CALL RLOSS(1549.D0,2.D0,6.D0,2.64D8,OM,edens,Z,TE,12.d0,
     &columi(9),colum(9),C241)             
      UVCARB(IK,5)=C241/12.57d0                                         
      coolc4=c241
C
      COCARB=coolc1+coolc2+coolc3+coolc4
      COOLFR(14)=XEL*C221                                           
      COOLFR(15)=XEL*C222                                           
      COOLFR(16)=XEL*C232                                           
      COOLFR(17)=XEL*C231                                           
      COOLFR(18)=XEL*C241                                           
C                                                                       
C   ***** Oxygen lines *****                                            
C                                                                       
C     O VI 1034    (Osterbrock & Wallace, Astr.Lett. -77)               
C                  (T4 > 6, Gaetz & Salpeter, Mendoza '83 Om=5.05)
C                                                                       
      Z=AB(3)*XO(6)                                                     
      OM=5.05D0*t4**0.01
      CALL RLOSS(1033.8D0,2.D0,6.D0,4.14D8,OM,edens,Z,TE,16.d0,
     &columi(6),colum(6),C161)            
      UVOXYG(IK,1)=C161/12.57D0                                         
      coolo6=c161
C                                                                       
C     O V 1218     (Mendoza, IAU 103 -83) (T4 > 8, Dufton et al.,
C                   A&A -78; A-value from Fleming et al. MNRAS '96)
C                                                                       
      Z=AB(3)*XO(5)                                                     
      OM=0.718D0*T4**(-0.168)                                           
      CALL RLOSS(1218.3D0,1.D0,9.D0,7.60D2,OM,edens,Z,TE,16.d0,
     &columi(5),colum(5),C151)            
      UVOXYG(IK,2)=C151/12.57D0                                         
      coolo5=c151
C
C     O IV  25.89 mu, 1401 A, 76.10 mu, 54.20 mu
C
      Z=AB(3)*XO(4)                                                     
      call bforb(te,3,edens,hdens,z,columi(17),colum(17),redo(1),C142,
     &redo(2),redo(3),coolo4)
      UVOXYG(IK,3)=C142
C                                                                       
C     O III 88.34 mu, 51.81 mu, 4959-5007 A, 2321 A, 1663 A, 4363 A 
C                                                                       
      Z=AB(3)*XO(3) 
      call cforb(te,3,edens,z,columi(16),colum(16),redo(4),redo(5),
     &foro(1),foro(2),C131,foro(3),coolo3)
      UVOXYG(IK,4)=C131
C
C     O II  3726-29 A, 2470 A, 7319-31 A
C                                                                       
      Z=AB(3)*XO(2) 
      call nforb(te,2,edens,z,columi(15),colum(15),foro(4),foro(5),
     &foro(6),coolo2)
      fro21=frr21
      fro41=frr41
C
C     O I  63.17 mu, 145.6 mu, 6300-64 A, 2958-72 A, 5577 A 
C
      Z=AB(3)*XO(1) 
      call oforb(te,1,edens,z,columi(15),colum(15),redo(6),redo(7),
     &foro(7),foro(8),foro(9),coolo1)
C
      COOX=coolo1+coolo2+coolo3+coolo4+coolo5+coolo6 
      COOLFR(19)=XEL*C161                                           
      COOLFR(20)=XEL*C151                                           
      COOLFR(21)=XEL*C142                                           
      COOLFR(22)=XEL*C131                                           
C                                                                       
C   ***** Nitrogen lines *****                                          
C                                                                       
C     N I   5199-5202 A, 3468 A, 10401-10410  A                       
c
      Z=AB(5)*XN(1) 
      call nforb(te,1,edens,z,columi(18),colum(18),forn(1),forn(2),
     &forn(3),cooln1)
C
C     N II  1085 A    (Gaetz & Salpeter)
C                                                                       
      Z=AB(5)*XN(2)                                                     
      OM=7.3D0*T4**0.061
      CALL RLOSS(1085.1D0,9.D0,15.D0,3.70D8,OM,edens,Z,TE,14.d0,
     &columi(19),colum(19),cooln2)              
C
C     N II  204 mu, 122 mu, 6548-83 A, 3063 A, 2143 A, 5755 A
C                                                                       
      call cforb(te,2,edens,z,columi(19),colum(19),redn(1),redn(2),
     &forn(4),forn(5),C321,forn(6),cooln)
      UVNITR(IK,1)=C321
C
      cooln2=cooln2+cooln
C                                                                       
C     N III  57.33 mu, 1749 A, 167.5 mu, 123.3 mu
C                                                                       
      Z=AB(5)*XN(3)                                                     
      call bforb(te,2,edens,hdens,z,columi(20),colum(20),redn(3),C331,
     &redn(4),redn(5),cooln)
      UVNITR(IK,2)=C331
C
C     N III  991 A (Gaetz & Salpeter, A-value from Brage et al. '95)
C                                                                       
      OM=4.92D0*T4**0.061                                               
      CALL RLOSS(991.D0,6.D0,10.D0,5.1D8,OM,edens,Z,TE,14.d0,
     &columi(20),colum(20),C332)              
      UVNITR(IK,3)=C332/12.57D0                                         
C
      cooln3=cooln+c332
C                                                                       
C     N IV 1486      (From isoelectronic fit by Keenan et al., Phys.Scripta
C                     34, 216, 1986; A-value from Fleming et al. '96)
C                                                                       
      Z=AB(5)*XN(4)                                                     
      OM=0.14035d0+0.51339d0*telog-0.080565*telog**2
      CALL RLOSS(1486.5D0,1.D0,9.D0,1.933D2,OM,edens,Z,TE,14.d0,
     &columi(21),colum(21),C341)            
      UVNITR(IK,4)=C341/12.57D0                                         
      cooln4=c341
C
C     DIEL. REC. TO N IV 1486 (STOREY 81),WITH SUPRESSION 0.20          
C     (Nussb. & Storey -83)                                             
C                                                                       
      Z=AB(5)*XN(5)                                                     
      if((ab(5)*xn(4)).ge.1.d-10) then
       pbesc=besc
      else
       pbesc=0.d0
      endif
      IF(DEN(IK).GT.1.D9) SUPR=0.20                                     
      EXXX=0.2665D0/T4                                                  
      ALDB=0.                                                           
      IF(EXXX.GT.100.) GOTO 8283                                        
      ALDB=1.D-12*(0.3327D0/T4-2.1374D0+16.0006D0*T4-1.3683D0*T4        
     &*T4)*EXPFN(0.2665D0/T4)/T4**1.5                                   
      IF(ALDB.LE.0.0D0) ALDB=0.0D0                                      
8283  CONTINUE                                                          
      REN41=Z*SUPR*1.337D-11*ALDB                                       
      UVNITR(IK,6)=(pbesc*REN41)/12.57D0                                        
C                                                                       
C     N V 1240       (Osterbrock & Wallace)                             
C                    (Mendoza '83; High-energy Cochrane & McWhirter '82)
C                                                                       
      OM=6.76D0*T4**0.025
      CALL RLOSSN(1240.1D0,2.D0,6.D0,3.40D8,OM,edens,Z,TE,14.d0,
     &columi(22),colum(22),C351,rlcool,IMAX)
      UVNITR(IK,5)=C351/12.57D0                                         
      cooln5=rlcool
C                                                                       
      CONI=cooln1+cooln2+cooln3+cooln4+cooln5
      COOLFR(23)=XEL*C321                                           
      COOLFR(24)=XEL*C331                                           
      COOLFR(25)=XEL*C332                                           
      COOLFR(26)=XEL*C341                                           
      COOLFR(27)=XEL*C351                                           
C
C  ***** Sulphur lines *****
C                                                                       
C     S I  25.25 mu, 56.31 mu, 10824-11309 A, 4509-4590 A, 7727 A 
C
      Z=AB(6)*XA(2,25)
      call oforb(te,3,edens,z,columi(25),colum(25),reds(1),reds(2),
     &fors(1),fors(2),fors(3),cools1)
C
C     S II 6716-31 A, 4069-76 A, 10287-370 A
C
      Z=AB(6)*XA(2,26)
      call nforb(te,4,edens,z,columi(26),colum(26),fors(4),fors(5),
     &fors(6),cools2)
      frs21=frr21
      frs41=frr41
C
C     S II 1256 A  (Ho & Henry, ApJ, 351, 701, '90) 
C                   New! Ramsbottom et al. '96 (see NFORB for ref.). 
C
      Z=AB(6)*XA(2,26)
C      if(t4.lt.4.0d0) then
C	OM=7.01d0*t4**.228
C      else
C        OM=7.01d0*T4**(-.185)
C      endif
      OM=6.78d0*t4**(-.192)
      CALL RLOSS(1256.D0,4.D0,12.D0,4.37d7,OM,edens,Z,TE,32.1d0,
     &columi(26),colum(26),coolsu)
      cools2=cools2+coolsu
C
C     S III  33.5 mu, 18.7 mu, 9069-9531 A, 3722 A, 1713-29 A, 6312 A 
C
      Z=AB(6)*XA(2,26)
      call cforb(te,5,edens,z,columi(26),colum(26),reds(3),reds(4),
     &fors(7),fors(8),fors(9),fors(10),cools)
C
C     S III  1197.5 A   (G & S)
C
      Z=AB(6)*XA(2,27)
      OM=2.17D0*T4**0.113
      CALL RLOSS(1197.6D0,9.D0,15.D0,7.00D7,OM,edens,Z,TE,32.1d0,
     &columi(27),colum(27),CSUL3)
      cools3=cools+csul3
C                                                                       
C     S IV  10.52 mu, 1411 A, 29.07 mu, 18.28 mu
C                                                                       
      Z=AB(6)*XA(2,28)                                                
      call bforb(te,4,edens,hdens,z,columi(28),colum(28),reds(5),CSUL4,
     &reds(6),reds(7),cools)
      CSUL4=12.57d0*CSUL4
C
C     S IV  1070 A   (G & S; Mendoza et al., J.Phys.B, 28, 3485, '95)
C
      OM=5.24D0*T4**0.028
      CALL RLOSS(1063.D0,6.D0,10.D0,1.69D8,OM,edens,Z,TE,32.1d0,
     &columi(28),colum(28),CSUL2)
      cools4=cools+csul2
C
C     S V 1194     (Mendoza '83 and Dufton et al. J.Phys.B '85.)
C
      Z=AB(6)*XA(2,29)
      OM=2.04d0*t4**(-.373)
      CALL RLOSS(1194.D0,1.D0,9.D0,4.2d4,OM,edens,Z,TE,32.1d0,
     &columi(29),colum(29),CSUL0)
      cools5=csul0
C
C     S VI 937.1     (Mendoza '83)
C
      Z=AB(6)*XA(2,30)
      OM=11.9d0
      CALL RLOSS(937.1D0,2.D0,6.D0,1.68D9,OM,edens,Z,TE,32.1d0,
     &columi(30),colum(30),CSUL1)
      cools6=csul1
C
      COSULF=cools1+cools2+cools3+cools4+cools5+cools6
      COOLFR(90)=XEL*CSUL1
      COOLFR(91)=XEL*CSUL2
      COOLFR(92)=XEL*CSUL3
      COOLFR(93)=XEL*CSUL4
C
C  ***** Neon lines *****
C
C     Ne III  15.6 mu, 36.0 mu, 3869-3968 A, 1815 A, 3342 A
C
      Z=AB(7)*XA(2,37)
      call oforb(te,2,edens,z,columi(37),colum(37),rede(1),rede(2),
     &fore(1),fore(2),fore(3),coole3)
C
C     Ne IV  2422-24 A, 1602 A, 4714-24 A 
C
      Z=AB(7)*XA(2,38)
      call nforb(te,3,edens,z,columi(38),colum(38),fore(4),fore(5),
     &fore(6),coole4)
C
C     Ne V  24.3 mu, 14.3 mu, 3346-3426 A, 1575 A, 1137-46 A, 2975 A
C
      Z=AB(7)*XA(2,39)
      call cforb(te,4,edens,z,columi(39),colum(39),rede(3),rede(4),
     &fore(7),fore(8),fore(9),fore(10),coole5)
C
C     Ne VI  7.66 mu, 993-1010 A, 22.8 mu, 15.5 mu
C                                                                       
      Z=AB(7)*XA(2,40)
      call bforb(te,6,edens,hdens,z,columi(40),colum(40),rede(5),
     &fore(11),rede(6),rede(7),coole6)
C
      CONEON=coole3+coole4+coole5+coole6
C
C  *****Silicon lines *****
C
C     Si II  34.8 mu, 2341 A, 92.3 mu, 57.0 mu
C                                                                       
      Z=AB(9)*XA(2,59)
      call bforb(te,5,edens,hdens,z,columi(59),colum(59),redsi(1),CSI1,
     &redsi(2),redsi(3),coolsi2)
      UVSI(IK,1)=CSI1
C
C     Si II 991.7     (G & S; Mendoza et al. '95)
C
      OM=0.584D0*(TE/1.78D4)**.159
      CALL RLOSS(991.7D0,6.D0,10.D0,6.92D8,OM,edens,Z,TE,28.09d0,
     &columi(59),colum(59),CSI2)
C
C     Si II 1194     (G & S; Mendoza et al. '95)
C
      OM=0.469D0*(TE/1.78D4)**.888
      CALL RLOSS(1194.D0,6.D0,6.D0,4.10D9,OM,edens,Z,TE,28.09d0,
     &columi(59),colum(59),CSI3)
C
C     Si II 1263     (G & S; Mendoza et al. '95)
C
      OM=1.73D0*(TE/1.78D4)**.439
      CALL RLOSS(1263.D0,6.D0,10.D0,2.96D9,OM,edens,Z,TE,28.09d0,
     &columi(59),colum(59),CSI4)
C
C     Si II 1308     (G & S; Mendoza et al. '95)
C
      OM=0.268D0*(TE/1.78D4)**.205
      CALL RLOSS(1308.D0,6.D0,2.D0,1.02D9,OM,edens,Z,TE,28.09d0,
     &columi(59),colum(59),CSI5)
C
C     Si II 1531     (G & S; Mendoza et al. '95)
C
      OM=0.452D0*(TE/1.78D4)**.248
      CALL RLOSS(1531.D0,6.D0,2.D0,1.12d9,OM,edens,Z,TE,28.09d0,
     &columi(59),colum(59),CSI6)
C
C     Si II 1814     (G & S; Mendoza et al. '95)
C
      OM=0.1D0*(TE/1.78D4)**.133
      CALL RLOSS(1814.D0,6.D0,10.D0,3.10D6,OM,edens,Z,TE,28.09d0,
     &columi(59),colum(59),CSI7)
C
      coolsi2=coolsi2+CSI2+CSI3+CSI4+CSI5+CSI6+CSI7
C
C Si III 1206.5  (Baluja et al., J.Phys.B '80, Becker & Butler, A&A 235,326,'90)
C
      Z=AB(9)*XA(2,60)
      OM=5.818D0*T4**.206
      CALL RLOSS(1206.5D0,1.D0,3.D0,2.57D9,OM,edens,Z,TE,28.09d0,
     &columi(60),colum(60),CSI8)
      UVSI(IK,2)=CSI8/12.57D0
C
C     Si III 1892  (Baluja et al., J.Phys.B 1981; A-value Ojha et al. 
C                   J.Phys.B '88)
C
      OM=5.428D0*T4**(-.374)
      CALL RLOSS(1892.D0,1.D0,9.D0,5.6D3,OM,edens,Z,TE,28.09d0,
     &columi(60),colum(60),CSI9)
      UVSI(IK,3)=CSI9/12.57D0
C
      coolsi3=CSI8+CSI9
C
C Si IV 1397 (Osterbrock & Wallace, Astrophys.Lett. 19,'77; Becker & Butler '90)
C
      Z=AB(9)*XA(2,61)
      OM=1.7D1*T4**.0085
      CALL RLOSS(1396.7D0,2.D0,6.D0,8.93D8,OM,edens,Z,TE,28.09d0,
     &columi(61),colum(61),CSI10)
      UVSI(IK,4)=CSI10/12.57D0
C
      coolsi4=CSI10
C
C     Si VI  1.96 mu     (Mendoza '83)
C
      Z=AB(9)*XA(2,63)
      OM=0.242d0
      CALL RLOSS(1.963d4,4.D0,2.D0,2.38d0,OM,edens,Z,TE,28.09d0,
     &columi(63),colum(63),coolsi6)
C
C     Si VIII
C
      Z=AB(9)*XA(2,65)
      call nforb(te,8,edens,z,columi(65),colum(65),forsi(1),forsi(2),
     &forsi(3),coolsi8)
C
      COSIL=coolsi2+coolsi3+coolsi4+coolsi6+coolsi8
C
C  *****Sodium lines *****
C
C     Na III  7.32 mu     (Mendoza '83)
C
      Z=AB(10)*XA(2,70)
      OM=0.3d0
      CALL RLOSS(7.317d4,4.D0,2.D0,4.59d-2,OM,edens,Z,TE,22.99d0,
     &columi(70),colum(70),coolna3)
C
C     Na IV  9.04 mu, 21.3 mu, 3242-3417 A, 1504-1529 A, 2805 A
C
      Z=AB(10)*XA(2,71)
      call oforb(te,4,edens,z,columi(71),colum(71),redna(1),
     &redna(2),forna(1),forna(2),forna(3),coolna4)
C
C     Na V  2068-70 A, 1366 A, 4012-25 A 
C
      Z=AB(10)*XA(2,72)
      call nforb(te,5,edens,z,columi(72),colum(72),forna(4),
     &forna(5),forna(6),coolna5)
C
      CONATR=coolna3+coolna4+coolna5
C
C  *****Magnesium lines *****
C
C     Mg II 2796-2803 A  (old [Mendoza '83], new Sigut & Pradhan '95,
C                         J.Phys.B 28, 4879)
C
      Z=AB(11)*XA(2,78)
C      OM=16.5d0*t4**.101
      OM=16.9d0*t4**.162
      CALL RLOSS(2.801d3,2.D0,4.D0,2.67d8,OM,edens,Z,TE,24.305d0,
     &columi(78),colum(78),coolmg2)
      formg(1)=coolmg2/12.57d0
C
C     Mg IV  4.49 mu     (Mendoza '83)
C
      Z=AB(11)*XA(2,80)
      OM=0.3d0
      CALL RLOSS(4.485d4,4.D0,2.D0,0.199d0,OM,edens,Z,TE,24.305d0,
     &columi(80),colum(80),coolmg4)
C
C     Mg V  5.61 mu, 13.5 mu, 2783-2993 A, 1294-1324 A, 2418 A
C
      Z=AB(11)*XA(2,81)
      call oforb(te,5,edens,z,columi(81),colum(81),redmg(1),
     &redmg(2),formg(2),formg(3),formg(4),coolmg5)
C
C     Mg VI
C
      Z=AB(11)*XA(2,82)
      call nforb(te,7,edens,z,columi(82),colum(82),formg(5),formg(6),
     &formg(7),coolmg6)
C
C     Mg VII  8.87 mu, 5.52 mu, 2510-2630 A, 1190 A, < 912 A, 2262 A
C
      Z=AB(11)*XA(2,83)
      call cforb(te,8,edens,z,columi(83),colum(83),redmg(3),
     &redmg(4),formg(8),formg(9),formg(10),formg(11),coolmg7)
C
C     Mg VIII  3.03 mu
C
      Z=AB(11)*XA(2,84)
      OM=0.744d0*t4**(-.566)
      CALL RLOSS(3.031d4,2.D0,4.D0,0.34d0,OM,edens,Z,TE,24.305d0,
     &columi(84),colum(84),coolmg8)
C
      COMAG=coolmg2+coolmg4+coolmg5+coolmg6+coolmg7+coolmg8
C
C  *****Aluminum lines *****
C
C     Al II  1671 A, 2660-69 A
C
      Z=AB(12)*XA(2,88)
      call beforb(te,3,edens,z,columi(88),colum(88),al2669,al2660,
     &al1671,al2087,al3901,coolal2)
      foralu(1)=al1671
      foralu(2)=al2660+al2669
C                                                                       
C     Al IX  2.05 mu (guessed isoelectronically)
C
      Z=AB(12)*XA(2,95)
      OM=0.65d0*t4**(-.594)
      CALL RLOSS(2.047d4,2.D0,4.D0,1.12d0,OM,edens,Z,TE,26.982d0,
     &columi(95),colum(95),coolal9)
C
      COALU=coolal2+coolal9
C
C  *****Argon lines *****
C
C     Ar II  6.99 mu     (Mendoza '83; Pelan & Berrington AAS, '95)
C
      Z=AB(13)*XA(2,99)
      if(telog.lt.4.6d0) then
        OM=2.845d0*t4**.085
      else
        OM=3.78d0*t4**(-.12)
      endif
      CALL RLOSS(6.985d4,4.D0,2.D0,5.27d-2,OM,edens,Z,TE,39.948d0,
     &columi(99),colum(99),coolar2)
C
C     Ar III  9.0 mu, 21.8 mu, 7136-8036 A, 3005-3109 A, 5192 A
C
      Z=AB(13)*XA(2,100)
      call oforb(te,6,edens,z,columi(100),colum(100),redar(1),
     &redar(2),forar(1),forar(2),forar(3),coolar3)
C
C     Ar IV  4711-40 A, 2854-68 A, 7171-7331 A 
C
      Z=AB(13)*XA(2,101)
      call nforb(te,6,edens,z,columi(101),colum(101),forar(4),
     &forar(5),forar(6),coolar4)
C
C     Ar V  13.1 mu, 7.9 mu, 6435-7006 A, 2691 A, < 912 A, 4626 A
C
      Z=AB(13)*XA(2,102)
      call cforb(te,6,edens,z,columi(102),colum(102),redar(3),
     &redar(4),forar(7),forar(8),forar(9),forar(10),coolar5)
C
      COARG=coolar2+coolar3+coolar4+coolar5
C
C  *****Calcium lines *****
C
C     Ca II  7291-7324, 3934-68, 8498-8542, 3706-37
C
      Z=AB(14)*XA(2,109)
      CALL KFORB(TE,1,edens,Z,columi(109),colum(109),ca7324,
     &ca7291,ca3968,ca3934,caneari,ca3711,coolca2)
      forca(1)=ca7324+ca7291
      forca(2)=ca3968+ca3934
C
C     Ca IV  3.21 mu     (Mendoza '83; Pelan & Berrington AAS, '95)
C
      Z=AB(14)*XA(2,111)
      if(telog.lt.5.2d0) then
        OM=1.00d0*t4**.395
      else
        OM=8.102d0*t4**(-.362)
      endif
      CALL RLOSS(3.207d4,4.D0,2.D0,0.545d0,OM,edens,Z,TE,40.08d0,
     &columi(111),colum(111),coolca4)
C
C     Ca V  4.16 mu, 11.5 mu, 5311-6429 A, 2281-2414 A, 3999 A
C
      Z=AB(14)*XA(2,112)
      call oforb(te,7,edens,z,columi(112),colum(112),redca(1),
     &redca(2),forca(3),forca(4),forca(5),coolca5)
C
C     Ca VII  6.14 mu, 4.09 mu, 4574-5622 A, 2112-2226 A, < 912 A, 3687 A
C
      Z=AB(14)*XA(2,114)
      call cforb(te,7,edens,z,columi(114),colum(114),redca(3),
     &redca(4),forca(6),forca(7),forca(8),forca(9),coolca7)
C
      COCAL=coolca2+coolca4+coolca5+coolca7
C
C-------------------------------------------------------- 
C --  EUV and X-ray lines 
C     (line cooling only modified by contnuum absorption,
C     no collisional deexcitation included.)
C-------------------------------------------------------- 
C                                                                       
      DO 5009 IW=1,14
 5009 CEX(IW)=0.d0                                                        
C
C -- Escape and continuum abs. probabilities --                         
       do 7783 ieuv=1,79
       iskipeu(ieuv)=0
       rcont(ieuv)=0.0d0
       kk=ieuv+14
       if(ieuv.ge.16) kk=ieuv+15
       abxa=ab(jqeu(kk))*xa(2,ioeu(kk))
       etv=eeveu(kk)/tev
       if(ieuv.eq.1) iskipeu(ieuv)=1
       if(abxa.le.1.d-10) iskipeu(ieuv)=1
       if(etv.gt.2.d1) iskipeu(ieuv)=1
       if(iskipeu(ieuv).eq.1) goto 7783
       taeuvi(ieuv)=0.0d0
       taeuvo(ieuv)=0.0d0
       tceuvi(ieuv)=0.0d0
       tceuvo(ieuv)=0.0d0
       dtauc(ieuv)=0.0d0
       if(abseuv(ik,ieuv).le.1.d-100) then
        cl(ieuv)=1.d50
       else
        cl(ieuv)=taxa(ik,leuv(ieuv))/abseuv(ik,ieuv)
        if(ieuv.eq.4) then
         cl(ieuv)=cl(ieuv)+(den(ik)*ab(4)*si(13,7)*xa(2,13))/
     &   abseuv(ik,ieuv)
        endif
       endif
        do 7784 j=1,ik
        taeuvi(ieuv)=taeuvi(ieuv)+dxabs(1,ik,j)*abseuv(j,ieuv)
        if(j.ne.ik) then
         tceuvi(ieuv)=tceuvi(ieuv)+dxabs(1,ik,j)*taxa(j,leuv(ieuv))
         if(ieuv.eq.4) then
          tceuvi(ieuv)=tceuvi(ieuv)+dxabs(1,ik,j)*den(j)*ab(4)*
     &    si(13,7)*xa(2,13)
         endif
        else
         dtauc(ieuv)=dtauc(ieuv)+dxabs(1,ik,j)*taxa(j,leuv(ieuv))
         if(ieuv.eq.4) then
          dtauc(ieuv)=dtauc(ieuv)+dxabs(1,ik,j)*den(j)*ab(4)*
     &    si(13,7)*xa(2,13)
         endif
        endif
 7784   continue
        do 7785 j=ik,imax
        taeuvo(ieuv)=taeuvo(ieuv)+dxabs(3,ik,j)*abseuv(j,ieuv)
        if(j.ne.ik) then
         tceuvo(ieuv)=tceuvo(ieuv)+dxabs(3,ik,j)*taxa(j,leuv(ieuv))
         if(ieuv.eq.4) then
          tceuvo(ieuv)=tceuvo(ieuv)+dxabs(3,ik,j)*den(j)*ab(4)*
     &    si(13,7)*xa(2,13)
         endif
        else
         dtauc(ieuv)=dtauc(ieuv)+dxabs(3,ik,j)*taxa(j,leuv(ieuv))
         if(ieuv.eq.4) then
          dtauc(ieuv)=dtauc(ieuv)+dxabs(3,ik,j)*den(j)*ab(4)*
     &    si(13,7)*xa(2,13)
         endif
        endif
 7785   continue
       if(taeuvi(ieuv).le.0.0d0) taeuvi(ieuv)=0.0d0
       if(tceuvi(ieuv).le.0.0d0) tceuvi(ieuv)=0.0d0
       if(taeuvo(ieuv).le.0.0d0) taeuvo(ieuv)=0.0d0
       if(tceuvo(ieuv).le.0.0d0) tceuvo(ieuv)=0.0d0
       if(dtauc(ieuv).le.0.0d0) dtauc(ieuv)=0.0d0
       if(cl(ieuv).le.0.0d0) cl(ieuv)=0.0d0
       call esc(taeuvi(ieuv),tceuvi(ieuv),cl(ieuv),pesci,plini,
     & pconti,dtauc(ieuv))
       call esc(taeuvo(ieuv),tceuvo(ieuv),cl(ieuv),pesco,plino,
     & pconto,dtauc(ieuv))
       pesc=0.5d0*(pesci+pesco)
       plin=0.5d0*(plini+plino)
       pcont=0.5d0*(pconti+pconto)
       eudown=pesc+pcont
C !!! Change this !!!
       rlofac(ieuv)=(pesc+pcont)/eudown
       rloesc(ieuv)=pesc/eudown
C !!! Change this !!!
       rcofac(ieuv)=pcont/eudown
 7783  continue
C
      DO 5001 KK=1,NLINE                                                
      RLO=0.0D0                                                         
      peuv=1.d0
      IF(KK.EQ.1) GOTO 5008                                             
      if(kk.ge.15.and.kk.ne.30) then
       if(kk.lt.30) ieuv=kk-14
       if(kk.gt.30) ieuv=kk-15
       if(iskipeu(ieuv).eq.1) goto 5008
       if(kk.ge.79.and.kk.le.84) goto 5008
C       if(kk.ge.79) goto 5008
      endif
      ETV=EEVEU(KK)/TEV 
      abxa=xa(2,ioeu(kk))*ab(jqeu(kk))
      if(abxa.le.1.d-10) goto 5008
      IF(ETV.GT.2.d1) GOTO 5008                                         
      TTM=1.D1**TMEU(KK)
      if(iityp(kk).eq.0) then
         OMEFF=ALQEU(KK)*(TE/TTM)**BEQEU(KK)
      else
	 OMEFF=ALQEU(KK)*EXPFN((BEQEU(KK)*TTM)/TE)
      endif
      RLO=8.63D-6*ABXA*OMEFF*EXPFN(ETV)/DSQRT(TE)
      RLO=1.602D-12*EEVEU(KK)*RLO 
c -- Absolute rates of continuum trapping --
      IF(KK.LE.14.OR.KK.EQ.30) GOTO 9348                                
      RLO=RLO*RLOFAC(ieuv)
      peuv=rloesc(ieuv)
      rcont(ieuv)=(rlo*den(ik)**2*xel*rcofac(ieuv))/(rlofac
     &(ieuv)*1.602d-12*eeveu(kk))
 9348 CONTINUE                                                          
 5008 CONTINUE
c -- Escaping line photons --
      IF(KK.LE.30) COOLFR(27+KK)=XEL*RLO*peuv
      IF(KK.GE.31.AND.KK.LE.41) COOLFR(48+KK)=XEL*RLO*peuv
      IF(KK.GE.42) COOLFR(70+KK)=XEL*RLO*peuv
C -- Cooling due to line photons. Overestimate since continuum trapped
C -- photons cool less than escaping photons. (Modify !!) --
      CEX(JQEU(KK))=CEX(JQEU(KK))+RLO 
 5001 CONTINUE
C                                                                       
c  -- Fe II-line cooling
      z=ab(8)*xa(2,44)
      if(z.gt.1.d-10) then
       CALL FEII(TE,EDENS,COLUMI(44),COLUM(44),FECOL2)
      else
       fecol2=0.0d0
      endif
      fecol2=z*fecol2
C
c  -- Fe III-line cooling
      z=ab(8)*xa(2,45)
      telg=dlog10(te)
      if(telg.lt.3.7d0) telg=3.7d0
      if(telg.gt.4.4d0) telg=4.4d0
      telg=telg-4.d0
      fecol3=1.d1**(-20.15d0+1.58d0*telg)
      fecol3=z*fecol3
C
C  -- Fe IV-line cooling
C
      Z=AB(8)*XA(2,46)
      if(z.gt.1.d-10) then
       IF(TE.GT.1.D3.AND.TE.LT.3.D4) THEN
        CALL FEIV(TE,EDENS,FECOL4)
        if(fecol4.le.0.0d0) fecol4=0.0d0
       ELSE
        FECOL4=0.0D0
       ENDIF
      else
       fecol4=0.0d0
      endif
      FECOL4=Z*FECOL4
C
C  -- Fe VII-line cooling
C
      Z=AB(8)*XA(2,49)
       DO 8205 LLL=1,17
 8205  FEADD(LLL)=0.0D0
      FECOL7=0.0D0
      IF(TE.GT.1.D3.and.z.gt.1.d-10) THEN
       CALL FEVII(TE,EDENS,FECOL7)
       if(fecol7.le.0.0d0) fecol7=0.0d0
       CALL FEVII_NEW(TE,EDENS,FEDUMMY)
      else
       fecol7=0.0d0
      ENDIF
       DO 8206 LLL=1,17
 8206  FEADD(LLL)=Z*FEADD(LLL)
C
C     Fe VII  5922 A  (=forbfe1)
C
      forbfe1=feadd(14)
      FECOL7=Z*FECOL7
C
C     Fe X  6374 A  (Corliss & Sugar, J.Phys.Chem. Ref.Data '82; 
C           Pelan & Berrington AAS, '95; A-value isoelec. guessed.)
C
      Z=AB(8)*XA(2,52)
      OM=6.38d0*t4**(-.276)
      CALL RLOSS(6.374d3,4.D0,2.D0,69.5d0,OM,edens,Z,TE,55.847d0,
     &columi(52),colum(52),fecol10)
C
C     Fe XIII  2579-3388 (=forbfe1_old)
C
      Z=AB(8)*XA(2,55)                                                  
      call cforb(te,9,edens,z,columi(55),colum(55),redfe1,redfe2,
     &forbfe1_old,forbfe2,forbfe3,forbfe4,fecol13)
C
      FECOOL=FECOL2+FECOL3+FECOL4+FECOL7+FECOL10+fecol13
      coiron=FECOOL
C     **************************************************************    
C     *****                                                             
C     COLL. IONIZATION COOLING                                          
C     *****                                                             
C     **************************************************************    
      COHE=8.704D-11*CHE*XA(2,3)*AB(2)                                  
      COHE=COHE+3.939D-11*CHEO*XA(2,2)*AB(2)                            
      COH=2.1787D-11*XA(2,1)*AB(1)*CIONH                                
      COLLC=1.D-11*AB(4)*(1.804D0*XA(2,12)*COC(1)+3.907D0*XA(2,13)      
     &*COC(2)+7.672D0*XA(2,8)*COC(3)+10.333D0*XA(2,9)*COC(4)+           
     &62.819D0*XA(2,10)*COC(5)+78.505D0*XA(2,11)*COC(6))                
      COLLN=1.D-11*AB(5)*(2.329D0*XA(2,18)*CON(1)+4.743D0*XA(2,19)      
     &*CON(2)+7.602D0*XA(2,20)*CON(3)+12.413D0*XA(2,21)*CON(4)+         
     &15.684D0*XA(2,22)*CON(5)+88.451D0*XA(2,23)*CON(6)+106.87D0        
     &*XA(2,24)*CON(7))                                                 
      COLLO=1.D-11*AB(3)*(2.182D0*XA(2,14)*COO(1)+5.626D0*XA(2,15)      
     &*COO(2)+8.802D0*XA(2,16)*COO(3)+12.403D0*XA(2,17)*COO(4)+         
     &18.249D0*XA(2,7)*COO(5)+22.13D0*XA(2,4)*COO(6)+118.45D0           
     &*XA(2,5)*COO(7)+139.61D0*XA(2,6)*COO(8))                          
      COLLSU=1.D-11*AB(6)*(1.66D0*XA(2,25)*COSU(1)+3.74D0*XA(2,26)*
     &COSU(2)+5.58D0*XA(2,27)*COSU(3)+7.58D0*XA(2,28)*COSU(4)+11.64D0
     &*XA(2,29)*COSU(5)+14.11D0*XA(2,30)*COSU(6)+44.86D0*XA(2,31)*
     &COSU(7)+52.60D0*XA(2,32)*COSU(8)+60.74D0*XA(2,33)*COSU(9)+
     &71.63D0*XA(2,34)*COSU(10))
      COLLNE=1.D-11*AB(7)*(3.455D0*XA(2,35)*CONE(1)+6.56D0*XA(2,36)
     &*CONE(2)+10.17D0*XA(2,37)*CONE(3)+15.56D0*XA(2,38)*CONE(4)+
     &20.22D0*XA(2,39)*CONE(5)+27.58D0*XA(2,40)*CONE(6)+33.40D0
     &*XA(2,41)*CONE(7)+38.31D0*XA(2,42)*CONE(8))
      COLLFE=1.D-11*AB(8)*(1.261D0*XA(2,43)*COFE(1)+2.589D0*XA(2,44)*
     &COFE(2)+4.911D0*XA(2,45)*COFE(3)+8.780D0*XA(2,46)*COFE(4)+
     &12.10D0*XA(2,47)*COFE(5)+16.02D0*XA(2,48)*COFE(6)+20.56D0*
     &XA(2,49)*COFE(7)+24.21D0*XA(2,50)*COFE(8)+37.65D0*XA(2,51)*
     &COFE(9)+41.99D0*XA(2,52)*COFE(10)+46.53D0*XA(2,53)*COFE(11)+
     &53.03D0*XA(2,54)*COFE(12)+57.84D0*XA(2,55)*COFE(13)+62.81D0*
     &XA(2,56)*COFE(14)+73.22D0*XA(2,57)*COFE(15))
      COLLSI=1.D-11*AB(9)*(1.306d0*xa(2,58)*cosi(1)+2.62d0*xa(2,59)*
     &cosi(2)+5.37d0*xa(2,60)*cosi(3)+7.23d0*xa(2,61)*cosi(4)+
     &26.72d0*xa(2,62)*cosi(5)+32.86d0*xa(2,63)*cosi(6)+42.38d0*
     &xa(2,64)*cosi(7)+48.57d0*xa(2,65)*cosi(8)+56.25d0*xa(2,66)*
     &cosi(9)+64.31d0*xa(2,67)*cosi(10))
      COLLNA=0.0d0
	do iicoll=1,9
	collna=collna+1.6022d-12*ab(10)*el(67+iicoll)*
     &         xa(2,67+iicoll)*cona(iicoll)
	enddo
      COLLMG=0.0d0
	do iicoll=1,10
	collmg=collmg+1.6022d-12*ab(11)*el(76+iicoll)*
     &         xa(2,76+iicoll)*comg(iicoll)
	enddo
      COLLAL=0.0d0
	do iicoll=1,11
	collal=collal+1.6022d-12*ab(12)*el(86+iicoll)*
     &	       xa(2,86+iicoll)*coal(iicoll)
	enddo
      COLLAR=0.0d0
	do iicoll=1,10
	collar=collar+1.6022d-12*ab(13)*el(97+iicoll)*
     &	       xa(2,97+iicoll)*coar(iicoll)
	enddo
      COLLCA=0.0d0
	do iicoll=1,10
	collca=collca+1.6022d-12*ab(14)*el(107+iicoll)*
     &	       xa(2,107+iicoll)*coca(iicoll)
	enddo
      COOLFR(73)=DEL(IK)*COH
      COOLFR(74)=DEL(IK)*COHE
      COOLFR(75)=DEL(IK)*COLLC
      COOLFR(76)=DEL(IK)*COLLN
      COOLFR(77)=DEL(IK)*COLLO
      COOLFR(109)=DEL(IK)*COLLSU
      COOLFR(110)=DEL(IK)*COLLNE
      COOLFR(111)=DEL(IK)*COLLFE
      COOLFR(112)=DEL(IK)*FECOOL
C     **************************************************************    
C     *****                                                             
C     PHOTOIONIZATION HEATING                                           
C     *****                                                             
C     **************************************************************    
 9124 FORMAT(1X,'COOX2',4(1X,E10.4))                                    
      DO 7836 I=1,117
 7836 GEA(I)=GE(I)/(DEN(IK)*CONS1)                                      
      ph=0.0d0
      pheo=0.0d0
      phei=0.0d0
      if(qxh(ik,2,1).gt.1.d-10) PH=AB(1)*qxh(ik,2,1)*GEA(1)
      if(qxhe(ik,2,1).gt.1.d-9) PHEO=AB(2)*qxhe(ik,2,1)*GEA(2)
      if(qxhe(ik,2,levhe1+1).gt.1.d-9) 
     &                       PHEI=AB(2)*qxhe(ik,2,levhe1+1)*GEA(3)
      POXT=0.                                                           
      DO 620 L=1,4                                                      
  620 PO(L)=AB(3)*XA(2,L+3)*GEA(L+3)                                     
      DO 621 L=5,8                                                      
  621 PO(L)=AB(3)*XA(2,L+9)*GEA(L+9)                                     
      DO 630 L=1,4                                                      
  630 PC(L)=AB(4)*XA(2,L+7)*GEA(L+7)                                     
      DO 631 L=1,2                                                      
  631 PC(L+4)=AB(4)*XA(2,L+11)*GEA(L+11)                                 
      DO 640 L=18,24                                                    
  640 PN(L-17)=AB(5)*XA(2,L)*GEA(L)                                      
      DO 943 L=25,34                                                    
  943 PSU(L-24)=AB(6)*XA(2,L)*GEA(L)                                     
      DO 9746 L=35,42                                                   
 9746 PNE(L-34)=AB(7)*XA(2,L)*GEA(L)                                     
      DO 642 L=58,67
  642 PSI(L-57)=AB(9)*XA(2,L)*GEA(L)                                 
      DO 7452 L=43,57                                               
 7452 PF(L-42)=AB(8)*XA(2,L)*GEA(L)                                  
      do l=68,76
	pona(l-67)=ab(10)*xa(2,l)*gea(l)
      enddo
      do l=77,86
	pomg(l-76)=ab(11)*xa(2,l)*gea(l)
      enddo
      do l=87,97
	poal(l-86)=ab(12)*xa(2,l)*gea(l)
      enddo
      do l=98,107
	poar(l-97)=ab(13)*xa(2,l)*gea(l)
      enddo
      do l=108,117
	poca(l-107)=ab(14)*xa(2,l)*gea(l)
      enddo
C     **************************************************************    
C     *****                                                             
C     TOTAL COOLING AND HEATING RATES                                   
C     *****                                                             
C     **************************************************************    
      HE=hecool
      COCARB=COCARB+CEX(4)                                                  
      CONI=CONI+CEX(5)                                                  
      COOX=COOX+CEX(3)                                                  
      COSULF=COSULF+CEX(6)                                            
      CONEON=CONEON+CEX(7) 
      COIRON=COIRON+CEX(8)                                             
      COSIL=COSIL+CEX(9)                                             
      CONATR=CONATR+CEX(10)
      COMAG=COMAG+CEX(11)
      COALU=COALU+CEX(12)
      COARG=COARG+CEX(13)
      COCAL=COCAL+CEX(14)
C     WRITE(6,9728)CEX(4),CEX(5),CEX(3),CEX(6),CEX(8)                   
 9728 FORMAT(1X,'CL',7E10.4)                                            
 8306 continue
C      abratd=(ab(1)*(1.-xa(2,1))+ab(2)*(1.-xa(2,2))+del(ik))/del(ik)
C      dustcl=1.4d-32*(te**1.5)*fudfed*abratd
      dustcl=0.d0
      COOL=RE(1)+RE(2)+RE(3)+FF+HE+COHE+COOX+COCARB+CONI+COSULF+
     &CONEON+hcool1+hcool2+hcool3+hcool4+COH+COLLC+COLLN+COLLO+
     &COLLSU+COLLNE+COIRON+COLLFE+COLLSI+COSIL+CONATR+COMAG+COALU+
     &COARG+COCAL+dustcl
C     DO 7652 KK=1,142                                               
C7652 COOLFR(KK)=COOLFR(KK)/(DEL(IK)*COOL)                              
C     IF(ICOM.EQ.1) GOTO 1921                                           
      POX=0.0D0                                                         
      DO 633 L=1,8                                                      
  633 POX=POX+PO(L)                                                     
      PCA=0.0D0                                                         
      DO 634 L=1,6                                                      
  634 PCA=PCA+PC(L)                                                     
      PNA=0.                                                            
      DO 645 L=1,7                                                      
  645 PNA=PNA+PN(L)                                                     
      PSULF=0.                                                          
      DO 6461 L=1,10                                                    
 6461 PSULF=PSULF+PSU(L)                                               
      PNEON=0.                                                          
      DO 6542 L=1,8                                                     
 6542 PNEON=PNEON+PNE(L)                                               
      PFE=0.
      DO 6543 L=1,15
 6543 PFE=PFE+PF(L)
      PSIL=0.
      DO 6544 L=1,10
 6544 PSIL=PSIL+PSI(L)
      PNAA=0.
      do l=1,9
	pnaa=pnaa+pona(l)
      enddo
      PMG=0.
      do l=1,10
	pmg=pmg+pomg(l)
      enddo
      PAL=0.
      do l=1,11
	pal=pal+poal(l)
      enddo
      PAR=0.
      do l=1,10
	par=par+poar(l)
      enddo
      PCAA=0.
      do l=1,10
	pcaa=pcaa+poca(l)
      enddo
C  -- Inverse Compton cooling --                                        
      CO=0.0D0                                                          
C                                                                       
      HEAT=CO+PH+PHEO+PHEI+POX+PCA+PNA+PFE+PSULF+PNEON+PSIL+PNAA+
     &PMG+PAL+PAR+PCAA
c     IF(IGAMM.EQ.1) HEAT=HEAT+1.072D-25*GAHE/((DEN(IK)*CONS1)/1.D9)    
c    &+CR*1.56D-24/((DEN(IK)*CONS1)/1.6D-3)                             
 1921 RAD=(HEAT-COOL*DEL(IK))                                           
      RADR=RAD                                                          
      HYR=HY                                                            
      IF(ITEX.eq.0.OR.TDAY.LT.1.25D-2) GOTO 1239                     
c     WRITE(6,7365)RE(1),RE(2),RE(3),FF,HEII,HEI,COHE,COOX,COCA,CONI,   
c    &CONEON,COSULF,COIRON
 7365 FORMAT(1X,'CO',15E10.3)                                           
c     WRITE(6,7366)CO,PH,PHEO,PHEI,POX,PCA,PNA,PFE,PSULF,PNEON,PSI   
 7366 FORMAT(1X,'HEAT',12E12.4)                                         
9456    format('x',5e11.4)                                              
9457    format('ge',5e11.4)                                             
c     WRITE(6,7362)XEL,HY,HYCO,CO,HEAT,COOL,RAD                         
 7362 FORMAT(1X,'RAD.',7E10.3)                                          
C ----COOLANTS                                                          
c     WRITE(6,2942)RE(1),RE(2),RE(3)                                    
 2942 FORMAT(1X,'REC.COOL,H,He',3E12.4)                                 
c     WRITE(6,2943)FF,HY,HYCO,HEII,HEI                                  
 2943 FORMAT(1X,'FF,HY,HYCO,HE2,HE1',5E12.4)                            
c     WRITE(6,2944)C221,C222,C232,C231,C241                             
 2944 FORMAT(1X,'CARBON COOL',6E12.4)                                   
c     WRITE(6,2946)C161,C151,C142,C131                                  
 2946 FORMAT(1X,'OXYG.COOL',5E12.4)                                     
c     WRITE(6,2947)C321,C331,C332,C341,C351                             
 2947 FORMAT(1X,'NITR.COOL',6E12.4)                                     
c     WRITE(6,2948)CEX(3),CEX(4),CEX(5)                                 
 2948 FORMAT(1X,'UV,X-ray:O,C,N',3E12.4)                                
c     WRITE(6,2949)C311,C322,C111,C121,C131                             
 2949 FORMAT(1X,'FORB.COOL',5E12.4)                                     
c     WRITE(6,2951)COH,COHE,COCA,CONI,COOX,CONE,COSULF,COIRON,COOL 
 2951 FORMAT(1X,'COH,COHE,C,N,O,COOL',9E12.4)                           
 1239 CONTINUE                                                          
C
      RETURN                                                            
      END                                                               
C*********************************************************************
C*********************************************************************
      SUBROUTINE RLOSS                                                  
     &(WL,G1,G2,A,OM,EDENS,Z,TE,AMASS,COLTOTI,COLTOT,RL) 
C***************************************************************        
C  This subroutine calculates the local heat loss in lines              
C  using a two-level approximation. As for all rates in RAD,            
C  the cooling rates are divided by Xe*DEN**2.                          
C***************************************************************        
      IMPLICIT REAL*8(A-H,O-Z)                                          
      parameter (md=75)
C
      COMMON/RLBESC/BESC
C                                                                       
      DATA ELCH/1.6022D-12/,H/6.626D-27/,C/2.998D10/                    
C                                                                       
      rl=0.0d0
      besc=0.d0
      TEV=TE/1.1605D4                                                   
      EEV=(1.D8*C*H)/(WL*ELCH)                                          
      EX=EEV/TEV                                                        
      if(z.gt.1.d-10.and.EX.lt.3.d1) then
       C21=(8.629D-6*OM)/(G2*DSQRT(TE))                                
       C12=(C21*G2)/(G1*DEXP(EX))                                      
       elab=(elch*z)/edens
       m=0
 9981  continue
       m=m+1
       if(m.eq.1) tcorr=1.d0
       if(m.ne.1) tcorr=(1.d0-(x2x1*g1)/g2)/(1.d0+x2x1)
       call escacf(eev,a,a,g1,g2,te,amass,tcorr,coltoti,coltot,p21)
       x2x1=(c12*edens)/(c21*edens+a*p21)
       if(m.le.2.and.x2x1.gt.1.d-2) goto 9981
       x2=x2x1/(1.d0+x2x1)
       rl=elab*eev*x2*a*p21
       besc=p21
      endif
C                                                                       
      RETURN                                                            
      END                                                               
C*********************************************************************
C*********************************************************************
      SUBROUTINE RLOSSN                                                  
     &(WL,G1,G2,A,OM,EDENS,Z,TE,AMASS,COLTOTI,COLTOT,RL,rlcool,
     &IMAX) 
C***************************************************************        
C  This subroutine calculates the local heat loss in lines              
C  using a two-level approximation, including photoexcitation.
C  As for all rates in RAD, the cooling rates are divided by 
C  Xe*DEN**2.                          
C  Escape probability is also calculated.
C***************************************************************        
      IMPLICIT REAL*8(A-H,O-Z)                                          
      parameter (md=75)
C
      COMMON/RLBESC/BESC
      COMMON/PHEXC/SINT(300)                                               
      COMMON/IND/I
      COMMON/RADIE/R(md)                                                
      COMMON/TEM/TEMPA(md)                                               
      COMMON/PHY/DEN(md)                                                
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)       
      COMMON/ABSHN/ABSNV(md)
      COMMON/CONSTS/PI,PISQRT,CLIGHT
      COMMON/NVLEV/x1n5(md),p21in(md),p21out(md)
C                                                                       
      DIMENSION DED(md),AA(md),SINTB(300)
C
      DATA ELCH/1.6022D-12/,H/6.626D-27/,C/2.998D10/                    
C                                                                       
      rl=0.0d0
      rlcool=0.0d0
      besc=0.d0
      TEV=TE/1.1605D4                                                   
      EEV=(1.D8*C*H)/(WL*ELCH)                                          
      EX=EEV/TEV                                                        
      p21in(i)=0.5d0
      p21out(i)=0.5d0
      if(z.gt.1.d-10.and.EX.lt.3.d1) then
       C21=(8.629D-6*OM)/(G2*DSQRT(TE))                                
       C12=(C21*G2)/(G1*DEXP(EX))                                      
       elab=(elch*z)/edens
C --- Photoexcitation rate --
C --- Assume 15 Doppler widths for the current shell.                   
       DED(I)=4.286D-7*EEV*DSQRT(TE)                                     
       EEMAX=1.5D1*DED(I)+EEV                                            
       AA(I)=(H*A)/(4.D0*PI*ELCH*DED(I))                       
C -- N must be odd !
       N=201                                                              
       IF(I.EQ.1) GOTO 4000                                              
         DO 4001 IJ=1,I-1                                                  
         DED(IJ)=4.286D-7*EEV*DSQRT(TEMPA(IJ))                              
 4001    AA(IJ)=(H*A)/(4.D0*PI*ELCH*DED(IJ))                     
 4000  CONTINUE                                                          
       if(i.ne.imax) then
         DO IJ=I+1,imax                                                  
         DED(IJ)=4.286D-7*EEV*DSQRT(TEMPA(IJ))                              
         AA(IJ)=(H*A)/(4.D0*PI*ELCH*DED(IJ))                     
	 ENDDO
       endif
       DEE=(EEMAX-EEV)/DBLE(N-1)                                         
         DO 4002 M=1,N                                                     
         EE=EEV+DEE*DBLE(M-1)                                              
         SUMTAU=0.0D0                                                      
         SUMTAU2=0.0D0                                                      
         IF(I.EQ.1) GOTO 4003                                              
           DO 4004 IJ=1,I-1                                                  
           XXX=(EE-EEV)/DED(IJ)                                              
           CALL VOIGT(AA(IJ),XXX,HVOI)                                       
           SUMTAU=SUMTAU+(ABSNV(IJ)*HVOI*1.D15*(R(IJ+1)-R(IJ)))
     &     /PISQRT
 4004      CONTINUE
 4003    CONTINUE                                                          
	 if(i.ne.imax) then
           DO IJ=I+1,imax
           XXX=(EE-EEV)/DED(IJ)
           CALL VOIGT(AA(IJ),XXX,HVOI)
           SUMTAU2=SUMTAU2+(ABSNV(IJ)*HVOI*1.D15*(R(IJ+1)-R(IJ)))
     &     /PISQRT
	   ENDDO
         endif
         XXX=(EE-EEV)/DED(I)
         CALL VOIGT(AA(I),XXX,HVOI)
         DTAU=(1.D15*(R(I+1)-R(I))*HVOI*ABSNV(I))/PISQRT
	 if(dtau.lt.1.d-8) then
	   dddexp=dtau
         else
	   dddexp=1.d0-expfn(dtau)
         endif
         SINTB(M)=2.D0*expfn(SUMTAU2)*dddexp
 4002    SINT(M)=2.D0*expfn(SUMTAU)*dddexp
       CALL SIMPA(N,DEE,Q)
	 do m=1,n
	 sint(m)=sintb(m)
	 enddo
       CALL SIMPA(N,DEE,Q2)
       PHEX18=(4.D0*PI*FL(I,1)*Q)/(Z*DEN(I)*ELCH*EEV*1.D15*
     & (R(I+1)-R(I)))
       PPP=0.5d0/(1.d15*ded(i)*absnv(i)*(r(i+1)-r(i)))
       p21out(i)=ppp*q2
       p21in(i)=ppp*q
       p21=p21out(i)+p21in(i)
C
C      call escacf(eev,a,a,g1,g2,te,amass,tcorr,coltoti,coltot,p21)
       x2x1=(c12*edens+phex18)/(c21*edens+a*p21+phex18*(g1/g2))
       x2=x2x1/(1.d0+x2x1)
       x1=1.d0-x2
       rl=elab*eev*x2*a*p21
       rlcool=elab*eev*(c12*x1-c21*x2)
C -- Correct the absorption coeff. for stim.emission (used next time
C    step.)
       x1n5(i)=(1.d0-(x2x1*g1)/g2)/(1.d0+x2x1)
       besc=p21
      endif
C                                                                       
      RETURN                                                            
      END                                                               
C*********************************************************************
C************************************************************           
      SUBROUTINE DIEL(A,T0,B,T1,TE,ALD)                                 
      IMPLICIT REAL*8(A-H,O-Z)                                          
C     ***************************************************************   
C     ******                                                            
C     DIELECTRIC RECOMBINATION RATE ACCORDING TO FORMULA IN ALD. AND    
C     PEQ.                                                              
C     ******                                                            
C     ***************************************************************   
C     DIELECTRIC RECOMBINATION RATE                                     
      EX=T0/TE                                                          
      ALD=0.                                                            
      IF(EX.LT.100)ALD=A*EXPFN(T0/TE)*(1.+B*EXPFN(T1/TE))/              
     &(TE*DSQRT(TE))                                                    
      RETURN                                                            
      END                                                               
C************************************************************           
C************************************************************           
      SUBROUTINE DIELB(A,B,C,D,F,T4,ALDB)                               
      IMPLICIT REAL*8(A-H,O-Z)                                          
C     ***************************************************************   
C     ******                                                            
C     DIELECTRIC RECOMBINATION RATE ACCORDING TO FORMULA IN NUSSBAUMER  
C     AND STOREY (A&A 126,75 (1983))                                    
C     ******                                                            
C     ***************************************************************   
C     DIELECTRIC RECOMBINATION RATE                                     
      EX=F/T4                                                           
      ALDB=0.                                                           
      IF(T4.GT.6.) GOTO 100                                             
      IF(EX.GT.100.) GOTO 100                                           
      ALDB=1.D-12*(A/T4+B+C*T4+D*T4*T4)*EXPFN(F/T4)/T4**1.5             
 100  CONTINUE                                                          
      RETURN                                                            
      END                                                               
C************************************************************           
C************************************************************           
      SUBROUTINE BADN1(TE,ALDNE7)
C************************************************************           
C Fit to Badnell's dielectronic rec. calcs. for recombining 
C Be-like Ne VII for 1E5 < T < 1E6 K. (J.Phys.B 20 2081, 1987).
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
      DIMENSION T(6),ANE7(6)
      DATA T/5.,5.2,5.4,5.6,5.8,6./                                     
      DATA ANE7/-10.366,-10.281,-10.3,-10.407,-10.575,-10.785/
C
      TLOG=DLOG10(TE)                                                   
      IF(TLOG.GE.T(6)) then                                
       qte=(t(6)/te)**1.5
       aldne7=1.d1**ane7(6)*qte
      else
       DO 100 I=1,5                                                      
       IF(TLOG.GE.T(I).AND.TLOG.LT.T(I+1)) GOTO 101                      
       GOTO 100                                                          
  101  QQ=(TLOG-T(I))/(T(I+1)-T(I))                                      
       ALDNE7=ANE7(I)+QQ*(ANE7(I+1)-ANE7(I))
  100  CONTINUE                                                          
       ALDNE7=1.D1**ALDNE7
      endif
C                                                                       
      RETURN                                                            
      END                                                               
C************************************************************           
C************************************************************           
      SUBROUTINE BADN2(TE,ALDC3,ALDN4,ALDO5)                            
C************************************************************           
C Fit to Badnell's dielectronic rec.data for C III, N IV and            
C O V. (J Phys.B 20 2081, 1987 & J.Phys.B 21 749, 1988 and 
C for O V also Phys Rev.A 39 1690, 1989).
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
      DIMENSION T(16),AC3(16),AN4(16),AO5(16)                           
      DATA T/3.0,3.2,3.4,3.6,3.8,4.,4.2,4.4,4.6,4.8,5.,5.2,
     &5.4,5.6,5.8,6./
      DATA AC3/-11.545,-11.176,-11.015,-11.006,-11.093,-11.218,         
     &-11.305,-11.146,-10.78,-10.55,-10.496,-10.57,-10.71,
     &-10.91,-11.14,-11.4/                     
      DATA AN4/-11.575,-11.141,-10.815,-10.648,-10.595,-10.614,         
     &-10.674,-10.726,-10.642,-10.458,-10.365,-10.394,-10.514,
     &-10.693,-10.914,-11.166/
      DATA AO5/-9.6716,-9.8182,-9.9706,-10.094,-10.177,-10.249,         
     &-10.32,-10.386,-10.421,-10.361,-10.287,-10.297,-10.396,
     &-10.562,-10.775,-11.017/
C                                                                       
      TLOG=DLOG10(TE)                                                   
      IF(TLOG.GE.T(16)) GOTO 200                                        
      IF(TLOG.LT.T(1)) GOTO 201                                        
      DO 100 I=1,15                                                     
      IF(TLOG.GE.T(I).AND.TLOG.LT.T(I+1)) GOTO 101                      
      GOTO 100                                                          
  101 QQ=(TLOG-T(I))/(T(I+1)-T(I))                                      
      ALDC3=AC3(I)+QQ*(AC3(I+1)-AC3(I))                                 
      ALDN4=AN4(I)+QQ*(AN4(I+1)-AN4(I))                                 
      ALDO5=AO5(I)+QQ*(AO5(I+1)-AO5(I))                                 
      GOTO 9999                                                         
  100 CONTINUE                                                          
  200 QQ=(TLOG-T(16))/(T(16)-T(15))                                     
      ALDC3=AC3(15)+QQ*(AC3(16)-AC3(15))                                
      ALDN4=AN4(15)+QQ*(AN4(16)-AN4(15))                                
      ALDO5=AO5(15)+QQ*(AO5(16)-AO5(15))                                
      goto 9999
  201 qq=(tlog-t(2))/(t(2)-t(1))
      ALDC3=AC3(1)+QQ*(AC3(2)-AC3(1))                                
      ALDN4=AN4(1)+QQ*(AN4(2)-AN4(1))                                
      ALDO5=AO5(1)+QQ*(AO5(2)-AO5(1))                                
 9999 ALDC3=1.D1**ALDC3                                                 
      ALDN4=1.D1**ALDN4                                                 
      ALDO5=1.D1**ALDO5                                                 
C                                                                       
      RETURN                                                            
      END                                                               
C************************************************************           
C************************************************************           
      SUBROUTINE BADN3(TE,alds2,alds3,alds4,alds5,alds6)
C************************************************************           
C Fit to Badnell's dielectronic rec.data for recombining S II-VI
C for T>3E4 K. (ApJ, 379 356, 1991) Scaled as T**-1.5 at higher
C temps than 1E6 K. For temps lower than 3E4 K, assume that the
C add Nussbaumer  .... ????
C & Storey's low temp results for Oxygen, scaled by diel.rec
C from Badnell at 1E5 K.
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
      DIMENSION T(16),A2(16),A3(16),A4(16),A5(16),A6(16)
      DATA T/4.5,4.6,4.7,4.8,4.9,5.,5.1,5.2,5.3,5.4,5.5,5.6,
     &5.7,5.8,5.9,6./
      data a2/11.82,11.53,11.31,11.16,11.07,11.02,11.02,11.04,
     &11.09,11.16,11.25,11.35,11.46,11.58,11.70,11.83/
      data a3/11.04,10.69,10.42,10.23,10.10,10.03,10.01,10.02,
     &10.06,10.12,10.20,10.29,10.40,10.51,10.64,10.76/
      data a4/10.54,10.32,10.12,9.96,9.86,9.80,9.79,9.81,
     &9.85,9.92,10.00,10.10,10.20,10.32,10.44,10.57/
      data a5/10.29,10.19,10.06,9.96,9.89,9.85,9.85,9.88,
     &9.93,10.00,10.09,10.19,10.30,10.42,10.54,10.68/
      data a6/10.33,10.14,10.00,9.91,9.86,9.84,9.86,9.91,
     &9.98,10.06,10.16,10.27,10.38,10.51,10.64,10.77/
C                                                                       
      IF(TE.GE.1.d6) then
       t6=te/1.d6
       alds2=(1.d1**(-a2(16)))/t6**1.5
       alds3=(1.d1**(-a3(16)))/t6**1.5
       alds4=(1.d1**(-a4(16)))/t6**1.5
       alds5=(1.d1**(-a5(16)))/t6**1.5
       alds6=(1.d1**(-a6(16)))/t6**1.5
      else
       TLOG=DLOG10(TE)
       if(tlog.lt.4.5) then
        is=1
       else
        DO 100 I=1,15
        IF(TLOG.GE.T(I).AND.TLOG.LT.T(I+1)) then
         is=i
        endif
  100   continue
       endif
       i=is
       QQ=(TLOG-T(i))/(T(i+1)-T(i))           
       if(i.eq.1) qq=1.d1*qq
       alds2=1.d1**(-a2(i)-qq*(a2(i+1)-a2(i)))
       alds3=1.d1**(-a3(i)-qq*(a3(i+1)-a3(i)))
       alds4=1.d1**(-a4(i)-qq*(a4(i+1)-a4(i)))
       alds5=1.d1**(-a5(i)-qq*(a5(i+1)-a5(i)))
       alds6=1.d1**(-a6(i)-qq*(a6(i+1)-a6(i)))
      endif
C                                                                       
      RETURN                                                            
      END                                                               
C************************************************************           
C************************************************************           
      SUBROUTINE BADN4(TE,aldo2,aldo3,aldo4,aldo6)
C************************************************************           
C Fit to Badnell & Pindzola's results for dielectronic rec.data
C for recombining O II-VI for T>4E4 K. (Phys Rev A, 39 1690,
C 1989) Scaled as T**-1.5 at higher temps than 6.3E6 K. For 
C T less than 6E4 K, add Nussbaumer & Storey's low temp results,
C except for recombining O V. For this take Badnell's more
C exact calc. (see BADN2).
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
      DIMENSION T(12),A2(12),A3(12),A4(12),A5(12),A6(12)
      DATA T/4.6,4.8,5.,5.2,5.4,5.6,5.8,6.,6.2,6.4,6.6,6.8/
      data a2/11.79,11.36,11.20,11.21,11.33,11.51,11.74,12.00,
     &12.27,12.55,12.84,13.13/
      data a3/11.44,10.97,10.75,10.71,10.78,10.94,11.15,11.39,
     &11.65,11.92,12.21,12.49/
      data a4/11.25,10.76,10.50,10.42,10.48,10.62,10.82,11.06,
     &11.32,11.60,11.88,12.17/
      data a5/10.42,10.36,10.29,10.30,10.40,10.56,10.78,11.02,
     &11.29,11.57,11.87,12.16/
      data a6/10.63,10.40,10.35,10.43,10.60,10.81,11.05,11.32,
     &11.60,10.88,12.17,12.47/
C                                                                       
      IF(TE.GT.6.3d6) then
       t6=te/1.d6
       aldo2=(1.d1**(-a2(12)))/t6**1.5
       aldo3=(1.d1**(-a3(12)))/t6**1.5
       aldo4=(1.d1**(-a4(12)))/t6**1.5
       aldo5=(1.d1**(-a5(12)))/t6**1.5
       aldo6=(1.d1**(-a6(12)))/t6**1.5
      else
       TLOG=DLOG10(TE)
       if(tlog.le.t(1))then
        aldo2=0.0d0
        aldo3=0.0d0
        aldo4=0.0d0
        aldo5=0.0d0
        aldo6=0.0d0
       else
        DO 100 I=1,11
        IF(TLOG.GT.T(I).AND.TLOG.LE.T(I+1)) GOTO 101                      
        GOTO 100                                                          
  101   QQ=(TLOG-T(I))/(T(I+1)-T(I))                                      
        aldo2=1.d1**(-a2(i)-qq*(a2(i+1)-a2(i)))
        aldo3=1.d1**(-a3(i)-qq*(a3(i+1)-a3(i)))
        aldo4=1.d1**(-a4(i)-qq*(a4(i+1)-a4(i)))
        aldo5=1.d1**(-a5(i)-qq*(a5(i+1)-a5(i)))
        aldo6=1.d1**(-a6(i)-qq*(a6(i+1)-a6(i)))
  100   CONTINUE                                                          
       endif
      endif
C                                                                       
      RETURN                                                            
      END                                                               
C************************************************************           
C************************************************************           
      SUBROUTINE BADN5(TE,aldc2,aldn3)
C************************************************************           
C Fit to Badnell & Pindzola's dielectronic rec.data for recombining
C C II and N III for T>2.5E4 K. (Phys Rev A, 320, 1685, 1989) Scaled 
C as T**-1.5 at higher than 6.3E6 K. For T less than 6E4 K, add
C Nussbaumer & Storey's low temp results.
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
      DIMENSION T(14),A2(14),A3(14)
      DATA T/4.4,4.5,4.6,4.8,5.,5.2,5.4,5.6,5.8,6.,6.2,6.4,6.6,
     &6.8/
      data a2/11.76,11.44,11.20,10.93,10.86,10.92,11.07,11.27,
     &11.51,11.77,12.05,12.33,12.62,12.91/
      data a3/11.89,11.48,11.16,10.74,10.56,10.55,10.66,10.78,
     &11.05,11.30,11.57,11.85,12.13,12.43/
C                                                                       
      IF(TE.GT.6.3d6) then
       t6=te/1.d6
       aldc2=(1.d1**(-a2(14)))/t6**1.5
       aldn3=(1.d1**(-a3(14)))/t6**1.5
      else
       TLOG=DLOG10(TE)
       if(tlog.le.t(1)) then
        aldc2=0.0d0
        aldn3=0.0d0
       else
        DO 100 I=1,13
        IF(TLOG.GT.T(I).AND.TLOG.LE.T(I+1)) GOTO 101                      
        GOTO 100                                                          
  101   QQ=(TLOG-T(I))/(T(I+1)-T(I))                                      
        aldc2=1.d1**(-a2(i)-qq*(a2(i+1)-a2(i)))
        aldn3=1.d1**(-a3(i)-qq*(a3(i+1)-a3(i)))
  100   CONTINUE                                                          
       endif
      endif
C                                                                       
      RETURN                                                            
      END                                                               
C************************************************************
C************************************************************
      subroutine romn5(TRYD,alrom)
C************************************************************
C N V dielectronic recomb. (C.Romanik, priv.comm. 1993)
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      dimension romai(7),romti(7),rn5ai(7),rn5ti(7)
      data rn5ai/1.42d-14,4.75d-15,1.d-13,1.01d-12,1.28d-12,
     &7.16d-11,2.03d-11/,rn5ti/0.0144d0,0.0335d0,0.058d0,
     &0.0928d0,0.341d0,0.718d0,3.72d0/
C
      do i=1,7
      romai(i)=rn5ai(i)
      romti(i)=rn5ti(i)
      enddo
      sum=0.0d0
        do i=1,7
        sum=sum+romai(i)*expfn(romti(i)/tryd)
	enddo
      alrom=sum/tryd**1.5
C
      return
      end
C************************************************************
C************************************************************
      subroutine verferl(a,b,te,t0,t1,radrec)
C************************************************************
C Radiative recombination from Verner & Ferland (1995, preprint).
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      x0=te/t0
      x1=te/t1
      radrec=a/((dsqrt(x0))*((1.d0+dsqrt(x0))**(1.d0-b))*
     &((1.d0+dsqrt(x1))**(1.d0+b)))
C
      return
      end
C************************************************************
C************************************************************           
      double precision function arf(a,b,c,d,pot,tev) 
C************************************************************           
C Collisional ionization according to Arnaud & Rothenflug '85
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z) 
C
      y=0.0d0
      x=pot/tev
      if(x.gt.1.d2) goto 100
      f1=arf1(x)
      f2=arf2(x)
      y1=a*(1.d0-x*f1)+b*(1.d0+x-x*(2.d0+x)*f1)+c*f1+d*x*f2
      y=(6.69d-7*y1*expfn(x))/(x*tev**1.5d0)
  100 continue
      arf=y
C
      return
      end
C************************************************************           
C************************************************************           
      double precision function arauli(iz,tev) 
C************************************************************           
C Auto-ionization according to Arnaud & Rothenflug (Li-sequence).
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z) 
C
      q=0.0d0
      z=dble(iz)
      potea=13.6d0*((z-0.835d0)**2-0.25d0*(z-1.62d0)**2)
      y=potea/tev
      if(y.gt.1.d2) goto 100
       zeff=z-0.43d0
       b=1.d0/(1.d0+2.d-4*z**3)
       f1=arf1(y)
       gy=2.22d0*f1+0.67d0*(1.d0-y*f1)+0.49d0*y*f1+1.2d0*y*(1.d0-
     & y*f1)
       q=(1.92d-7*b*gy*expfn(y))/(zeff**2*(dsqrt(tev)))
       if(iz.eq.6) q=0.6d0*q
       if(iz.eq.7) q=0.8d0*q
       if(iz.eq.8) q=1.25d0*q
  100 continue
      arauli=q
C
      return
      end
C************************************************************           
C************************************************************           
      double precision function arau(iz,iseq,tev) 
C************************************************************           
C Auto-ionization according to Arnaud & Rothenflug (Except Li-
C sequence).
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z) 
C
      q=0.0d0
      z=dble(iz)
      if(iseq.eq.11.and.iz.le.16) potea=26.d0*(z-10.d0)
      if(iseq.eq.11.and.iz.gt.16) potea=11.d0*(z-10.d0)**1.5
      if(iseq.eq.12) potea=10.3d0*(z-10.d0)**1.52
      if(iseq.eq.13) potea=18.d0*(z-11.d0)**1.33
      if(iseq.eq.14) potea=18.4d0*(z-12.d0)**1.36
      if(iseq.eq.15) potea=23.7d0*(z-13.d0)**1.29
      if(iseq.eq.16) potea=40.1d0*(z-14.d0)**1.1
      if(iseq.eq.19) potea=29.d0
      if(iseq.eq.20) potea=25.d0
      if(iseq.eq.21) potea=73.d0
      if(iseq.eq.22) potea=60.d0
      y=potea/tev
      if(y.gt.1.d2) goto 100
       f1=arf1(y)
       if(iseq.eq.11) then
        if(iz.le.16) then
         a=2.8d-17/(z-10.d0)**0.7
         y1=1.d0-y*f1
        else
         a=1.3d-14/(z-10.d0)**3.73
         y1=1.d0-0.5d0*(y-y**2+f1*y**3)
        endif
       elseif(iseq.ge.12.and.iseq.le.16) then
        a=4.0d-13/(potea*z**2)
        y1=1.d0-0.5d0*(y-y**2+f1*y**3)
       elseif(iseq.ge.19) then
        if(iseq.eq.19) then
         b=1.12d0
         a=9.8d-17
        elseif(iseq.eq.20) then
         b=1.12d0
         a=6.0d-17
        elseif(iseq.eq.21) then
         b=1.0d0
         a=5.0d-18
        elseif(iseq.eq.22) then
         b=1.0d0
         a=1.8d-17
        endif
        y1=1.d0+b*f1
       endif
       q=(6.69d7*a*potea*y1*expfn(y))/dsqrt(tev)
  100 continue
      arau=q
C
      return
      end
C************************************************************           
C************************************************************           
      double precision function arf1(x) 
C
      IMPLICIT REAL*8(A-H,O-Z) 
C
      if(x.le.0.02d0) then
       f1=dexp(x)*(x-0.5772d0-dlog(x))
      elseif(x.gt.0.02d0.and.x.lt.10.d0) then
       a1=dlog((x+1.d0)/x)
       if(x.lt.1.5d0) then 
        a=-0.5d0
       else
        a=0.5d0
       endif
       a2=(0.36d0+0.03d0*(x+0.01d0)**a)/(x+1.d0)**2
       f1=a1-a2
      elseif(x.ge.10.d0) then
       f1=(1.d0/x)*(1.d0-1.d0/x+2.d0/x**2-6.d0/x**3+24.d0/x**4)
      endif
      arf1=f1
C
      return
      end
C************************************************************           
C************************************************************           
      double precision function arf2(x) 
C
      IMPLICIT REAL*8(A-H,O-Z) 
C
      dimension p(14),q(15)
      DATA p/1.0d0,2.1658d2,2.0336d4,1.0911d6,3.7114d7,8.3963d8,
     &1.2889d10,1.3449d11,9.4002d11,4.2571d12,1.1743d13,1.7549d13,
     &1.0806d13,4.9776d11/ 
      DATA q/1.0d0,2.1958d2,2.0984d4,1.1517d6,4.0349d7,9.4900d8,
     &1.5345d10,1.7182d11,1.3249d12,6.9071d12,2.3531d13,4.9432d13,
     &5.7760d13,3.0225d13,3.3641d12/
      sump=0.0d0
      sumq=0.0d0
       do 100 i=1,14
       y=dble(1-i)
       sump=sump+p(i)*x**y
       sumq=sumq+q(i)*x**y
  100  continue
      sumq=sumq+q(15)/x**1.4d1
      arf2=sump/(sumq*x**2)
C
      return
      end
C************************************************************           
C************************************************************
      subroutine totnah(j,te,a)
C************************************************************
C Nahar's total recombination coeffs. C I = 1, N II = 2,
C O III = 3, F IV = 4, Ne V = 5, Na VI = 6, Mg VII = 7,
C Al VIII = 8, S XI = 10, Si I = 11, Si II = 12, S II = 13,
C S III = 14, C II = 15, Fe III = 16, C I-VI = 17-22,
C N I-VII = 23-29
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      common/recnah/tenah(81),alnah(29,81)
C
      telog=dlog10(te)
      if(telog.le.1.d0) telog=1.d0
      if(telog.ge.8.999999d0) telog=8.999999d0
      do i=1,80
	if(telog.ge.tenah(i).and.telog.lt.tenah(i+1)) then
	  qq=(telog-tenah(i))/(tenah(i+1)-tenah(i))
	  arec=alnah(j,i)*(1.d0-qq)+qq*alnah(j,i)
	  a=1.d1**arec
        endif
      enddo
C
      return
      end
C*********************************************************************
C*********************************************************************
      SUBROUTINE BEFORB(TE,I,DENE,ABUND,COLTOTI,COLTOT,F31,F41,F51,
     &F6T43,F65,COOL)
C*********************************************************************
C  Forbidden lines (Be I-like). Enumeration of the ions are: (= I)
C
C  1 = C III  2 = Mg I  3 = Al II  4 = Si III
C
C  The atom is treated as a 6-level atom.
C  (Returned to RAD when E21/Tev > 20).
C
C  A-values for C III from Allard et al. (AAS 84, 563, '90) (superseded
C  by Verner, Verner & Ferland '96), 
C  Fleming et al. (Phys.Scr. 49, 316, '94; MNRAS, 279, 1289) and 
C  Nussbaumer & Storey (A&A 64, 139, '78). 
C  Energies from Mendoza ('83) and Daresbury ('86).
C  Omegas from Seaton (J.Phys.B. 20, 6431, '87), and Keenan et al.
C  (ApJ 389, 443, '92).
C  A-values for Al II from Johnson et al. (ApJ 308, 1013, '86) 
C  and Aggarwal & Keenan (J.Phys.B 27, 2343, '94). Energies from 
C  the ADS NIST Database ('95). A-values for Al II and Si III updated
C  with Zou & Froese Fischer (J.Phys.B 34, 915, '01). Omegas from 
C  Keenan et al. (ApJ 385, 375, '92) and Aggarwal & Keenan 
C  (J.Phys.B 27, 5321, '94). 
C  Energies for Mg I and Si III from the ADS NIST Database ('95). 
C  (Note that for Mg I there are 4s levels (not included) intervening 
C  the model atom below. Note also that the 3PE level is not included 
C  for C III. This lies below 1D, which is included.)
C*********************************************************************
      implicit real*8(a-h,o-z)
C
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
c
      dimension ene2(4),ene3(4),ene4(4),ene5(4),ene6(4),
     &a31(4),a32(4),a41(4),a42(4),a43(4),a51(4),a63(4),
     &a64(4),a65(4),om1t234(4),om15(4),om16(4),om23(4),om24(4),
     &om34(4),om234t5(4),om234t6(4),om56(4),al1t234(4),al15(4),
     &al16(4),al23(4),al24(4),al34(4),al234t5(4),al234t6(4),
     &al56(4),g(6),ama(4),a52(4),a53(4),a54(4)
c
      data ene2/6.4924,2.7092,4.6472,6.5411/
      data ene3/6.4954,2.7117,4.6547,6.5570/
      data ene4/6.5024,2.7168,4.6702,6.5896/
      data ene5/12.690,4.3460,7.4210,10.2768/
      data ene6/18.090,5.7535,10.5988,15.1534/
      data a31/1.038d2,0.,3.262d3,1.720d4/
      data a32/2.39d-7,1.45d-7,4.061d-6,3.771d-5/
      data a41/5.19d-3,0.,3.454d-3,1.265d-2/
      data a42/2*0.,1.857d-10,2.878d-9/
      data a43/2.41d-6,9.10d-7,2.565d-5,2.398d-4/
      data a51/1.79d9,0.,1.438d9,2.509d9/
      data a52/2*0.,2.293d-3,1.574d-2/
      data a53/2*0.,1.74d-3,1.188d-2/
      data a54/2*0.,2.779d-3,1.893d-2/
      data a63/4.85d3,0.,7.75d3,0./
      data a64/2.97d4,0.,2.03d4,0./
      data a65/1.40d8,0.,2.63d8,0./
      data om1t234/1.095d0,0.,3.167d0,0./
      data om15/4.35d0,0.,7.382d0,0./
      data om16/0.415d0,0.,1.326d0,0./
      data om23/0.95d0,0.,1.67d0,0./
      data om24/0.695d0,0.,2.00d0,0./
      data om34/2.75d0,0.,6.54d0,0./
      data om234t5/3.408d0,0.,7.635d0,0./
      data om234t6/1.425d0,0.,11.63d0,0./
      data om56/10.48d0,0.,5.324d0,0./
      data al1t234/-0.0734d0,0.,0.189d0,0./
      data al15/0.063d0,0.,0.0835d0,0./
      data al16/-0.0419d0,0.,-0.126d0,0./
      data al23/0.18d0,0.,0.039d0,0./
      data al24/0.27d0,0.,0.076d0,0./
      data al34/0.27d0,0.,0.0656d0,0./
      data al234t5/-0.163d0,0.,-0.186d0,0./
      data al234t6/-0.0665d0,0.,-0.364d0,0./
      data al56/0.0490d0,0.,-0.27d0,0./
      data g/1.,1.,3.,5.,3.,5./
      data ama/12.,24.305,26.982,28.086/
C
      cool=0.0d0
      f31=0.0d0
      f41=0.0d0
      f51=0.0d0
      f6t43=0.0d0
      f65=0.0d0
      if(abund.ge.1.d-10) then
c
      tev=te/1.1609d4
      t4=te/1.d4
      cons=8.63d-6*dene/dsqrt(te)
c
      e21=ene2(i)
        IF((E21/TEV).lt.2.d1) THEN
      e31=ene3(i)
      e41=ene4(i)
      e51=ene5(i)
      e61=ene6(i)
      e32=ene3(i)-ene2(i)
      e42=ene4(i)-ene2(i)
      e52=ene5(i)-ene2(i)
      e62=ene6(i)-ene2(i)
      e43=ene4(i)-ene3(i)
      e53=ene5(i)-ene3(i)
      e63=ene6(i)-ene3(i)
      e54=ene5(i)-ene4(i)
      e64=ene6(i)-ene4(i)
      e65=ene6(i)-ene5(i)
C
      o1t234=om1t234(i)*t4**al1t234(i)
      o15=om15(i)*t4**al15(i)
      o16=om16(i)*t4**al16(i)
      o23=om23(i)*t4**al23(i)
      o24=om24(i)*t4**al24(i)
      o34=om34(i)*t4**al34(i)
      o234t5=om234t5(i)*t4**al234t5(i)
      o234t6=om234t6(i)*t4**al234t6(i)
      o56=om56(i)*t4**al56(i)
      if(i.eq.3) then
	if(te.gt.1.5d4) then
          o1t234=3.419d0*(t4/1.5d0)**(-.169)
	endif
	if(te.gt.4.d4) then
          o16=1.113d0*(t4/4.d0)**.185
	endif
      endif
      gsum=g(2)+g(3)+g(4)
      o12=(g(2)*o1t234)/gsum
      o13=(g(3)*o1t234)/gsum
      o14=(g(4)*o1t234)/gsum
      o25=(g(2)*o234t5)/gsum
      o26=(g(2)*o234t6)/gsum
      o35=(g(3)*o234t5)/gsum
      o36=(g(3)*o234t6)/gsum
      o45=(g(4)*o234t5)/gsum
      o46=(g(4)*o234t6)/gsum
      c21=cons*o12/g(2)
      c31=cons*o13/g(3)
      c41=cons*o14/g(4)
      c51=cons*o15/g(5)
      c61=cons*o16/g(6)
      c32=cons*o23/g(3)
      c42=cons*o24/g(4)
      c52=cons*o25/g(5)
      c62=cons*o26/g(6)
      c43=cons*o34/g(4)
      c53=cons*o35/g(5)
      c63=cons*o36/g(6)
      c54=cons*o45/g(5)
      c64=cons*o46/g(6)
      c65=cons*o56/g(6)
      c12=c21*expfn(e21/tev)*g(2)/g(1)
      c13=c31*expfn(e31/tev)*g(3)/g(1)
      c14=c41*expfn(e41/tev)*g(4)/g(1)
      c15=c51*expfn(e51/tev)*g(5)/g(1)
      c16=c61*expfn(e61/tev)*g(6)/g(1)
      c23=c32*expfn(e32/tev)*g(3)/g(2)
      c24=c42*expfn(e42/tev)*g(4)/g(2)
      c25=c52*expfn(e52/tev)*g(5)/g(2)
      c26=c62*expfn(e62/tev)*g(6)/g(2)
      c34=c43*expfn(e43/tev)*g(4)/g(3)
      c35=c53*expfn(e53/tev)*g(5)/g(3)
      c36=c63*expfn(e63/tev)*g(6)/g(3)
      c45=c54*expfn(e54/tev)*g(5)/g(4)
      c46=c64*expfn(e64/tev)*g(6)/g(4)
      c56=c65*expfn(e65/tev)*g(6)/g(5)
c
      amass=ama(i)
      m=0
 9988 continue
      m=m+1
      adamp=a31(i)+a32(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(3)*g(1))/(x(1)*g(3)))
      call escacf(e31,a31(i),adamp,g(1),g(3),te,amass,tcorr,
     &coltoti,coltot,p31)
      adamp=a41(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(4)*g(1))/(x(1)*g(4)))
      call escacf(e41,a41(i),adamp,g(1),g(4),te,amass,tcorr,
     &coltoti,coltot,p41)
      adamp=a51(i)+a52(i)+a53(i)+a54(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(5)*g(1))/(x(1)*g(5)))
      call escacf(e51,a51(i),adamp,g(1),g(5),te,amass,tcorr,
     &coltoti,coltot,p51)
      adamp=a31(i)+a32(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(3)*g(2))/(x(2)*g(3)))
      call escacf(e32,a32(i),adamp,g(2),g(3),te,amass,tcorr,
     &coltoti,coltot,p32)
      adamp=a41(i)+a43(i)+a32(i)+a31(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(4)*g(3))/(x(3)*g(4)))
      call escacf(e43,a43(i),adamp,g(3),g(4),te,amass,tcorr,
     &coltoti,coltot,p43)
      adamp=a63(i)+a64(i)+a65(i)+a31(i)+a32(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(6)*g(3))/(x(3)*g(6)))
      call escacf(e63,a63(i),adamp,g(3),g(6),te,amass,tcorr,
     &coltoti,coltot,p63)
      adamp=a63(i)+a64(i)+a65(i)+a41(i)+a43(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(4)*(1.d0-(x(6)*g(4))/(x(4)*g(6)))
      call escacf(e64,a64(i),adamp,g(4),g(6),te,amass,tcorr,
     &coltoti,coltot,p64)
      adamp=a63(i)+a64(i)+a65(i)+a51(i)+a52(i)+a53(i)+a54(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(5)*(1.d0-(x(6)*g(5))/(x(5)*g(6)))
      call escacf(e65,a65(i),adamp,g(5),g(6),te,amass,tcorr,
     &coltoti,coltot,p65)
      adamp=a51(i)+a52(i)+a53(i)+a54(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(5)*g(2))/(x(2)*g(5)))
      call escacf(e52,a52(i),adamp,g(2),g(5),te,amass,tcorr,
     &coltoti,coltot,p52)
      adamp=a51(i)+a52(i)+a53(i)+a54(i)+a31(i)+a32(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(5)*g(3))/(x(3)*g(5)))
      call escacf(e53,a53(i),adamp,g(3),g(5),te,amass,tcorr,
     &coltoti,coltot,p53)
      adamp=a51(i)+a52(i)+a53(i)+a54(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(4)*(1.d0-(x(5)*g(4))/(x(4)*g(5)))
      call escacf(e54,a54(i),adamp,g(4),g(5),te,amass,tcorr,
     &coltoti,coltot,p54)
      a(1,1)=c12+c13+c14+c15+c16
      a(1,2)=-c21
      a(1,3)=-(c31+p31*a31(i))
      a(1,4)=-(c41+p41*a41(i))
      a(1,5)=-(c51+p51*a51(i))
      a(1,6)=-c61
      a(2,1)=-c12
      a(2,2)=c21+c23+c24+c25+c26
      a(2,3)=-(c32+p32*a32(i))
      a(2,4)=-(c42+a42(i))
      a(2,5)=-(c52+p52*a52(i))
      a(2,6)=-c62
      do 100 j=1,6
      x(j)=0.d0
      a(3,j)=1.d0
  100 continue
      x(3)=1.d0
      a(4,1)=-c14
      a(4,2)=-c24
      a(4,3)=-c34
      a(4,4)=c41+c42+c43+c45+c46+p41*a41(i)+a42(i)+p43*a43(i)
      a(4,5)=-(c54+p54*a54(i))
      a(4,6)=-(c64+p64*a64(i))
      a(5,1)=-c15
      a(5,2)=-c25
      a(5,3)=-c35
      a(5,4)=-c45
      a(5,5)=c51+c52+c53+c54+c56+p51*a51(i)+p52*a52(i)+p53*a53(i)+
     &p54*a54(i)
      a(5,6)=-(c65+p65*a65(i))
      a(6,1)=-c16
      a(6,2)=-c26
      a(6,3)=-c36
      a(6,4)=-c46
      a(6,5)=-c56
      a(6,6)=c61+c62+c63+c64+c64+p63*a63(i)+p64*a64(i)+p65*a65(i)
c
      call matrix(6)
      if(m.lt.2) goto 9988
c
      elab=(1.6022d-12*abund)/(12.57d0*dene)
      cool=elab*(e21*(x(1)*c12-x(2)*c21)+e31*
     &(x(1)*c13-x(3)*c31)+e41*(x(1)*c14-x(4)*c41)+e51*(x(1)*c15-
     &x(5)*c51)+e61*(x(1)*c16-x(6)*c61)+e32*(x(2)*c23-x(3)*c32)+
     &e42*(x(2)*c24-x(4)*c42)+e52*(x(2)*c25-x(5)*c52)+e62*(x(2)*
     &c26-x(6)*c62)+e43*(x(3)*c34-x(4)*c43)+e53*(x(3)*c35-x(5)*
     &c53)+e63*(x(3)*c36-x(6)*c63)+e54*(x(4)*c45-x(5)*c54)+
     &e64*(x(4)*c46-x(6)*c64)+e65*(x(5)*c56-x(6)*c65))
      cool=12.57d0*cool
      f31=elab*x(3)*a31(i)*e31*p31
      f41=elab*x(4)*a41(i)*e41*p41
      f51=elab*x(5)*a51(i)*e51*p51
      f6t43=elab*x(6)*(p64*e64*a64(i)+p63*e63*a63(i))
      f65=elab*x(6)*e65*a65(i)*p65
c
        ENDIF
      endif
C
      RETURN
      END
C*********************************************************************
C*********************************************************************
      SUBROUTINE BFORB(TE,I,DENE,DENH,ABUND,COLTOTI,COLTOT,F21,F345T12,
     &F43,F54,COOL)
C*********************************************************************
C  Forbidden lines (B I-like). Enumeration of the ions are: (= I)
C
C  1 = C II  2 = N III  3 = O IV  4 = S IV  5 = Si II  6 = Ne VI
C  (Ar VI is mainly in the EUV/UV.)
C
C  The atom is treated as a 5-level atom.
C  (Reduced to a 2-level atom for E32/Tev > 20)
C
C  A-values not mentioned specifically are from Mendoza ('83).
C  A-values for C II from Biemont et al. (J. Phys.B, 27, 5841, '94.) 
C  A-values for N III from Brage et al. (ApJ, 445, 457, '95). 
C  (These A-values are somewhat lower than those of Biemont et al, 
C  J.Phys.B, '94.) A-values for O IV from Pradhan's compilation 
C  (Jan. '96; Merkelis et al. '94.), updated with Brage, Judge & Brekke
C  (ApJ, 464, 1030, '96). Si II from Calamai et al, 
C  (ApJ, 415, L59, '93). A43 and A54 are isoelectronically guessed for 
C  N III. For S IV, Si II and Ne VI they are put equal to the values for 
C  O IV. A21 and om21 for Si II from Kafatos & Lynch, ApJS 42, 611, 1980. 
C  A21 of Ne VI from Mendoza ('83) and for intercombination lines from
C  Dankwort & Trefftz (A & A, 65, 93, '78).
C  For C and O we also use Opacity Project data (Blum & Pradhan, ApJS 
C  80, 425, 1992). For O IV and Ne VI we use the collisional strengths
C  of Zhang et al. (A & A, 283, 319, '93). N III is updated with Stafford, 
C  Bell, & Hibbert, MN 266, 715 (1994). om34, om35 and om45 isoelectronically 
C  guessed for Si at 1E4 K (use S IV's T-dependence for Si II).
C  Collisions with H for fine structure lines of C II and Si II are
C  included (E.Roueff, A&A 234, 567, 1990).
C  Energy levels of C II, N III & O IV slightly changed by Peng & Pradhan
C  (AAS, 112, 151, '95.)
C*********************************************************************
      implicit real*8(a-h,o-z)
C
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
c
      dimension ene2(6),ene3(6),ene4(6),ene5(6),a21(6),a31(6),
     &a41(6),a32(6),a42(6),a52(6),a43(6),a54(6),om12(6),om13(6),
     &om14(6),om15(6),om23(6),om24(6),om25(6),om34(6),om35(6),
     &om45(6),al12(6),al13(6),al14(6),al15(6),al23(6),al24(6),
     &al25(6),al34(6),al35(6),al45(6),g(5),ama(6)
c
      data ene2/.007932,.02163,.04785,.1177,.03558,.16182/
      data ene3/5.3283,7.0869,8.8574,8.7861,5.3095,12.4231/
      data ene4/5.3310,7.0943,8.8736,8.8361,5.3230,12.4775/
      data ene5/5.3346,7.1044,8.8966,8.8590,5.3447,12.5578/
      data a21/2.29d-6,4.77d-5,5.20d-4,7.73d-3,2.17d-4,2.02d-2/
      data a31/7.27d1,3.61d2,1.47d3,5.50d4,5.20d3,1.27d4/
      data a41/1.88,9.11,38.4,140.,13.2,301./
      data a32/82.8,372.,1.43d3,3.39d4,4.41d3,1.11d4/
      data a42/10.3,65.1,294.,1.95d4,1.22d3,2.80d3/
      data a52/50.9,282.,1.17d3,3.95d4,2.46d3,9.87d3/
      data a43/2.39d-7,3.5d-6,4*5.07d-5/
      data a54/3.67d-7,6.1d-6,4*1.02d-4/
      data om12/0.,1.44,2.60,6.42,3.78,1.48/
      data om13/.242,.198,.139,.51,.406,.058/
      data om14/.362,.298,.217,.87,.692,.09/
      data om15/.235,.201,.169,.95,.752,.074/
      data om23/.177,.151,.123,.66,.545,.054/
      data om24/.477,.399,.309,1.47,1.17,.131/
      data om25/1.02,.844,.619,2.53,2.01,.259/
      data om34/.824,1.10,1.20,3.04,2.00,.587/
      data om35/.853,.668,.822,2.53,1.85,.401/
      data om45/1.98,2.04,2.35,5.85,4.50,.586/
      data al12/0.,.182,-.398,2*0.,-.569/
      data al13/.029,.063,-.309,2*-.16,-.284/
      data al14/.035,.083,-.328,2*-.133,-.409/
      data al15/.062,.185,.091,2*-.121,-.509/
      data al23/.057,.16,-.377,2*-.122,-.466/
      data al24/.046,.126,-.357,2*-.134,-.451/
      data al25/.036,.089,-.33,2*-.123,-.396/
      data al34/.365,.072,-.557,2*-.174,-.434/
      data al35/.186,.08,-.275,2*-.141,-.399/
      data al45/.269,.08,-.287,2*-.197,-.419/
      data g/2.,4.,2.,4.,6./
      data ama/12.,14.,16.,32.1,28.09,20.179/
C
      cool=0.0d0
      f21=0.0d0
      f345t12=0.0d0
      f43=0.0d0
      f54=0.0d0
      if(abund.ge.1.d-10) then
c
      tmx=2.d5
      tev=te/1.1609d4
      t4=te/1.d4
      cons=8.63d-6*dene/dsqrt(te)
c
      e21=ene2(i)
      e31=ene3(i)
      e41=ene4(i)
      e51=ene5(i)
      e32=ene3(i)-ene2(i)
      e42=ene4(i)-ene2(i)
      e52=ene5(i)-ene2(i)
      e43=ene4(i)-ene3(i)
      e53=ene5(i)-ene3(i)
      e54=ene5(i)-ene4(i)
c
      if(i.eq.1) then
       o12=dmax1(1.58d0,1.58d0*(te/1.d3)**0.11d0)
      elseif(i.eq.2) then
       o12=cheb(tmx,te,0.724d0,0.2349d0,-0.0682d0,-0.0517d0)
      elseif(i.eq.3) then
        if(t4.gt.0.6d0) then
           o12=2.70d0*(te/6.d3)**(-.25)
        else
           o12=2.70d0*(te/6.d3)**(.13)
        endif
      elseif(i.ge.4) then
        o12=om12(i)*t4**al12(i)
      endif
      c21=cons*o12/g(2)
C - H-atom collisions for C II and Si II
      if(i.eq.1) c21=c21+2.05d-9*denh*(te/2.d3)**.0759
      if(i.eq.5) c21=c21+1.1d-9*denh*(te/2.d3)**.345
      c12=c21*expfn(e21/tev)*g(2)/g(1)
       IF((E32/TEV).lt.2.d1) then
      if(i.ne.2.or.i.ne.3) then
       o13=om13(i)*t4**al13(i)
       o14=om14(i)*t4**al14(i)
       o15=om15(i)*t4**al15(i)
       o23=om23(i)*t4**al23(i)
       o24=om24(i)*t4**al24(i)
       o25=om25(i)*t4**al25(i)
       o34=om34(i)*t4**al34(i)
       o35=om35(i)*t4**al35(i)
       o45=om45(i)*t4**al45(i)
      elseif(i.eq.2) then
       o13=cheb(tmx,te,-3.6087d0,-0.0656d0,-0.0831d0,-0.0165d0)
       o14=cheb(tmx,te,-2.7917d0,-0.0501d0,-0.0914d0,-0.0197d0)
       o15=cheb(tmx,te,-3.5774d0,0.0274d0,-0.1330d0,-0.0337d0)
       o23=cheb(tmx,te,-4.1572d0,0.0119d0,-0.1247d0,-0.0311d0)
       o24=cheb(tmx,te,-2.2053d0,-0.0191d0,-0.1080d0,-0.0257d0)
       o25=cheb(tmx,te,-0.7075d0,-0.0465d0,-0.0934d0,-0.0204d0)
       o34=cheb(tmx,te,-0.0975d0,0.0837d0,-0.0437d0,-0.0195d0)
       o35=cheb(tmx,te,-0.7808d0,0.1475d0,-0.0418d0,-0.0211d0)
       o45=cheb(tmx,te,1.2787d0,0.1128d0,-0.0426d0,-0.0203d0)
      elseif(i.eq.3) then
        if(t4.gt.0.4d0) then
           o13=0.154d0*(te/4.d3)**(-.24)
        else
           o13=0.154d0*(te/4.d3)**(.12)
        endif
        if(t4.gt.0.4d0) then
           o14=0.24d0*(te/4.d3)**(-.25)
        else
           o14=0.24d0*(te/4.d3)**(.14)
        endif
        if(t4.gt.0.5d0) then
           o15=0.189d0*(te/5.d3)**(-.32)
        else
           o15=0.189d0*(te/5.d3)**(.22)
        endif
        if(t4.gt.0.4d0) then
           o23=0.137d0*(te/4.d3)**(-.28)
        else
           o23=0.137d0*(te/4.d3)**(.23)
        endif
        if(t4.gt.0.4d0) then
           o24=0.342d0*(te/4.d3)**(-.27)
        else
           o24=0.342d0*(te/4.d3)**(.18)
        endif
        if(t4.gt.0.4d0) then
           o25=0.189d0*(te/4.d3)**(-.25)
        else
           o25=0.189d0*(te/4.d3)**(.15)
        endif
        if(t4.gt.0.7d0) then
           o34=1.23d0*(te/7.d3)**(-.25)
        else
           o34=1.23d0*(te/7.d3)**(.064)
        endif
        if(t4.gt.0.7d0) then
           o35=0.836d0*(te/7.d3)**(-.23)
        else
           o35=0.836d0*(te/7.d3)**(.055)
        endif
        if(t4.gt.0.7d0) then
           o45=2.40d0*(te/7.d3)**(-.25)
        else
           o45=2.40d0*(te/7.d3)**(.060)
        endif
      endif
      c31=cons*o13/g(3)
      c41=cons*o14/g(4)
      c51=cons*o15/g(5)
      c32=cons*o23/g(3)
      c42=cons*o24/g(4)
      c52=cons*o25/g(5)
      c43=cons*o34/g(4)
      c53=cons*o35/g(5)
      c54=cons*o45/g(5)
      c13=c31*expfn(e31/tev)*g(3)/g(1)
      c14=c41*expfn(e41/tev)*g(4)/g(1)
      c15=c51*expfn(e51/tev)*g(5)/g(1)
      c23=c32*expfn(e32/tev)*g(3)/g(2)
      c24=c42*expfn(e42/tev)*g(4)/g(2)
      c25=c52*expfn(e52/tev)*g(5)/g(2)
      c34=c43*expfn(e43/tev)*g(4)/g(3)
      c35=c53*expfn(e53/tev)*g(5)/g(3)
      c45=c54*expfn(e54/tev)*g(5)/g(4)
       ENDIF
c
      amass=ama(i)
      m=0
 9988 continue
      m=m+1
      adamp=a21(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(2)*g(1))/(x(1)*g(2)))
      call escacf(e21,a21(i),adamp,g(1),g(2),te,amass,tcorr,
     &coltoti,coltot,p21)
       IF((E32/TEV).lt.2.d1) then
      adamp=a31(i)+a32(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(3)*g(1))/(x(1)*g(3)))
      call escacf(e31,a31(i),adamp,g(1),g(3),te,amass,tcorr,
     &coltoti,coltot,p31)
      adamp=a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(4)*g(1))/(x(1)*g(4)))
      call escacf(e41,a41(i),adamp,g(1),g(4),te,amass,tcorr,
     &coltoti,coltot,p41)
      adamp=a21(i)+a32(i)+a31(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(3)*g(2))/(x(2)*g(3)))
      call escacf(e32,a32(i),adamp,g(2),g(3),te,amass,tcorr,
     &coltoti,coltot,p32)
      adamp=a21(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(4)*g(2))/(x(2)*g(4)))
      call escacf(e42,a42(i),adamp,g(2),g(4),te,amass,tcorr,
     &coltoti,coltot,p42)
      adamp=a21(i)+a52(i)+a54(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(5)*g(2))/(x(2)*g(5)))
      call escacf(e52,a52(i),adamp,g(2),g(5),te,amass,tcorr,
     &coltoti,coltot,p52)
      adamp=a31(i)+a32(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(4)*g(3))/(x(3)*g(4)))
      call escacf(e43,a43(i),adamp,g(3),g(4),te,amass,tcorr,
     &coltoti,coltot,p43)
      adamp=a52(i)+a54(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(4)*(1.d0-(x(5)*g(4))/(x(4)*g(5)))
      call escacf(e54,a54(i),adamp,g(4),g(5),te,amass,tcorr,
     &coltoti,coltot,p54)
      do 100 j=1,5
      x(j)=0.d0
      a(1,j)=1.d0
  100 continue
      x(1)=1.d0
      a(2,1)=-c12
      a(2,2)=c21+c23+c24+c25+p21*a21(i)
      a(2,3)=-(c32+p32*a32(i))
      a(2,4)=-(c42+p42*a42(i))
      a(2,5)=-(c52+p52*a52(i))
      a(3,1)=-c13
      a(3,2)=-c23
      a(3,3)=c31+c32+c34+c35+p31*a31(i)+p32*a32(i)
      a(3,4)=-(c43+p43*a43(i))
      a(3,5)=-c53
      a(4,1)=-c14
      a(4,2)=-c24
      a(4,3)=-c34
      a(4,4)=c41+c42+c43+c45+p41*a41(i)+p42*a42(i)+p43*a43(i)
      a(4,5)=-(c54+p54*a54(i))
      a(5,1)=-c15
      a(5,2)=-c25
      a(5,3)=-c35
      a(5,4)=-c45
      a(5,5)=c51+c52+c53+c54+p52*a52(i)+p54*a54(i)
c
      call matrix(5)
       ELSE
      a(1,1)=1.d0
      a(1,2)=1.d0
      a(2,1)=-c12
      a(2,2)=c21+p21*a21(i)
      x(1)=1.d0
      x(2)=0.0d0
      call matrix(2)
       ENDIF
      if(m.lt.2) goto 9988
c
      elab=(1.6022d-12*abund)/(12.57d0*dene)
       IF((E32/TEV).lt.2.d1) then
      cool=elab*(e21*(x(1)*c12-x(2)*c21)+e31*
     &(x(1)*c13-x(3)*c31)+e41*(x(1)*c14-x(4)*c41)+e51*(x(1)*c15-
     &x(5)*c51)+e32*(x(2)*c23-x(3)*c32)+
     &e42*(x(2)*c24-x(4)*c42)+e52*(x(2)*c25-x(5)*c52)+
     &e43*(x(3)*c34-x(4)*c43)+e53*(x(3)*c35-x(5)*c53)+
     &e54*(x(4)*c45-x(5)*c54))
      f345t12=elab*(x(4)*(p41*e41*a41(i)+p42*e42*a42(i))+
     &x(5)*p52*e52*a52(i)+x(3)*(p31*e31*a31(i)+p32*e32*a32(i)))
      f43=elab*x(4)*a43(i)*e43*p43
      f54=elab*x(5)*a54(i)*e54*p54
       ELSE
      cool=elab*(e21*(x(1)*c12-x(2)*c21))
       ENDIF
      cool=12.57d0*cool
      f21=elab*x(2)*a21(i)*e21*p21
c
      endif
C
      RETURN
      END
C*********************************************************************
C*********************************************************************
      SUBROUTINE CFORB(TE,I,DENE,ABUND,COLTOTI,COLTOT,F21,F32,F4T123,
     &F5T23,F6T23,F54,COOL)
C*********************************************************************
C  Forbidden lines (C I-like). Enumeration of the ions are: (= I)
C
C  [last updated May 16, 1997]
C
C  1 = C I  2 = N II  3 = O III  4 = Ne V  5 = S III  6 = Ar V  
C  7 = Ca VII  8 = Mg VII  9 = Fe XIII
C
C  The atom is treated as a 6-level atom.
C  (Reduced to a 3-level atom for E43/Tev > 20)
C
C  WARNING !! Note that coll.strengths may be for multiplets. om23t5 and
C      om23t6 are for ALL transitions from levels 1,2 & 3 to 5 & 6.
C
C  Energy levels for O III from Bashkin & Stoner '75 (experimental)
C  Ne V from Lennon & Burke (MNRAS 251, 628; 1991 & AAS 103, 273; 1994).
C  (Ne V Omegas for fine structure lines may be wrong by up to a factor
C  of 3; Oliva et al. 1996, A&A 305, L21.)
C  S III 5S2 A-values from Hayes 1986, S III Coll.strengths from Tayal
C  (ApJ 1997), also from S III Galavis et al. (AAS, 111, 347, '95). Ar V and 
C  Ca VII from Galavis et al. Ca VII multiplet levels from Nahar & Pradhan
C  (J.Phys.B, 26, 1103, '93.). Fine structure levels from Bashkin & 
C  Stoner ('78). A-values for Ca VII put equal to O III A-values 
C  (which should give a lower limit to the cooling.)
C  A-values A62,A63 of N II from Brage et al. (ApJ, 478, 423, '97).
C  S, Ar, Ca & Mg  essentially treated as a 5-level atom. 
C  Mg Omegas from Lennon & Burke (1994). A-values and level energies 
C  from Nussbaumer & Rusca (A&A 72, 129, '79).
C  Fe XIII energy levels from Corliss & Sugar ('82) and Fawcett 
C  (J.Phys.B. '71). A-values from J.Phys.Chem.Ref. Data ('88), 
C  and Omegas from Tayal (ApJ 446, 895, '95). (Omegas are probably
C  valid only for T > 1E5 K.)
C  C I coll. strengths from Johnson, Burke & Kingston, J.Phys.B 20, 2553 
C  (1987). O III from Aggarwal, ApJ Suppl. 85, 197 (1993) and N II from 
C  Stafford et al., MNRAS 268, 816 (1994)
C  A-values of C, N, O, Ne and Mg updated with Galavis et al. 1997, A&AS,
C  123, 159
C*********************************************************************
      implicit real*8(a-h,o-z)
C
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
c
      dimension ene2(9),ene3(9),ene4(9),ene5(9),ene6(9),a21(9),
     &a31(9),a32(9),a41(9),a42(9),a43(9),a52(9),a53(9),a54(9),
     &a62(9),a63(9),om12(9),om13(9),om23(9),om123t4(9),om23t5(9),
     &om45(9),om23t6(9),al12(9),al13(9),al23(9),al123t4(9),
     &al23t5(9),al45(9),al23t6(9),g(6),ama(9),a0(3,2),a1(3,2),
     &a2(3,2),a3(3,2),a4(3,2),yy(3)
c
      data ene2/.002033,.006038,.01403,.05113,.03685,.09471,
     &.0202,.1398,1.1531/
      data ene3/.005381,.01622,.03796,.1376,.1032,.2516,.0505,
     &.3645,2.3008/
      data ene4/1.2637,1.8989,2.4857,3.7555,1.4035,2.0208,2.394,
     &5.080,5.9559/
      data ene5/2.6839,4.0528,5.3250,7.9240,3.3677,4.3492,5.755,
     &10.562,11.338/
      data ene6/4.1825,5.8004,7.4490,10.9553,7.200,3*12.0,35.589/
      data a21/7.93d-8,2.08d-6,2.66d-5,1.27d-3,4.72d-4,8.0d-3,
     &2.66d-5,2.51d-2,14.d0/
      data a31/2.05d-14,1.13d-12,3.09d-11,4.97d-9,4.61d-8,
     &1.2d-6,3.09d-11,2.32d-7,6.3d-3/
      data a32/2.65d-7,7.46d-6,9.70d-5,4.59d-3,2.07d-3,2.7d-2,
     &9.70d-5,8.05d-2,9.86d0/
      data a41/5.92d-8,3.55d-7,1.69d-6,1.94d-5,5.82d-6,3.5d-5,
     &1.69d-6,1.16d-4,0.d0/
      data a42/7.48d-5,1.02d-3,6.99d-3,1.25d-1,2.21d-2,2.0d-1,
     &6.99d-3,1.19d0,63.04d0/
      data a43/2.23d-4,3.00d-3,2.04d-2,3.50d-1,5.76d-2,4.8d-1,
     &2.04d-2,3.11d0,75.07d0/
      data a52/2.38d-3,3.30d-2,2.27d-1,3.99d0,7.96d-1,6.6d0,
     &2.27d-1,35.8d0,1.0d3/
      data a53/2.12d-5,1.32d-4,6.09d-4,6.29d-3,1.05d-2,5.7d-2,
     &6.09d-4,3.61d-2,3.8d0/
      data a54/6.42d-1,1.02d0,1.56d0,2.83d0,2.22d0,3.3d0,
     &1.56d0,3.96d0,8.1d0/
C     data a62/6.94d0,4.8d1,2.12d2,2.37d3,5*0./
C     data a63/1.56d1,1.07d2,5.22d2,6.06d3,5*0./
      data a62/6.94d0,5.36d1,2.12d2,2.37d3,6.22d3,4*0./
      data a63/1.56d1,1.306d2,5.22d2,6.06d3,1.70d4,4*0./
      data om12/0.,0.401,0.545,1.408,2.33,2.94,1.15,0.34,10.3/
      data om13/0.,0.279,0.271,1.81,1.11,1.84,1.00,0.30,3.96/
      data om23/0.,1.13,1.29,5.832,5.41,7.81,3.70,1.08,20.3/
      data om123t4/1.14,2.68,2.29,2.1,7.95,3.21,4.25,0.86,
     &0.0/
      data om23t5/0.252,0.352,0.292,0.246,1.15,0.559,0.86,
     &0.18,0.0/
      data om45/0.277,0.411,0.582,0.65,1.30,1.70,0.63,0.4,
     &0.668/
      data om23t6/0.671,1.28,1.18,1.43,5*1.d-10/
      data al12/3*0.,-.328,-.132,-.24,.142,0.11,-.684/
      data al13/2*0.,.044,-.494,0.07,-.155,.161,0.11,-.544/
      data al23/2*0.,.068,-.443,0.,-.194,.153,0.10,-.598/
      data al123t4/.969,0.,.129,0.,0.,.142,-.082,0.095,0./
      data al23t5/.896,0.,.287,2*0.,.103,-.233,-.074,0./
      data al45/.500,.035,.085,0.,.275,0.,.279,-.13,-.101/
      data al23t6/.616,0.,.120,6*0./
      data a0/-9.25141,-7.69735,-7.4387,444.6,350.609,386.186/
      data a1/-.773782,-1.30743,-.57443,-227.913,-187.474,
     &-202.192/
      data a2/.361184,.697638,.358264,42.5952,36.1803,38.5049/
      data a3/-.0150892,-.111338,-.0418166,-3.4762,-3.03283,
     &-3.19268/
      data a4/-.000656325,-.00705277,-.00235272,.105085,
     &.0938138,.0978573/
      data g/1.,3.,5.,5.,1.,5./
      data ama/12.,14.,16.,20.,32.1,39.948,40.08,24.305,55.847/
C
      cool=0.0d0
      f21=0.0d0
      f32=0.0d0
      f4t123=0.0d0
      f5t23=0.0d0
      f6t23=0.0d0
      f54=0.0d0
      if(abund.ge.1.d-10) then
c
      tev=te/1.1609d4
      t4=te/1.d4
      telog=dlog10(te)
      th=dmax1(telog,3.4d0)
      th=dmin1(5.3d0,th)
      tmx=1.25d5
      cons=8.63d-6*dene/dsqrt(te)
c
      e21=ene2(i)
      e31=ene3(i)
      e41=ene4(i)
      e51=ene5(i)
      e61=ene6(i)
      e32=ene3(i)-ene2(i)
      e42=ene4(i)-ene2(i)
      e52=ene5(i)-ene2(i)
      e62=ene6(i)-ene2(i)
      e43=ene4(i)-ene3(i)
      e53=ene5(i)-ene3(i)
      e63=ene6(i)-ene3(i)
      e54=ene5(i)-ene4(i)
      if(i.eq.1) then
       y=dmin1(dlog(te),9.2103d0)
       ipc=1
       if(te.gt.1.d3) ipc=2
        do 5001 ip=1,3
        yy(ip)=a0(ip,ipc)+a1(ip,ipc)*y+a2(ip,ipc)*y**2+
     &  a3(ip,ipc)*y**3+a4(ip,ipc)*y**4
 5001   continue
       o12=dexp(yy(1))
       o13=dexp(yy(2))
       o23=dexp(yy(3))
      elseif(i.eq.2) then
       o12=cheb1(tmx,te,-1.534d0,0.1436d0,0.0107d0,-0.0076d0)
       o13=cheb1(tmx,te,-2.4004d0,0.1962d0,-0.044d0,-0.0296d0)
       o23=cheb1(tmx,te,0.4597d0,0.1717d0,-0.0189d0,-0.0198d0)
      else
       o12=om12(i)*t4**al12(i)
       o13=om13(i)*t4**al13(i)
       o23=om23(i)*t4**al23(i)
      endif
      c21=cons*o12/g(2)
      c31=cons*o13/g(3)
      c32=cons*o23/g(3)
      c12=c21*expfn(e21/tev)*g(2)/g(1)
      c13=c31*expfn(e31/tev)*g(3)/g(1)
      c23=c32*expfn(e32/tev)*g(3)/g(2)
C
       IF((E43/TEV).lt.2.d1) then
      gsum=g(1)+g(2)+g(3)
      if(i.eq.3) then
       fc=(g(2)+g(3))/gsum
       o123t4=35.674-25.2742*th+6.2041*th**2-0.4944*th**3
       o23t5=fc*(5.9018-4.1547*th+1.0034*th**2-0.0788*th**3)
       o23t6=fc*(14.1717-10.5248*th+2.7476*th**2-0.233*th**3)
       o45=42.233-40.2293*th+14.2334*th**2-2.196*th**3+0.125*
     & th**4
      elseif(i.eq.5) then
       fc=(g(2)+g(3))/gsum
       if(te.le.3.d4) then
         o123t4=7.37d0*(te/3.d4)**0.0146
         o23t5=fc*1.44d0*(te/3.d4)**0.0614
         o23t6=fc*3.35d0*(te/3.d4)**0.112
       else
	 o123t4=7.37d0*(te/3.d4)**(-0.176)
         o23t5=fc*1.44d0*(te/3.d4)**(-0.124)
         o23t6=fc*3.35d0*(te/3.d4)**(-0.158)
       endif
       if(te.le.6.d4) then
         o45=1.88d0*(te/6.d4)**0.116
       else
         o45=1.88d0*(te/6.d4)**(-0.085)
       endif
      elseif(i.eq.2) then
       o14=cheb1(tmx,te,-2.1882d0,-0.0023d0,0.0082d0,-0.0005d0)
       o15=cheb1(tmx,te,-6.3792d0,-0.0076d0,0.019d0,-0.0013d0)
       o16=cheb1(tmx,te,-4.1428d0,-0.0268d0,-0.0218d0,-0.0125d0)
       o24=cheb1(tmx,te,0.009d0,-0.0023d0,0.0082d0,-0.0006d0)
       o25=cheb1(tmx,te,-4.182d0,-0.0076d0,0.019d0,-0.0013d0)
       o26=cheb1(tmx,te,-1.9489d0,-0.0296d0,-0.0239d0,-0.0139d0)
       o34=cheb1(tmx,te,1.0307d0,-0.0023d0,0.0082d0,-0.0006d0)
       o35=cheb1(tmx,te,-3.1604d0,-0.0076d0,0.019d0,-0.0013d0)
       o36=cheb1(tmx,te,-0.9239d0,-0.0268d0,-0.0217d0,-0.0125d0)
       o45=cheb1(tmx,te,-1.2463d0,0.0961d0,0.0291d0,-0.0024d0)
      elseif(i.eq.9) then
       o14=4.11d0*t4**(-.367)
       o15=0.32d0*t4**(-.705)
       o16=1.d-10
       o24=2.84d0*t4**(-.594)
       o25=0.36d0*t4**(-.655)
       o26=1.d-10
       o34=10.5d0*t4**(-.589)
       o35=0.20d0*t4**(-.266)
       o36=1.d-10
       o45=0.668d0*t4**(-.101)
      else
       fc=(g(2)+g(3))/gsum
       o123t4=om123t4(i)*t4**al123t4(i)
       o23t5=fc*om23t5(i)*t4**al23t5(i)
       o23t6=fc*om23t6(i)*t4**al23t6(i)
       o45=om45(i)*t4**al45(i)
      endif
c
      if(i.eq.1.or.i.ge.3) then
       if(i.le.8) then
       o14=g(1)*o123t4/gsum
       o24=g(2)*o14/g(1)
       o34=g(3)*o14/g(1)
       o25=g(2)*o23t5/(g(2)+g(3))
       o15=g(1)*o25/g(2)
       o35=g(3)*o25/g(2)
       o26=g(2)*o23t6/(g(2)+g(3))
       o16=g(1)*o26/g(2)
       o36=g(3)*o26/g(2)
       endif
      endif
      c41=cons*o14/g(4)
      c51=cons*o15/g(5)
      c61=cons*o16/g(6)
      c42=cons*o24/g(4)
      c52=cons*o25/g(5)
      c62=cons*o26/g(6)
      c43=cons*o34/g(4)
      c53=cons*o35/g(5)
      c63=cons*o36/g(6)
      c54=cons*o45/g(5)
      c14=c41*expfn(e41/tev)*g(4)/g(1)
      c15=c51*expfn(e51/tev)*g(5)/g(1)
      c16=c61*expfn(e61/tev)*g(6)/g(1)
      c24=c42*expfn(e42/tev)*g(4)/g(2)
      c25=c52*expfn(e52/tev)*g(5)/g(2)
      c26=c62*expfn(e62/tev)*g(6)/g(2)
      c34=c43*expfn(e43/tev)*g(4)/g(3)
      c35=c53*expfn(e53/tev)*g(5)/g(3)
      c36=c63*expfn(e63/tev)*g(6)/g(3)
      c45=c54*expfn(e54/tev)*g(5)/g(4)
       ENDIF
c
      amass=ama(i)
      m=0
 9988 continue
      m=m+1
      adamp=a21(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(2)*g(1))/(x(1)*g(2)))
      call escacf(e21,a21(i),adamp,g(1),g(2),te,amass,tcorr,
     &coltoti,coltot,p21)
      adamp=a31(i)+a32(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(3)*g(1))/(x(1)*g(3)))
      call escacf(e31,a31(i),adamp,g(1),g(3),te,amass,tcorr,
     &coltoti,coltot,p31)
      adamp=a21(i)+a32(i)+a31(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(3)*g(2))/(x(2)*g(3)))
      call escacf(e32,a32(i),adamp,g(2),g(3),te,amass,tcorr,
     &coltoti,coltot,p32)
       IF((E43/TEV).lt.2.d1) then
      adamp=a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(4)*g(1))/(x(1)*g(4)))
      call escacf(e41,a41(i),adamp,g(1),g(4),te,amass,tcorr,
     &coltoti,coltot,p41)
      adamp=a21(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(4)*g(2))/(x(2)*g(4)))
      call escacf(e42,a42(i),adamp,g(2),g(4),te,amass,tcorr,
     &coltoti,coltot,p42)
      adamp=a21(i)+a52(i)+a54(i)+a53(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(5)*g(2))/(x(2)*g(5)))
      call escacf(e52,a52(i),adamp,g(2),g(5),te,amass,tcorr,
     &coltoti,coltot,p52)
      adamp=a31(i)+a32(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(4)*g(3))/(x(3)*g(4)))
      call escacf(e43,a43(i),adamp,g(3),g(4),te,amass,tcorr,
     &coltoti,coltot,p43)
      adamp=a52(i)+a54(i)+a53(i)+a31(i)+a32(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(5)*g(3))/(x(3)*g(5)))
      call escacf(e53,a53(i),adamp,g(3),g(5),te,amass,tcorr,
     &coltoti,coltot,p53)
      adamp=a52(i)+a54(i)+a53(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(4)*(1.d0-(x(5)*g(4))/(x(4)*g(5)))
      call escacf(e54,a54(i),adamp,g(4),g(5),te,amass,tcorr,
     &coltoti,coltot,p54)
      adamp=a62(i)+a63(i)+a21(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(6)*g(2))/(x(2)*g(6)))
      call escacf(e62,a62(i),adamp,g(2),g(6),te,amass,tcorr,
     &coltoti,coltot,p62)
      adamp=a62(i)+a63(i)+a31(i)+a32(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(6)*g(3))/(x(3)*g(6)))
      call escacf(e63,a63(i),adamp,g(3),g(6),te,amass,tcorr,
     &coltoti,coltot,p63)
      a(1,1)=c12+c13+c14+c15+c16
      a(1,2)=-(c21+p21*a21(i))
      a(1,3)=-(c31+p31*a31(i))
      a(1,4)=-(c41+p41*a41(i))
      a(1,5)=-c51
      a(1,6)=-c61
      a(2,1)=-c12
      a(2,2)=c21+c23+c24+c25+c26+p21*a21(i)
      a(2,3)=-(c32+p32*a32(i))
      a(2,4)=-(c42+p42*a42(i))
      a(2,5)=-(c52+p52*a52(i))
      a(2,6)=-(c62+p62*a62(i))
      do 100 j=1,6
      x(j)=0.d0
      a(3,j)=1.d0
  100 continue
      x(3)=1.d0
      a(4,1)=-c14
      a(4,2)=-c24
      a(4,3)=-c34
      a(4,4)=c41+c42+c43+c45+p41*a41(i)+p42*a42(i)+p43*a43(i)
      a(4,5)=-(c54+p54*a54(i))
      a(4,6)=0.0d0
      a(5,1)=-c15
      a(5,2)=-c25
      a(5,3)=-c35
      a(5,4)=-c45
      a(5,5)=c51+c52+c53+c54+p52*a52(i)+p53*a53(i)+p54*a54(i)
      a(5,6)=0.0d0
      a(6,1)=-c16
      a(6,2)=-c26
      a(6,3)=-c36
      a(6,4)=0.0d0
      a(6,5)=0.0d0
      a(6,6)=c61+c62+c63+p62*a62(i)+p63*a63(i)
c
      call matrix(6)
       ELSE
      a(1,1)=c12+c13
      a(1,2)=-(c21+p21*a21(i))
      a(1,3)=-(c31+p31*a31(i))
      a(2,1)=-c12
      a(2,2)=c21+c23+p21*a21(i)
      a(2,3)=-(c32+p32*a32(i))
      a(3,1)=1.d0
      a(3,2)=1.d0
      a(3,3)=1.d0
      x(1)=0.0d0
      x(2)=0.0d0
      x(3)=1.d0
c
      call matrix(3)
       ENDIF
      if(m.lt.2) goto 9988
c
      elab=(1.6022d-12*abund)/(12.57d0*dene)
       IF((E43/TEV).lt.2.d1) then
      cool=elab*(e21*(x(1)*c12-x(2)*c21)+e31*
     &(x(1)*c13-x(3)*c31)+e41*(x(1)*c14-x(4)*c41)+e51*(x(1)*c15-
     &x(5)*c51)+e61*(x(1)*c16-x(6)*c61)+e32*(x(2)*c23-x(3)*c32)+
     &e42*(x(2)*c24-x(4)*c42)+e52*(x(2)*c25-x(5)*c52)+e62*(x(2)*
     &c26-x(6)*c62)+e43*(x(3)*c34-x(4)*c43)+e53*(x(3)*c35-x(5)*
     &c53)+e63*(x(3)*c36-x(6)*c63)+e54*(x(4)*c45-x(5)*c54))
      f4t123=elab*x(4)*(p41*e41*a41(i)+p42*e42*a42(i)+
     &p43*e43*a43(i))
      f5t23=elab*x(5)*(p52*e52*a52(i)+p53*e53*a53(i))
      f6t23=elab*x(6)*(p62*e62*a62(i)+p63*e63*a63(i))
      f54=elab*x(5)*e54*a54(i)*p54
       ELSE
      cool=elab*(e21*(x(1)*c12-x(2)*c21)+e31*(x(1)*c13-x(3)*c31)+
     &e32*(x(2)*c23-x(3)*c32))
       ENDIF
      cool=12.57d0*cool
      f21=elab*x(2)*a21(i)*e21*p21
      f32=elab*x(3)*a32(i)*e32*p32
c
      endif
C
      RETURN
      END
C*********************************************************************
C*********************************************************************
      SUBROUTINE NFORB(TE,I,DENE,ABUND,COLTOTI,COLTOT,F23T1,F45T1,
     &F45T23,COOL)
C*********************************************************************
C  Forbidden lines (N I-like). Enumeration of the ions are: (= I)
C
C  1 = N I  2 = O II  3 = Ne IV  4 = S II  5 = Na V  6 = Ar IV
C  7 = Mg VI  8 = Si VIII
C
C  The atom is treated as a 5-level atom.
C  No levels are calculated if E21/TEV > 20.
C
C  Note that some of the levels have inverted energy levels !
C  N I Omegas Mendoza ('83) and A-values from Kaufman & Sugar (J.Phys.
C  Chem. '86). O II, Ne IV & Ar IV from Osterbrock ('89). 
C  O II & Ne IV checked against the original papers, and small changes
C  may occur. O II Omegas updated with McLaughlin & Bell (J.Phys.B '93).
C  New A-values for S II (Keenan et al., Phys. Scripta 48, 129 (1993). 
C  S II (Omegas) updated with Ramsbottom, Bell & Stafford (1996) [Different
C  from Cai & Pradhan, ApjS 88, 329 (1993)]. 
C  Na V from Mendoza ('83). A-values, energies and 10Ryd-Omegas for 
C  Mg VI and Si VIII from Bhatia & Manson (MNRAS 190, 925, '80).
C  (From isoel.scaling this actually gives rather good Omegas.)
C*********************************************************************
      implicit real*8(a-h,o-z)
C
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      common/robcum/fro21,fro41,frs21,frs41,frr21,frr41
c
      dimension ene2(8),ene3(8),ene4(8),ene5(8),a21(8),a31(8),
     &a32(8),a41(8),a42(8),a43(8),a51(8),a52(8),a53(8),a54(8),
     &om12(8),om13(8),om14(8),om15(8),om23(8),om24(8),om25(8),
     &om34(8),om35(8),om45(8),al12(8),al13(8),al14(8),al15(8),
     &al23(8),al24(8),al25(8),al34(8),al35(8),al45(8),g(5),ama(8)
c
      data ene2/2.381916,3.321846,5.108967,1.840287,5.9860,
     &2.6131,7.2166,8.9326/
      data ene3/2.382994,3.324262,5.114530,1.844227,5.9917,
     &2.6291,7.2171,8.9725/
      data ene4/3.573140,5.013923,7.735647,3.038635,9.0697,
     &4.3186,10.233,12.878/
      data ene5/3.573189,5.014060,7.736477,3.044446,9.0740,
     &4.3405,10.243,12.943/
      data a21/6.92d-6,3.54d-5,4.84d-4,8.9d-4,1.46d-3,2.23d-2,
     &0.145d0,1.96d0/
      data a31/1.62d-5,1.80d-4,5.54d-3,2.73d-4,2.69d-2,1.77d-3,
     &3.11d-3,3.75d-2/
      data a41/2.46d-3,5.75d-2,5.21d-1,9.06d-2,1.76d0,8.62d-1,
     &4.92d0,29.9d0/
      data a51/6.18d-3,2.36d-2,1.27d0,2.25d-1,4.27d0,2.6d0,
     &12.d0,71.2d0/
      data a32/1.07d-8,1.31d-7,1.48d-6,3.35d-7,1.55d-6,2.3d-5,
     &3.5d-5,3.54d-4/
      data a42/3.45d-2,1.07d-1,1.15d-1,1.41d-1,1.41d-1,6.03d-1,
     &1.85d0,11.4d0/
      data a52/5.48d-2,5.63d-2,4.00d-1,1.21d-1,9.18d-1,7.89d-1,
     &2.99d0,19.d0/
      data a43/4.71d-2,5.79d-2,3.93d-1,6.27d-2,9.55d-1,1.19d-1,
     &7.67d-2,0.119d0/
      data a53/2.47d-2,9.40d-2,4.37d-1,1.52d-1,1.29d0,5.98d1,
     &1.69d0,10.5d0/
      data a54/5.17d-13,1.35d-10,2.68d-9,1.03d-6,3.64d-7,
     &4.94d-5,4.36d-6,1.29d-3/
      data om12/.290,.825,.838,2.76,.551,1.30,.19,.15/
      data om13/.194,.55,.559,4.14,.368,1.94,.22,.18/
      data om14/.0567,.276,.156,1.17,.12,.147,.058,.048/
      data om15/.113,.138,.313,2.35,.239,.293,.10,.084/
      data om23/.269,1.17,1.36,7.47,.696,6.13,.28,.22/
      data om24/.109,.882,.368,1.79,.201,1.67,.15,.10/
      data om25/.266,.356,.900,3.00,.502,2.47,.19,.13/
      data om34/.097,.493,.336,2.20,.19,1.79,.14,.10/
      data om35/.147,.332,.509,4.99,.279,4.44,.37,.26/
      data om45/.071,.287,.343,2.71,.438,2.33,.15,.12/
      data al12/.904,.038,0.,-.161,4*0./
      data al13/.913,.038,0.,-.160,4*0./
      data al14/.928,.0738,0.,-.104,4*0./
      data al15/.921,.0738,0.,-.101,4*0./
      data al23/1.07,2*0.,-.135,4*0./
      data al24/.800,.1357,.054,-.153,4*0./
      data al25/.715,.1357,0.,-.120,4*0./
      data al34/.691,.1357,0.,-.115,4*0./
      data al35/.780,.1357,0.,-.141,4*0./
      data al45/1.110,.049,.109,-.158,4*0./
      data ama/14.,16.,20.,32.1,22.99,39.948,24.305,28.086/
C
      cool=0.0d0
      f23t1=0.0d0
      f45t1=0.0d0
      f45t23=0.0d0
      frr21=0.0d0
      frr41=0.0d0
      if(abund.ge.1.d-10) then
c
      tev=te/1.1609d4
      t4=te/1.d4
      telog=dlog10(te)
      th=telog
      if(th.le.3.d0) th=3.d0
      if(th.ge.4.6d0) th=4.6d0
      cons=8.63d-6*dene/dsqrt(te)
c
      g(1)=4.d0
      g(2)=4.d0
      g(3)=6.d0
      if(i.eq.1.or.i.eq.2) then
       g(2)=6.d0
       g(3)=4.d0
      endif
      if(i.eq.3.or.i.eq.5) then
       g(2)=6.d0
       g(3)=4.d0
      endif
      if(i.ne.2) then
       g(4)=2.d0
       g(5)=4.d0
      else
       g(4)=4.d0
       g(5)=2.d0
      endif
      e21=ene2(i)
       IF((E21/TEV).lt.2.d1) THEN
      e31=ene3(i)
      e41=ene4(i)
      e51=ene5(i)
      e32=ene3(i)-ene2(i)
      e42=ene4(i)-ene2(i)
      e52=ene5(i)-ene2(i)
      e43=ene4(i)-ene3(i)
      e53=ene5(i)-ene3(i)
      e54=ene5(i)-ene4(i)
      if(i.ne.4) then
       o12=om12(i)*t4**al12(i)
       o13=om13(i)*t4**al13(i)
       o14=om14(i)*t4**al14(i)
       o15=om15(i)*t4**al15(i)
       o23=om23(i)*t4**al23(i)
       o24=om24(i)*t4**al24(i)
       o25=om25(i)*t4**al25(i)
       o34=om34(i)*t4**al34(i)
       o35=om35(i)*t4**al35(i)
       o45=om45(i)*t4**al45(i)
      else
       o12=3.9949-1.1021*th+0.5487*th**2-0.0823*th**3
       o13=6.1758-1.836*th+0.8798*th**2-0.1291*th**3
       o14=13.1138-10.7869*th+2.9634*th**2-0.2608*th**3
       o15=26.7492-22.005*th+6.0445*th**2-0.5322*th**3
       o23=3.2171+3.0844*th-0.3551*th**2-0.038*th**3
       o24=7.0174-5.7352*th+1.8524*th**2-0.187*th**3
       o25=13.3032-10.9238*th+3.4266*th**2-0.3392*th**3
       o34=10.6418-8.6917*th+2.6787*th**2-0.2618*th**3
       o35=23.5642-19.3646*th+6.0722*th**2-0.602*th**3
       o45=37.3916-31.082*th+8.6052*th**2-0.7627*th**3
      endif
      c21=cons*o12/g(2)
      c31=cons*o13/g(3)
      c41=cons*o14/g(4)
      c51=cons*o15/g(5)
      c32=cons*o23/g(3)
      c42=cons*o24/g(4)
      c52=cons*o25/g(5)
      c43=cons*o34/g(4)
      c53=cons*o35/g(5)
      c54=cons*o45/g(5)
      c12=c21*expfn(e21/tev)*g(2)/g(1)
      c13=c31*expfn(e31/tev)*g(3)/g(1)
      c14=c41*expfn(e41/tev)*g(4)/g(1)
      c15=c51*expfn(e51/tev)*g(5)/g(1)
      c23=c32*expfn(e32/tev)*g(3)/g(2)
      c24=c42*expfn(e42/tev)*g(4)/g(2)
      c25=c52*expfn(e52/tev)*g(5)/g(2)
      c34=c43*expfn(e43/tev)*g(4)/g(3)
      c35=c53*expfn(e53/tev)*g(5)/g(3)
      c45=c54*expfn(e54/tev)*g(5)/g(4)
c
      amass=ama(i)
      m=0
 9988 continue
      m=m+1
      adamp=a21(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(2)*g(1))/(x(1)*g(2)))
      call escacf(e21,a21(i),adamp,g(1),g(2),te,amass,tcorr,
     &coltoti,coltot,p21)
      adamp=a31(i)+a32(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(3)*g(1))/(x(1)*g(3)))
      call escacf(e31,a31(i),adamp,g(1),g(3),te,amass,tcorr,
     &coltoti,coltot,p31)
      adamp=a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(4)*g(1))/(x(1)*g(4)))
      call escacf(e41,a41(i),adamp,g(1),g(4),te,amass,tcorr,
     &coltoti,coltot,p41)
      adamp=a41(i)+a42(i)+a43(i)+a51(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(5)*g(1))/(x(1)*g(5)))
      call escacf(e51,a51(i),adamp,g(1),g(5),te,amass,tcorr,
     &coltoti,coltot,p51)
      adamp=a21(i)+a32(i)+a31(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(3)*g(2))/(x(2)*g(3)))
      call escacf(e32,a32(i),adamp,g(2),g(3),te,amass,tcorr,
     &coltoti,coltot,p32)
      adamp=a21(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(4)*g(2))/(x(2)*g(4)))
      call escacf(e42,a42(i),adamp,g(2),g(4),te,amass,tcorr,
     &coltoti,coltot,p42)
      adamp=a21(i)+a51(i)+a52(i)+a54(i)+a53(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(5)*g(2))/(x(2)*g(5)))
      call escacf(e52,a52(i),adamp,g(2),g(5),te,amass,tcorr,
     &coltoti,coltot,p52)
      adamp=a31(i)+a32(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(4)*g(3))/(x(3)*g(4)))
      call escacf(e43,a43(i),adamp,g(3),g(4),te,amass,tcorr,
     &coltoti,coltot,p43)
      adamp=a51(i)+a52(i)+a54(i)+a53(i)+a31(i)+a32(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(5)*g(3))/(x(3)*g(5)))
      call escacf(e53,a53(i),adamp,g(3),g(5),te,amass,tcorr,
     &coltoti,coltot,p53)
      adamp=a51(i)+a52(i)+a54(i)+a53(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(4)*(1.d0-(x(5)*g(4))/(x(4)*g(5)))
      call escacf(e54,a54(i),adamp,g(4),g(5),te,amass,tcorr,
     &coltoti,coltot,p54)
c
      a(1,1)=c12+c13+c14+c15
      a(1,2)=-(c21+p21*a21(i))
      a(1,3)=-(c31+p31*a31(i))
      a(1,4)=-(c41+p41*a41(i))
      a(1,5)=-(c51+p51*a51(i))
      a(2,1)=-c12
      a(2,2)=c21+c23+c24+c25+p21*a21(i)
      a(2,3)=-(c32+p32*a32(i))
      a(2,4)=-(c42+p42*a42(i))
      a(2,5)=-(c52+p52*a52(i))
      do 100 j=1,5
      x(j)=0.d0
      a(3,j)=1.d0
  100 continue
      x(3)=1.d0
      a(4,1)=-c14
      a(4,2)=-c24
      a(4,3)=-c34
      a(4,4)=c41+c42+c43+c45+p41*a41(i)+p42*a42(i)+p43*a43(i)
      a(4,5)=-(c54+p54*a54(i))
      a(5,1)=-c15
      a(5,2)=-c25
      a(5,3)=-c35
      a(5,4)=-c45
      a(5,5)=c51+c52+c53+c54+p51*a51(i)+p52*a52(i)+p53*a53(i)+
     &p54*a54(i)
c
      call matrix(5)
      if(m.lt.2) goto 9988
c
      elab=(1.6022d-12*abund)/(12.57d0*dene)
      cool=elab*(e21*(x(1)*c12-x(2)*c21)+e31*
     &(x(1)*c13-x(3)*c31)+e41*(x(1)*c14-x(4)*c41)+e51*(x(1)*c15-
     &x(5)*c51)+e32*(x(2)*c23-x(3)*c32)+
     &e42*(x(2)*c24-x(4)*c42)+e52*(x(2)*c25-x(5)*c52)+
     &e43*(x(3)*c34-x(4)*c43)+e53*(x(3)*c35-x(5)*c53)+
     &e54*(x(4)*c45-x(5)*c54))
      cool=12.57d0*cool
      f23t1=elab*(x(2)*p21*a21(i)*e21+x(3)*p31*a31(i)*e31)
      f45t1=elab*(x(4)*p41*a41(i)*e41+x(5)*p51*a51(i)*e51)
      f45t23=elab*(x(4)*(e42*a42(i)*p42+e43*a43(i)*p43)+x(5)*
     &(e52*a52(i)*p52+e53*a53(i)*p53))
C - line ratios -
      frr21=(elab*x(2)*p21*a21(i)*e21)/f23t1
      if(i.ne.2) then
       frr41=(elab*x(4)*p41*a41(i)*e41)/f45t1
      else
       frr41=(elab/f45t23)*(x(4)*e43*a43(i)*p43+x(5)*e53*a53(i)*p53)
      endif
c
       ENDIF
      endif
C
      RETURN
      END
C*********************************************************************
C*********************************************************************
      SUBROUTINE OFORB(TE,I,DENE,ABUND,COLTOTI,COLTOT,F21,F32,F4T123,
     &F5T12,F54,COOL)
C*********************************************************************
C  Forbidden lines (O I-like). Enumeration of the ions are: (= I)
C
C  1 = O I  2 = Ne III  3 = S I  4 = Na IV  5 = Mg V  6 = Ar III  
C  7 = Ca V
C
C  The atom is treated as a 5-level atom. (The energy levels of lower
C  3P-state are inverted compared to C I-like states.)
C  (The atom is reduced to a 3-level atom for E43/Tev > 20).
C
C  ene4 & ene5 of Ar III from Hansen & Persson ('87). Other energies 
C  from Mendoza ('83). A-values and Omegas from Mendoza ('83) and
C  Osterbrock ('89), except Omegas of Ar III and Ca V which are from
C  Galavis et al. (AAS, 111, 347, '95), and of O I (Berrington, 
C  J.Phys.B 21, 1083, '87). S I Omegas are assumed to follow those of 
C  O I, which is probably an underestimate.
C
C  A-values of O, Ne, Na and Mg updated with Galavis et al. 1997, A&AS,
C  123, 159.
C*********************************************************************
      implicit real*8(a-h,o-z)
C
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
c
      dimension ene2(7),ene3(7),ene4(7),ene5(7),a21(7),a31(7),
     &a41(7),a51(7),a32(7),a42(7),a52(7),a43(7),a54(7),
     &om12(7),om13(7),om23(7),om123t4(7),om123t5(7),om45(7),
     &al12(7),al13(7),al23(7),al123t4(7),al123t5(7),al45(7),g(5),
     &ama(7)
c
      data ene2/.01965,.07966,.04908,.13707,.22093,.13779,.29794/
      data ene3/.02809,.1141,.07108,.19522,.31245,.19455,.40585/
      data ene4/1.9660,3.2017,1.1447,3.8211,4.4520,1.6685,2.3331/
      data ene5/4.1869,6.9075,2.7481,8.2389,9.5757,4.0455,5.4313/
      data a21/8.86d-5,5.97d-3,1.39d-3,3.05d-2,.127d0,3.08d-2,
     &.31d0/
      data a31/1.28d-10,2.08d-8,6.71d-8,1.60d-7,9.68d-7,2.37d-6,
     &3.67d-5/
      data a32/1.77d-5,1.16d-3,3.02d-4,5.56d-3,2.17d-2,5.17d-3,
     &3.54d-2/
      data a41/6.51d-3,.173d0,2.78d-2,.61d0,1.87d0,0.314d0,1.9d0/
      data a42/2.11d-3,5.34d-2,8.16d-3,.184d0,.535d0,8.23d-2,
     &.426d0/
      data a43/6.39d-7,8.27d-6,3.84d-6,2.20d-5,5.15d-5,2.21d-5,
     &8.42d-5/
      data a51/2.94d-4,3.98d-3,8.23d-3,1.07d-2,2.50d-2,4.17d-2,
     &.145d0/
      data a52/7.91d-2,2.03,.35d0,7.15d0,21.6d0,3.91d0,23.1d0/
      data a54/1.12d0,2.56d0,1.53d0,3.32d0,4.09d0,2.59d0,3.73d0/
      data om12/0.,.527,0.,.471,.4,3.1,2.3/
      data om13/0.,.131,0.,.111,.091,.671,.648/
      data om23/0.,.185,0.,.177,.156,1.261,.671/
      data om123t4/0.,1.34,0.,1.17,1.02,4.825,3.067/
      data om123t5/0.,.151,0.,.163,.146,.841,.522/
      data om45/0.,.236,0.,.157,.129,1.219,1.348/
      data al12/5*0.,.034,.163/
      data al13/5*0.,.16,.176/
      data al23/5*0.,-.098,.141/
      data al123t4/5*0.,-.065,.104/
      data al123t5/5*0.,-.099,.185/
      data al45/0.,.184,3*0.,-.152,0./
      data g/5.,3.,1.,5.,1./
      data ama/16.,20.,32.1,22.99,24.305,39.948,40.08/
C
      cool=0.0d0
      f21=0.0d0
      f32=0.0d0
      f4t123=0.0d0
      f5t12=0.0d0
      f54=0.0d0
      if(abund.ge.1.d-10) then
c
      tev=te/1.1609d4
      t4=te/1.d4
      telog=dlog10(te)
      th=dmax1(telog,3.4d0)
      th=dmin1(5.d0,th)
      cons=8.63d-6*dene/dsqrt(te)
c
      e21=ene2(i)
      e31=ene3(i)
      e41=ene4(i)
      e51=ene5(i)
      e32=ene3(i)-ene2(i)
      e42=ene4(i)-ene2(i)
      e52=ene5(i)-ene2(i)
      e43=ene4(i)-ene3(i)
      e53=ene5(i)-ene3(i)
      e54=ene5(i)-ene4(i)
c
      gsum=g(1)+g(2)+g(3)
C -- S I omegas assumed to follow those of O I (probably an
C    underestimate !) --
c -- O I from Berrington, J.Phys.B 21, 1083 (1987) --
      if(i.eq.1.or.i.eq.3) then
       o45=1.05d-1*t4**0.520d0
       if(te.le.1.d3) then
        o12=9.72d-3*(te/1.d3)**.773d0
        o13=4.11d-3*(te/1.d3)**.676d0
        o23=6.36d-4*(te/1.d3)**1.301d0
        o123t4=1.51d-2*(te/1.d3)**1.380d0
        o123t5=1.84d-3*(te/1.d3)**1.501d0
       else 
        o12=9.72d-3*(te/1.d3)**1.038d0
        o13=4.11d-3*(te/1.d3)**.893d0
        o23=6.36d-4*(te/1.d3)**1.648d0
        o123t4=2.66d-1*t4**1.246d0
        o123t5=3.24d-2*t4**1.246d0
       endif
      else
       o12=om12(i)*t4**al12(i)
       o13=om13(i)*t4**al13(i)
       o23=om23(i)*t4**al23(i)
       o123t4=om123t4(i)*t4**al123t4(i)
       o123t5=om123t5(i)*t4**al123t5(i)
       o45=om45(i)*t4**al45(i)
      endif
       if(i.eq.7.and.o123t5.gt.0.8d0) o123t5=0.8d0
      c21=cons*o12/g(2)
      c31=cons*o13/g(3)
      c32=cons*o23/g(3)
      c12=c21*expfn(e21/tev)*g(2)/g(1)
      c13=c31*expfn(e31/tev)*g(3)/g(1)
      c23=c32*expfn(e32/tev)*g(3)/g(2)
       IF((E43/TEV).lt.2.d1) then
      o14=g(1)*o123t4/gsum
      o24=g(2)*o14/g(1)
      o34=g(3)*o14/g(1)
      o25=g(2)*o123t5/gsum
      o15=g(1)*o25/g(2)
      o35=g(3)*o25/g(2)
      c41=cons*o14/g(4)
      c51=cons*o15/g(5)
      c42=cons*o24/g(4)
      c52=cons*o25/g(5)
      c43=cons*o34/g(4)
      c53=cons*o35/g(5)
      c54=cons*o45/g(5)
      c14=c41*expfn(e41/tev)*g(4)/g(1)
      c15=c51*expfn(e51/tev)*g(5)/g(1)
      c24=c42*expfn(e42/tev)*g(4)/g(2)
      c25=c52*expfn(e52/tev)*g(5)/g(2)
      c34=c43*expfn(e43/tev)*g(4)/g(3)
      c35=c53*expfn(e53/tev)*g(5)/g(3)
      c45=c54*expfn(e54/tev)*g(5)/g(4)
       ENDIF
c
      amass=ama(i)
      m=0
 9988 continue
      m=m+1
      adamp=a21(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(2)*g(1))/(x(1)*g(2)))
      call escacf(e21,a21(i),adamp,g(1),g(2),te,amass,tcorr,
     &coltoti,coltot,p21)
      adamp=a31(i)+a32(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(3)*g(1))/(x(1)*g(3)))
      call escacf(e31,a31(i),adamp,g(1),g(3),te,amass,tcorr,
     &coltoti,coltot,p31)
      adamp=a21(i)+a32(i)+a31(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(3)*g(2))/(x(2)*g(3)))
      call escacf(e32,a32(i),adamp,g(2),g(3),te,amass,tcorr,
     &coltoti,coltot,p32)
       IF((E43/TEV).lt.2.d1) then
      adamp=a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(4)*g(1))/(x(1)*g(4)))
      call escacf(e41,a41(i),adamp,g(1),g(4),te,amass,tcorr,
     &coltoti,coltot,p41)
      adamp=a51(i)+a52(i)+a54(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(5)*g(1))/(x(1)*g(5)))
      call escacf(e51,a51(i),adamp,g(1),g(5),te,amass,tcorr,
     &coltoti,coltot,p51)
      adamp=a21(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(4)*g(2))/(x(2)*g(4)))
      call escacf(e42,a42(i),adamp,g(2),g(4),te,amass,tcorr,
     &coltoti,coltot,p42)
      adamp=a21(i)+a52(i)+a54(i)+a51(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(5)*g(2))/(x(2)*g(5)))
      call escacf(e52,a52(i),adamp,g(2),g(5),te,amass,tcorr,
     &coltoti,coltot,p52)
      adamp=a31(i)+a32(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(4)*g(3))/(x(3)*g(4)))
      call escacf(e43,a43(i),adamp,g(3),g(4),te,amass,tcorr,
     &coltoti,coltot,p43)
      adamp=a52(i)+a54(i)+a51(i)+a41(i)+a42(i)+a43(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(4)*(1.d0-(x(5)*g(4))/(x(4)*g(5)))
      call escacf(e54,a54(i),adamp,g(4),g(5),te,amass,tcorr,
     &coltoti,coltot,p54)
c
      a(1,1)=c12+c13+c14+c15
      a(1,2)=-(c21+p21*a21(i))
      a(1,3)=-(c31+p31*a31(i))
      a(1,4)=-(c41+p41*a41(i))
      a(1,5)=-(c51+p51*a51(i))
      a(2,1)=-c12
      a(2,2)=c21+c23+c24+c25+p21*a21(i)
      a(2,3)=-(c32+p32*a32(i))
      a(2,4)=-(c42+p42*a42(i))
      a(2,5)=-(c52+p52*a52(i))
      do 100 j=1,5
      x(j)=0.d0
      a(3,j)=1.d0
  100 continue
      x(3)=1.d0
      a(4,1)=-c14
      a(4,2)=-c24
      a(4,3)=-c34
      a(4,4)=c41+c42+c43+c45+p41*a41(i)+p42*a42(i)+p43*a43(i)
      a(4,5)=-(c54+p54*a54(i))
      a(5,1)=-c15
      a(5,2)=-c25
      a(5,3)=-c35
      a(5,4)=-c45
      a(5,5)=c51+c52+c53+c54+p51*a51(i)+p52*a52(i)+p54*a54(i)
c
      call matrix(5)
       ELSE
      a(1,1)=c12+c13
      a(1,2)=-(c21+p21*a21(i))
      a(1,3)=-(c31+p31*a31(i))
      a(2,1)=-c12
      a(2,2)=c21+c23+p21*a21(i)
      a(2,3)=-(c32+p32*a32(i))
      a(3,1)=1.d0
      a(3,2)=1.d0
      a(3,3)=1.d0
      x(1)=0.0d0
      x(2)=0.0d0
      x(3)=1.d0
c
      call matrix(3)
       ENDIF
      if(m.lt.2) goto 9988
c
      elab=(1.6022d-12*abund)/(12.57d0*dene)
       IF((E43/TEV).lt.2.d1) then
      cool=elab*(e21*(x(1)*c12-x(2)*c21)+e31*
     &(x(1)*c13-x(3)*c31)+e41*(x(1)*c14-x(4)*c41)+e51*(x(1)*c15-
     &x(5)*c51)+e32*(x(2)*c23-x(3)*c32)+
     &e42*(x(2)*c24-x(4)*c42)+e52*(x(2)*c25-x(5)*c52)+
     &e43*(x(3)*c34-x(4)*c43)+e53*(x(3)*c35-x(5)*c53)+
     &e54*(x(4)*c45-x(5)*c54))
      f4t123=elab*x(4)*(e41*p41*a41(i)+e42*p42*a42(i)+
     &e43*p43*a43(i))
      if(i.eq.2) then
        f4t123=elab*x(4)*e41*p41*a41(i)
      endif
      f5t12=elab*x(5)*(e51*p51*a51(i)+e52*p52*a52(i))
      f54=elab*x(5)*e54*p54*a54(i)
       ELSE
      cool=elab*(e21*(x(1)*c12-x(2)*c21)+e31*(x(1)*c13-x(3)*c31)+
     &e32*(x(2)*c23-x(3)*c32))
       ENDIF
      cool=12.57d0*cool
      f21=elab*x(2)*p21*a21(i)*e21
      f32=elab*x(3)*p32*a32(i)*e32
c
      endif
C
      RETURN
      END
C*********************************************************************
C*********************************************************************
      SUBROUTINE KFORB(TE,I,DENE,ABUND,COLTOTI,COLTOT,F21,F31,F41,
     &F51,F45T23,F6T45,COOL)
C*********************************************************************
C  Lines (K I-like). Enumeration of the ions are: (= I)
C
C  1 = Ca II
C
C  The atom is treated as a 6-level atom.
C  (Returned to RAD when E21/Tev > 20).
C
C  A-values of Ca II from Ali & Kim (Phys.Rev.A 38, 3992, '88) and 
C  Zeippen (A & A 229, 248, '90) for 4s-3d. (Smaller than Osterbrock's 
C  value, 1.3, from '51 by 30 %.) 4p-5s and 4s-4p from Wiese et al.
C  ('69) scaled by the multiplet calculations of Burgess et al.
C  (A & A 300, 627, '95). 3d-4p from Wiese et al. ('69).
C  A31 from Ritter & Eichmann (J.Phys.B 30, L141 '97). Energies
C  are from the ADS NIST Database ('95). Omegas for most transitions
C  from Burgess et al. ('95), except between the sublevels of 3d which 
C  are from Li & McCray (ApJ 405, 730, '93; based on the results of 
C  Shine ('73)).
C*********************************************************************
      implicit real*8(a-h,o-z)
C
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
c
      dimension ene2(1),ene3(1),ene4(1),ene5(1),ene6(1),
     &a21(1),a31(1),a32(1),a41(1),a42(1),a51(1),a52(1),a53(1),
     &a64(1),a65(1),om1t23(1),om1t45(1),om16(1),om23(1),om23t45(1),
     &om23t6(1),om45t6(1),om45(1),al1t23(1),al1t45(1),al16(1),
     &al23(1),al23t45(1),al23t6(1),al45t6(1),al45(1),g(6),ama(1)
c
      data ene2/1.6926d0/
      data ene3/1.7001d0/
      data ene4/3.1237d0/
      data ene5/3.1514d0/
      data ene6/6.4687d0/
      data a21/0.97d0/
C     data a31/1.00d0/
      data a31/1.03d0/
      data a41/1.36d8/
      data a51/1.40d8/
      data a32/2.45d-6/
      data a42/9.45d6/
      data a52/9.90d5/
      data a53/8.83d6/
      data a64/8.8d7/
      data a65/1.7d8/
      data om1t23/0./
      data om1t45/0./
      data om16/0./
      data om23/21.1d0/
      data om23t45/0./
      data om23t6/0./
      data om45/9.85d0/
      data om45t6/0./
      data al1t23/0./
      data al1t45/0./
      data al16/0./
      data al23/0.01d0/
      data al23t45/0./
      data al23t6/0./
      data al45t6/0./
      data al45/-0.01d0/
      data g/2,4,6,2,4,2/
      data ama/40.08d0/
C
      cool=0.0d0
      f21=0.0d0
      f31=0.0d0
      f41=0.0d0
      f51=0.0d0
      f45t23=0.0d0
      f6t45=0.0d0
      if(abund.ge.1.d-10.and.te.ge.1.d3) then
c
      tev=te/1.1609d4
      t4=te/1.d4
      cons=8.63d-6*dene/dsqrt(te)
c
      e21=ene2(i)
        IF((E21/TEV).lt.2.d1) THEN
      e31=ene3(i)
      e41=ene4(i)
      e51=ene5(i)
      e61=ene6(i)
      e32=ene3(i)-ene2(i)
      e42=ene4(i)-ene2(i)
      e52=ene5(i)-ene2(i)
      e62=ene6(i)-ene2(i)
      e43=ene4(i)-ene3(i)
      e53=ene5(i)-ene3(i)
      e63=ene6(i)-ene3(i)
      e54=ene5(i)-ene4(i)
      e64=ene6(i)-ene4(i)
      e65=ene6(i)-ene5(i)
C
      if(i.eq.1) then
        o23=om23(i)*t4**al23(i)
        o45=om45(i)*t4**al45(i)
        o1t23=upsil(2,0.1248d0,1.4d0,2.56d0,2.954d0,3.762d0,
     &	            4.339d0,5.023d0,te)
        o16=upsil(2,0.4757d0,1.5d0,2.601d0,1.778d0,1.981d0,
     &  	  2.261d0,3.382d0,te)
        o23t6=upsil(2,0.3509d0,4.4d0,0.9722d0,0.7578d0,0.6744d0,
     &  	  0.5929d0,0.4160d0,te)
        o1t45=upsil(1,0.2311d0,2.9d0,2.196d1,2.953d1,3.235d1,
     &	            3.743d1,3.756d1,te)
        o23t45=upsil(1,0.1063d0,5.6d0,4.342d1,3.318d1,2.932d1,
     &	            2.786d1,2.418d1,te)
        o45t6=upsil(1,0.2446d0,2.6d0,7.667d0,5.636d0,7.14d0,
     &	            1.149d1,1.708d1,te)
      endif
      gsum1=g(2)+g(3)
      gsum2=g(4)+g(5)
      gmult=gsum1*gsum2
      o12=(g(2)*o1t23)/gsum1
      o13=(g(3)*o1t23)/gsum1
      o14=(g(4)*o1t45)/gsum2
      o15=(g(5)*o1t45)/gsum2
      o24=(g(2)*g(4)*o23t45)/gmult
      o25=(o24*g(5))/g(4)
      o26=(g(2)*o23t6)/gsum1
      o34=(g(3)*g(4)*o23t45)/gmult
      o35=(o34*g(5))/g(4)
      o36=(g(3)*o23t6)/gsum1
      o46=(g(4)*o45t6)/gsum2
      o56=(g(5)*o45t6)/gsum2
      c21=cons*o12/g(2)
      c31=cons*o13/g(3)
      c41=cons*o14/g(4)
      c51=cons*o15/g(5)
      c61=cons*o16/g(6)
      c32=cons*o23/g(3)
      c42=cons*o24/g(4)
      c52=cons*o25/g(5)
      c62=cons*o26/g(6)
      c43=cons*o34/g(4)
      c53=cons*o35/g(5)
      c63=cons*o36/g(6)
      c54=cons*o45/g(5)
      c64=cons*o46/g(6)
      c65=cons*o56/g(6)
      c12=c21*expfn(e21/tev)*g(2)/g(1)
      c13=c31*expfn(e31/tev)*g(3)/g(1)
      c14=c41*expfn(e41/tev)*g(4)/g(1)
      c15=c51*expfn(e51/tev)*g(5)/g(1)
      c16=c61*expfn(e61/tev)*g(6)/g(1)
      c23=c32*expfn(e32/tev)*g(3)/g(2)
      c24=c42*expfn(e42/tev)*g(4)/g(2)
      c25=c52*expfn(e52/tev)*g(5)/g(2)
      c26=c62*expfn(e62/tev)*g(6)/g(2)
      c34=c43*expfn(e43/tev)*g(4)/g(3)
      c35=c53*expfn(e53/tev)*g(5)/g(3)
      c36=c63*expfn(e63/tev)*g(6)/g(3)
      c45=c54*expfn(e54/tev)*g(5)/g(4)
      c46=c64*expfn(e64/tev)*g(6)/g(4)
      c56=c65*expfn(e65/tev)*g(6)/g(5)
c
      amass=ama(i)
      m=0
 9988 continue
      m=m+1
      adamp=a21(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(2)*g(1))/(x(1)*g(2)))
      call escacf(e21,a21(i),adamp,g(1),g(2),te,amass,tcorr,
     &coltoti,coltot,p21)
      adamp=a31(i)+a32(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(3)*g(1))/(x(1)*g(3)))
      call escacf(e31,a31(i),adamp,g(1),g(3),te,amass,tcorr,
     &coltoti,coltot,p31)
      adamp=a41(i)+a42(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(4)*g(1))/(x(1)*g(4)))
      call escacf(e41,a41(i),adamp,g(1),g(4),te,amass,tcorr,
     &coltoti,coltot,p41)
      adamp=a51(i)+a52(i)+a53(i)
      if(m.eq.1) tcorr=1.d0
      if(m.ne.1) tcorr=x(1)*(1.d0-(x(5)*g(1))/(x(1)*g(5)))
      call escacf(e51,a51(i),adamp,g(1),g(5),te,amass,tcorr,
     &coltoti,coltot,p51)
      adamp=a31(i)+a32(i)+a21(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(3)*g(2))/(x(2)*g(3)))
      call escacf(e32,a32(i),adamp,g(2),g(3),te,amass,tcorr,
     &coltoti,coltot,p32)
      adamp=a21(i)+a41(i)+a42(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(4)*g(2))/(x(2)*g(4)))
      call escacf(e42,a42(i),adamp,g(2),g(4),te,amass,tcorr,
     &coltoti,coltot,p42)
      adamp=a21(i)+a51(i)+a52(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(2)*(1.d0-(x(5)*g(2))/(x(2)*g(5)))
      call escacf(e52,a52(i),adamp,g(2),g(5),te,amass,tcorr,
     &coltoti,coltot,p52)
      adamp=a31(i)+a32(i)+a51(i)+a52(i)+a53(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(3)*(1.d0-(x(5)*g(3))/(x(3)*g(5)))
      call escacf(e53,a53(i),adamp,g(3),g(5),te,amass,tcorr,
     &coltoti,coltot,p53)
      adamp=a41(i)+a42(i)+a64(i)+a65(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(4)*(1.d0-(x(6)*g(4))/(x(4)*g(6)))
      call escacf(e64,a64(i),adamp,g(4),g(6),te,amass,tcorr,
     &coltoti,coltot,p64)
      adamp=a51(i)+a52(i)+a53(i)+a64(i)+a65(i)
      if(m.eq.1) tcorr=0.d0
      if(m.ne.1) tcorr=x(5)*(1.d0-(x(6)*g(5))/(x(5)*g(6)))
      call escacf(e65,a65(i),adamp,g(5),g(6),te,amass,tcorr,
     &coltoti,coltot,p65)
      a(1,1)=c12+c13+c14+c15+c16
      a(1,2)=-(c21+p21*a31(i))
      a(1,3)=-(c31+p31*a31(i))
      a(1,4)=-(c41+p41*a41(i))
      a(1,5)=-(c51+p51*a51(i))
      a(1,6)=-c61
      a(2,1)=-c12
      a(2,2)=c21+c23+c24+c25+c26+p21*a21(i)
      a(2,3)=-(c32+p32*a32(i))
      a(2,4)=-(c42+p42*a42(i))
      a(2,5)=-(c52+p52*a52(i))
      a(2,6)=-c62
      do 100 j=1,6
      x(j)=0.d0
      a(3,j)=1.d0
  100 continue
      x(3)=1.d0
      a(4,1)=-c14
      a(4,2)=-c24
      a(4,3)=-c34
      a(4,4)=c41+c42+c43+c45+c46+p41*a41(i)+p42*a42(i)
      a(4,5)=-c54
      a(4,6)=-(c64+p64*a64(i))
      a(5,1)=-c15
      a(5,2)=-c25
      a(5,3)=-c35
      a(5,4)=-c45
      a(5,5)=c51+c52+c53+c54+c56+p51*a51(i)+p52*a52(i)+
     &       p53*a53(i)
      a(5,6)=-(c65+p65*a65(i))
      a(6,1)=-c16
      a(6,2)=-c26
      a(6,3)=-c36
      a(6,4)=-c46
      a(6,5)=-c56
      a(6,6)=c61+c62+c63+c64+c64+p64*a64(i)+p65*a65(i)
c
      call matrix(6)
      if(m.lt.2) goto 9988
c
      elab=(1.6022d-12*abund)/(12.57d0*dene)
      cool=elab*(e21*(x(1)*c12-x(2)*c21)+e31*
     &(x(1)*c13-x(3)*c31)+e41*(x(1)*c14-x(4)*c41)+e51*(x(1)*c15-
     &x(5)*c51)+e61*(x(1)*c16-x(6)*c61)+e32*(x(2)*c23-x(3)*c32)+
     &e42*(x(2)*c24-x(4)*c42)+e52*(x(2)*c25-x(5)*c52)+e62*(x(2)*
     &c26-x(6)*c62)+e43*(x(3)*c34-x(4)*c43)+e53*(x(3)*c35-x(5)*
     &c53)+e63*(x(3)*c36-x(6)*c63)+e54*(x(4)*c45-x(5)*c54)+
     &e64*(x(4)*c46-x(6)*c64)+e65*(x(5)*c56-x(6)*c65))
      cool=12.57d0*cool
      f21=elab*x(2)*a21(i)*e21*p21
      f31=elab*x(3)*a31(i)*e31*p31
      f41=elab*x(4)*a41(i)*e41*p41
      f51=elab*x(5)*a51(i)*e51*p51
      f45t23=elab*(x(4)*e42*a42(i)*p42+x(5)*(e52*a52(i)*p52+
     &e53*a53(i)*p53))
      f6t45=elab*x(6)*(p64*e64*a64(i)+p65*e65*a65(i))
c
        ENDIF
      endif
C
      RETURN
      END
C*********************************************************************
c*********************************************************************   
      double precision function spline(p1,p2,p3,p4,p5,x)
C
C     Devised by Burgess & Tully (A & A 254, 436, '92).
C************************************************************           
      implicit real*8(a-h,o-z)
C
      s=1.d0/30.d0
      s2=32.d0*s*(19.d0*p1-43.d0*p2+30.d0*p3-7.d0*p4+p5)
      s3=160.d0*s*(-p1+7.d0*p2-12.d0*p3+7.d0*p4-p5)
      s4=32.d0*s*(p1-7.d0*p2+30.d0*p3-43.d0*p4+19.d0*p5)
      if(x.gt.0.25d0) goto 1
      x0=x-0.125d0
      t3=0.0d0
      t2=0.5d0*s2
      t1=4.d0*(p2-p1)
      t0=0.5d0*(p1+p2)-0.015625d0*t2
      goto 4
    1 if(x.gt.0.5d0) goto 2
      x0=x-0.375d0
      t3=20.d0*s*(s3-s2)
      t2=0.25d0*(s2+s3)
      t1=4.0d0*(p3-p2)-0.015625d0*t3
      t0=0.5d0*(p2+p3)-0.015625d0*t2
      goto 4
    2 if(x.gt.0.75d0) goto 3
      x0=x-0.625d0
      t3=20.d0*s*(s4-s3)
      t2=0.25d0*(s4+s3)
      t1=4.0d0*(p4-p3)-0.015625d0*t3
      t0=0.5d0*(p4+p3)-0.015625d0*t2
      goto 4
    3 x0=x-0.875d0
      t3=0.0d0
      t2=0.5d0*s4
      t1=4.d0*(p5-p4)
      t0=0.5d0*(p5+p4)-0.015625d0*t2
    4 spline=t0+x0*(t1+x0*(t2+x0*t3))
C  
      return
      end
c*********************************************************************   
c*********************************************************************   
      double precision function upsil(k,eij,c,p1,p2,p3,p4,p5,t)
C
C     Devised by Burgess & Tully (A & A 254, 436, '92).
C************************************************************           
      implicit real*8(a-h,o-z)
C
      teij=1.57888d5*eij
      e=dabs(t/teij)
      if(k.eq.1.or.k.eq.4) x=dlog((e+c)/c)/dlog(e+c)
      if(k.eq.2.or.k.eq.3) x=e/(e+c)
      y=spline(p1,p2,p3,p4,p5,x)
      if(k.eq.1) y=y*dlog(e+2.71828d0)
      if(k.eq.3) y=y/(e+1)
      if(k.eq.4) y=y*dlog(e+c)
      upsil=y
C
      return
      end
c*********************************************************************   
C*********************************************************************
      double precision function cheb(tmx,t,a0,a1,a2,a3)
      implicit real*8(a-h,o-z)
C
      y=t
      if(y.lt.5.d3) y=5.d3
      if(y.gt.tmx) y=tmx
      x=0.542519d0*dlog(y)-5.620738d0
      t1=x
      t2=2.d0*x**2-1.d0
      t3=x*(4.d0*x**2-3.d0)
      omlog=0.5d0*a0+a1*t1+a2*t2+a3*t3
      cheb=dexp(omlog)
C
      return
      end
C*********************************************************************
C*********************************************************************
      double precision function cheb1(tmx,t,a0,a1,a2,a3)
      implicit real*8(a-h,o-z)
C
      y=t
      if(y.lt.5.d3) y=5.d3
      if(y.gt.tmx) y=tmx
      x=0.6199646d0*dlog(y)-6.2803580d0
      t1=x
      t2=2.d0*x**2-1.d0
      t3=x*(4.d0*x**2-3.d0)
      omlog=0.5d0*a0+a1*t1+a2*t2+a3*t3
      cheb1=dexp(omlog)
C
      return
      end
C*********************************************************************
C*********************************************************************
      SUBROUTINE RATE(ik,enonth)
C*********************************************************************
C  Calculates heating and ionization of all elements. Above 200 eV we
C  use a parameterization for the inclusion of secondary electrons for
C  neutrals and once ionized ions. (Beware of that 200 eV is not a 
C  unique value by no means !!)
C*********************************************************************
C
      IMPLICIT REAL*8(A-H,O-Z)                                          
      parameter (md=75)
C
      COMMON/FRE/JMIN,JJ                                                
      COMMON/PHQ/ZE(117),GE(117),ZK(117),GET(117)                           
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)       
      COMMON/SIK/SK(117,-13:75)                                          
      COMMON/TRES/EL(117),EK(117)                                        
      COMMON/DTAU/FLUX(md,-13:75)                                       
      COMMON/BOWENS/BORATE(md,117)
      COMMON/ABUN/AB(15)
      COMMON/ABU/XA(2,117)
      COMMON/ELEC/DEL(md),MEL
      common/nontherm/zexc(2),inonth(117)
      common/abun2/abunda(117)
C
      DIMENSION CC(-13:75)
C                                                                       
      PI=3.14159D0
C
      do 297 k=1,117
      if(k.le.3) then
       zk(k)=borate(ik,k)
       ze(k)=0.0d0
      else
       zk(k)=0.0d0
       ze(k)=borate(ik,k)
      endif
      ge(k)=0.0d0
  297 continue
c
      flikj=0.0d0
      do 299 j=jmin,jj
      flikj=dmax1(flikj,fl(ik,j))
  299 continue
      DO 301 J=JMIN,JJ
      flratio=fl(ik,j)/flikj
      if(flratio.lt.1.d-10) then
       cc(j)=0.0d0
      else
       CC(J)=(4.*PI*FL(IK,J)*(E(J+1)-E(J)))/(1.6022D-12*E1(J))
      endif
  301 CONTINUE 
C
      call f_nonthermal(ik,f_exc_h,f_ion_h,f_exc_he,f_ion_he,
     &                  f_heat)
      GSEC=0.0d0
      DO 302 j=jmin,jj
      if(cc(j).ge.1.d-100) then
         DO 303 k=1,117
           if(inonth(k).eq.1.and.e1(j).ge.enonth) then
	      corr2=f_heat
           else
              corr2=1.d0
           endif
         gaa=0.0d0
         if(si(k,j).ge.1.d-100) then
           ZA=CC(J)*SI(K,J)
           if(inonth(k).eq.1.and.e1(j).ge.enonth) then
	      GSEC=GSEC+xa(2,k)*abunda(k)*za
           endif
           ZE(K)=ZE(K)+ZA
           GAA=1.6022D-12*ZA*corr2*(E1(J)-EL(K))
         endif
         if(sk(k,j).ge.1.d-100) then
           ZB=CC(J)*SK(K,J)
           ZK(K)=ZK(K)+ZB
           GAA=GAA+1.6022D-12*ZB*(E1(J)-EK(K))
         endif 
         if(k.le.2.and.e1(j).ge.enonth) then
           GE(K)=GE(K)+GAA*f_heat
	   GSEC=GSEC+xa(2,k)*abunda(k)*zb
         else
           GE(K)=GE(K)+GAA
         endif
  303    CONTINUE
      endif
  302 CONTINUE 
      GHION=(GSEC*f_ion_h)/(ab(1)*xa(2,1))
      GHEION=(GSEC*f_ion_he)/(ab(2)*xa(2,2))
      zk(1)=zk(1)+ghion
      zk(2)=zk(2)+gheion
      zexc(1)=(GSEC*f_exc_h)/(ab(1)*xa(1,2))
      zexc(2)=(GSEC*f_exc_he)/(ab(2)*xa(2,2))
C
      RETURN                                                            
      END                                                               
C************************************************************           
C************************************************************           
      SUBROUTINE SPEC(i) 

      IMPLICIT REAL*8(A-H,O-Z)                                          
      parameter (md=75)
C
      COMMON/PHOTOS/TEBB,RPA15                                          
      COMMON/TAUXX/TAXA(md,-13:75)                                      
      COMMON/LUMI/FSURF(-13:75)                                         
      COMMON/FRE/JMIN,JJ                                                
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)       
      COMMON/DIF/TAU(md,-13:75)                            
       COMMON/RADIE/R(md)                                               
C--------------------------------------------------------------         
C-------- FSURF = Photospheric Eddington flux ---------- 
C-------- FL = Mean intensity in shell  i
      if((r(i)/rpa15).gt.1.d4) then
       w1=0.25d0*(rpa15/r(i))**2
       w2=0.25d0*(rpa15/r(i+1))**2
      else
       w1=0.5d0*(1.d0-dsqrt(1.d0-(rpa15/r(i))**2))
       w2=0.5d0*(1.d0-dsqrt(1.d0-(rpa15/r(i+1))**2))
      endif
      wmean=0.5d0*(w1+w2)
       do 1000 j=jmin,jj
       if(i.eq.1) tau(i,j)=0.0d0
       dtt=1.d15*(r(i+1)-r(i))*taxa(i,j)
       tau(i+1,j)=tau(i,j)+dtt
       z=expfn(dtt)
       if(dabs(dtt).lt.1.d-5) then
        tmean=1.d0
       else
        tmean=(1.d0-z)/dtt
       endif
       FL(I,J)=4.D0*FSURF(J)*wmean*tmean*expfn(tau(i,j))
       IF(FL(I,J).LT.1.D-100) FL(I,J)=1.D-100
 1000  continue
c
      RETURN
      END
C************************************************************           
C************************************************************           
      SUBROUTINE f_nonthermal(i,f_exc_h,f_ion_h,f_exc_he,
     &                        f_ion_he,f_heat)
C
      IMPLICIT REAL*8(A-H,O-Z)
      parameter (md=75)
C
      COMMON/ABUN/AB(15)
      COMMON/ABU/XA(2,117)
      COMMON/ELEC/DEL(md),MEL
C
      absum=0.0d0
      do ii=1,14
	absum=absum+ab(ii)
      enddo
      abh1=(xa(2,1)*ab(1))/absum
      abhe1=(xa(2,2)*ab(2))/absum
      xel=del(i)
      if(xel.gt.0.99d0) xel=0.99d0
      if(xel.lt.1.d-4) xel=1.d-4
      xs=1.d-5+dlog10(xel/(1.d0-xel))
      xhfrac=1.d0/(0.9d0*(1.d0-xel))
      xhefrac=1.d0/(0.1d0*(1.d0-xel))
C
C -- fraction going into excitation of H I (=f_exc_h)
      if(xel.lt.1.d-4) then
	 a=4.00d-1
	 b=2.00d-2
	 f_exc_h = 5.75337d-1 + 2.47695d-2*xs - 1.26653d-3*xs**2
      else
	 a=8.40d-1
	 b=1.16d0
	 f_exc_h = 6.63175d-2 - 1.80824d-2*xs + 3.95371d-2*xs**2 +
     &             1.64258d-2*xs**3 + 1.80579d-3*xs**4
      endif 
      f_exc_h=f_exc_h*(1.d0-xel)*(b-datan(a*xs))
C
C -- fraction going into ionization of H I (=f_ion_h)
      if(xel.lt.1.d-4) then
	 a=3.60d-1
	 b=2.00d-2
	 f_ion_h = 5.91124d-1 + 5.79456d-2*xs + 1.84557d-3*xs**2
      else
	 a=7.80d-1
	 b=1.12d0
	 f_ion_h = 5.98357d-2 - 1.86917d-2*xs + 3.41991d-2*xs**2 +
     &             1.50335d-2*xs**3 + 1.67892d-3*xs**4
      endif 
      f_ion_h=f_ion_h*(1.d0-xel)*(b-datan(a*xs))
C
C -- fraction going into excitation of He I (=f_exc_he)
      if(xel.lt.1.d-4) then
	 a=3.40d-1
	 b=2.00d-2
	 f_exc_he = 2.9116d-3 + 6.11349d-4*xs + 5.56751d-5*xs**2
      else
	 a=9.40d-1
	 b=1.20d0
	 f_exc_he = 2.38400d-3 - 4.697638d-4*xs + 1.34543d-3*xs**2 +
     &              6.47551d-4*xs**3 + 7.93871d-5*xs**4
      endif 
      f_exc_he=f_exc_he*(1.d0-xel)*(b-datan(a*xs))
C
C -- fraction going into ionization of He I (=f_ion_he)
      if(xel.lt.1.d-4) then
	 f_ion_he=0.0d0
      else
	 a=8.40d-1
	 b=1.16d0
	 f_ion_he = 1.02592d-2 - 2.69490d-3*xs + 5.30982d-3*xs**2 +
     &              2.67128d-3*xs**3 + 3.32947d-4*xs**4
         f_ion_he=f_ion_he*(1.d0-xel)*(b-datan(a*xs))
      endif 
C
C -- correction to 
      f_exc_h=abh1*xhfrac*f_exc_h
      f_ion_h=abh1*xhfrac*f_ion_h
      f_exc_he=abhe1*xhefrac*f_exc_he
      f_ion_he=abhe1*xhefrac*f_ion_he
      f_heat=1.d0-f_exc_h-f_ion_h-f_exc_he-f_ion_he
C
      RETURN
      END
C************************************************************           
C************************************************************           
      SUBROUTINE TAUI(i)
C
      IMPLICIT REAL*8(A-H,O-Z)
      parameter (md=75)
C
      COMMON/FRE/JMIN,JJ
      COMMON/ION/XB(md,117)
      COMMON/ABU/XA(2,117)
      COMMON/PHY/DEN(md)
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)
      COMMON/TAUXX/TAXA(md,-13:75)
      COMMON/TEM/TE(md)
      COMMON/ELEC/DEL(md),MEL
      COMMON/SIK/SK(117,-13:75)
      common/abun2/abunda(117)
C
      DIMENSION XP(117),QQ(-13:75)
C---------------------------------------------------------------        
C  Calculate opacity (TAXA(I,J))
C---------------------------------------------------------------        
      DO 7128 K=1,117                                                    
 7128 XP(K)=XA(2,K)                                                     
 8345 CONTINUE                                                          
C
C     FREE FREE ABSORBTION (ALLEN)                                      
      cfree=(2.61d-35*(1.d0-xp(1))*den(i)*del(i))/dsqrt(te(i))
      tev=te(i)/1.1605d4
      do 825 j=jmin,jj
      TA=(cfree*(1.-EXPFN(E1(J)/tev)))/E1(J)**3. 
      if(ta.lt.1.d-45) ta=1.d-45
      qq(j)=ta
  825 continue
c
      do 810 k=1,117
      abxa=abunda(k)*xp(k)
      if(abxa.ge.1.d-14.or.k.le.3) then
       do 6659 j=jmin,jj
       qq(j)=qq(j)+abxa*(sk(k,j)+si(k,j))
 6659  continue
      endif
  810 continue
      do 811 j=jmin,jj
      taxa(i,j)=qq(j)*den(i)
  811 continue
C                          
      RETURN                                                            
      END                                                               
C*******************************************************************           
C*******************************************************************           
      SUBROUTINE BOWOTS(imax,ite,CORAT1,CORAT2,corat3,corat4,corat5,
     &corat6,corat7,corat8)                                  
C                                                                       
C      CORAT1, CORAT3 and CORAT2 are the total rates of absorbed
C      He II L-alpha, L-beta and He I L-beta photons, respectively.
C      CORAT4, CORAT5 and CORAT6 are total rates of produced
C      H I + He II (n=2), He I and He II (n=1) recombining photons.
C      CORAT7 = He I 2-gamma, and CORAT8 = He II 2-gamma.
C      To get the numbers re-entering as ionizations in subroutine
C      RATE, the recombinations are weighted by the emission pro-
C      file and the optical depth.
C      Divide by DEN(I)*AB(IJ)*XA before inserting in RATE.
C*******************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      parameter (md=75)
C
      COMMON/SIK/SK(117,-13:75)                                          
      COMMON/FRE/JMIN,JJ                                                
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)       
      COMMON/ION/XB(md,117)                                              
      COMMON/ABUN/AB(15)                                                
      common/abun2/abunda(117)
      COMMON/IND/I                                                      
      COMMON/BOWENS/BORATE(md,117)
      COMMON/PHY/DEN(md)                                                
      COMMON/ABU/XA(2,117)
      COMMON/TEM/TE(md)                                                 
      COMMON/TAUXX/TAXA(md,-13:75)                                      
      common/euvli2/rcont(79),leuv(79),iskipeu(79)
      common/radabs/rfix(md),rbound(md),dxabs(3,md,md)
      common/twexps/twoexp(-13:75)
      common/difrec/crate1(md,md,-13:75),crate2(md,md,-13:75),
     &              crate3(md,md,-13:75),crate4(md,md,-13:75),
     &              crate5(md,md,-13:75)
      COMMON/RADIE/R(md)
      COMMON/TWOHES/he1_2g(-13:75),he2_2g(-13:75),jhe1_2g,jhe2_2g
C                                                                       
      DIMENSION DORATE(117),CORATE(117),FORATE(117),ph1(-13:75),
     &ph2(-13:75),ph3(-13:75),pesc1(-13:75),pesc2(-13:75),
     &pesc3(-13:75),pesc4(-13:75),pesc5(-13:75),fi(5,117,-13:75),
     &sumj(5,-13:75),dtt(-13:75),abx(117),cr1sum(-13:75),
     &cr2sum(-13:75),cr3sum(-13:75),cr4sum(-13:75),cr5sum(-13:75)
C                                                                       
      height=0.05d0*r(1)
      pi=3.1415926535898d0
      pihalf=pi/2.d0
      DO 9000 IJ=1,117
      do 8001 kk=1,5
      do 8001 j=2,jj
      fi(kk,ij,j)=0.0d0
 8001 continue
      abx(ij)=abunda(ij)*xa(2,ij)
      if(abx(ij).le.1.d-50) abx(ij)=1.d-50
      BORATE(i,IJ)=0.0d0
      CORATE(IJ)=0.0D0
      forate(ij)=0.0d0
 9000 DORATE(IJ)=0.0D0
      do j=2,jj
	ph1(j)=0.0d0
	ph2(j)=0.0d0
	ph3(j)=0.0d0
	cr1sum(j)=0.0d0
	cr2sum(j)=0.0d0
	cr3sum(j)=0.0d0
	cr4sum(j)=0.0d0
	cr5sum(j)=0.0d0
      enddo
C
      tev=te(i)/1.1605d4
      dxx=dxabs(1,i,i)+dxabs(3,i,i)
C - H I and He II(n=2) emission profile and escape of continuum photons
C - make it right for optically thick situations (plane-parallel approach
C - increases the effective tau by factor of roughly 2 compared to a radial
C - tau.)
C - (skip the inclusion of vertical escape !!) Also include a geometrical 
C - covering factor assuming the ring has a height of 5% of its radius.
      sumh1=0.0d0
       do 2001 j=2,jj
          jj1=j
          ph1(j)=(expfn((e(j)-e(2))/tev))*(1.d0-expfn((e(j+1)-e(j))/
     &            tev))
          sumh1=sumh1+ph1(j)
          if(sumh1.gt.0.99999d0) goto 2002
 2001  continue
 2002 continue
       do 2011 j=2,jj1
          dtt(j)=2.d0*dxx*taxa(i,j)
          pesc1(j)=pexpfn(dtt(j))
          crate1(i,i,j)=ph1(j)*corat4*(1.d0-pesc1(j))
          psc=0.5d0*pesc1(j)*ph1(j)*corat4
	  if(i.ne.1) then
	    sumtau=0.0d0
	    do iik=i-1,1,-1
	       dttin=2.d0*dxabs(1,i,iik)*taxa(iik,j)
	       deltar=0.5d0*(r(i+1)+r(i)-r(iik+1)-r(iik))
	       dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttin))*pexpfn(dttin)
	       crate1(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
	       sumtau=dttin+sumtau
	       if(sumtau.gt.2.d1) goto 7631
	    enddo
          endif
 7631     continue
	  if(i.ne.imax) then
	    sumtau=0.0d0
	    do iik=i+1,imax
	       dttout=2.d0*dxabs(3,i,iik)*taxa(iik,j)
	       deltar=0.5d0*(r(iik+1)+r(iik)-r(i+1)-r(i))
	       dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttout))*pexpfn(dttout)
	       crate1(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
	       sumtau=dttout+sumtau
	       if(sumtau.gt.2.d1) goto 7632
	    enddo
          endif
 7632     continue
 2011  continue
C - He I emission profile and escape of continuum photons
      sumh2=0.0d0
       do 2003 j=7,jj
          jj2=j
          ph2(j)=(expfn((e(j)-e(7))/tev))*(1.d0-expfn((e(j+1)-e(j))/
     &            tev))
          sumh2=sumh2+ph2(j)
          if(sumh2.gt.0.99999d0) goto 2004
 2003  continue
 2004 continue
       do 2012 j=7,jj2
          dtt(j)=2.d0*dxx*taxa(i,j)
          pesc2(j)=pexpfn(dtt(j))
          crate2(i,i,j)=ph2(j)*corat5*(1.d0-pesc2(j))
          psc=0.5d0*pesc2(j)*ph2(j)*corat5
	  if(i.ne.1) then
	    sumtau=0.0d0
	    do iik=i-1,1,-1
	       dttin=2.d0*dxabs(1,i,iik)*taxa(iik,j)
               deltar=0.5d0*(r(i+1)+r(i)-r(iik+1)-r(iik))
               dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttin))*pexpfn(dttin)
               crate2(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
               sumtau=dttin+sumtau
               if(sumtau.gt.2.d1) goto 7633
	    enddo
          endif
 7633     continue
	  if(i.ne.imax) then
	    sumtau=0.0d0
	    do iik=i+1,imax
	       dttout=2.d0*dxabs(3,i,iik)*taxa(iik,j)
               deltar=0.5d0*(r(iik+1)+r(iik)-r(i+1)-r(i))
               dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttout))*pexpfn(dttout)
               crate2(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
               sumtau=dttout+sumtau
	       if(sumtau.gt.2.d1) goto 7634
	    enddo
          endif
 7634     continue
 2012  continue
C - He II (n=1) emission profile and escape of continuum photons
      sumh3=0.0d0
       do 2005 j=14,jj
          jj3=j
          ph3(j)=(expfn((e(j)-e(14))/tev))*(1.d0-expfn((e(j+1)-e(j))/
     &            tev))
          sumh3=sumh3+ph3(j)
          if(sumh3.gt.0.99999d0) goto 2006
 2005  continue
 2006 continue
       do 2013 j=14,jj3
          dtt(j)=2.d0*dxx*taxa(i,j)
          pesc3(j)=pexpfn(dtt(j))
          crate3(i,i,j)=ph3(j)*corat6*(1.d0-pesc3(j))
          psc=0.5d0*pesc3(j)*ph3(j)*corat6
	  if(i.ne.1) then
	    sumtau=0.0d0
	    do iik=i-1,1,-1
	       dttin=2.d0*dxabs(1,i,iik)*taxa(iik,j)
               deltar=0.5d0*(r(i+1)+r(i)-r(iik+1)-r(iik))
               dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttin))*pexpfn(dttin)
               crate3(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
               sumtau=dttin+sumtau
	       if(sumtau.gt.2.d1) goto 7635
	    enddo
          endif
 7635     continue
	  if(i.ne.imax) then
	    sumtau=0.0d0
	    do iik=i+1,imax
	       dttout=2.d0*dxabs(3,i,iik)*taxa(iik,j)
               deltar=0.5d0*(r(iik+1)+r(iik)-r(i+1)-r(i))
               dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttout))*pexpfn(dttout)
               crate3(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
               sumtau=dttout+sumtau
	       if(sumtau.gt.2.d1) goto 7636
	    enddo
          endif
 7636     continue
 2013  continue
C - He I two-photon excitation rates. Skip energies below 13.6 eV. Ions
C - with lower ionization potential are ionized anyway!
       do 2014 j=2,jhe1_2g
          dtt(j)=2.d0*dxx*taxa(i,j)
          pesc4(j)=pexpfn(dtt(j))
          crate4(i,i,j)=he1_2g(j)*corat7*(1.d0-pesc4(j))
          psc=0.5d0*pesc4(j)*he1_2g(j)*corat7
	  if(i.ne.1) then
	    sumtau=0.0d0
	    do iik=i-1,1,-1
	       dttin=2.d0*dxabs(1,i,iik)*taxa(iik,j)
	       deltar=0.5d0*(r(i+1)+r(i)-r(iik+1)-r(iik))
	       dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttin))*pexpfn(dttin)
	       crate4(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
	       sumtau=dttin+sumtau
	       if(sumtau.gt.2.d1) goto 7637
	    enddo
          endif
 7637     continue
	  if(i.ne.imax) then
	    sumtau=0.0d0
	    do iik=i+1,imax
	       dttout=2.d0*dxabs(3,i,iik)*taxa(iik,j)
	       deltar=0.5d0*(r(iik+1)+r(iik)-r(i+1)-r(i))
	       dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttout))*pexpfn(dttout)
	       crate4(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
	       sumtau=dttout+sumtau
	       if(sumtau.gt.2.d1) goto 7638
	    enddo
          endif
 7638     continue
 2014  continue
C - He II two-photon excitation rates. Skip energies below 13.6 eV. Ions
C - with lower ionization potential are ionized anyway!
       do 2015 j=2,jhe2_2g
          dtt(j)=2.d0*dxx*taxa(i,j)
          pesc5(j)=pexpfn(dtt(j))
          crate5(i,i,j)=he2_2g(j)*corat8*(1.d0-pesc5(j))
          psc=0.5d0*pesc5(j)*he2_2g(j)*corat8
	  if(i.ne.1) then
	    sumtau=0.0d0
	    do iik=i-1,1,-1
	       dttin=2.d0*dxabs(1,i,iik)*taxa(iik,j)
	       deltar=0.5d0*(r(i+1)+r(i)-r(iik+1)-r(iik))
	       dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttin))*pexpfn(dttin)
	       crate5(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
	       sumtau=dttin+sumtau
	       if(sumtau.gt.2.d1) goto 7639
	    enddo
          endif
 7639     continue
	  if(i.ne.imax) then
	    sumtau=0.0d0
	    do iik=i+1,imax
	       dttout=2.d0*dxabs(3,i,iik)*taxa(iik,j)
	       deltar=0.5d0*(r(iik+1)+r(iik)-r(i+1)-r(i))
	       dilute=datan(height/deltar)/pihalf
               fwei=(1.d0-expfn(dttout))*pexpfn(dttout)
	       crate5(i,iik,j)=psc*dilute*fwei*expfn(sumtau)
	       sumtau=dttout+sumtau
	       if(sumtau.gt.2.d1) goto 7640
	    enddo
          endif
 7640     continue
 2015  continue
      sum1=0.0d0
      sum2=0.0d0
      sum3=0.0d0
       do 2020 j=2,jj1
 2020  sumj(1,j)=0.0d0
       do 2016 j=7,jj2
 2016  sumj(2,j)=0.0d0
       do 2017 j=14,jj3
 2017  sumj(3,j)=0.0d0
       do 2018 j=2,jhe1_2g
 2018  sumj(4,j)=0.0d0
       do 2019 j=2,jhe2_2g
 2019  sumj(5,j)=0.0d0
      do 9001 ij=1,117
      abxa=abx(ij)
      if(abxa.ge.1.d-12.or.ij.le.3) then
       corate(ij)=(si(ij,11)+sk(ij,11))*abxa
       if(ij.eq.8.or.ij.eq.20) then
        dorate(ij)=(si(ij,13)+sk(ij,13))*abxa
       elseif(ij.eq.28) then
        dorate(ij)=(si(ij,13)+sk(ij,13))*abxa
       else
        dorate(ij)=(si(ij,12)+sk(ij,12))*abxa
       endif
       forate(ij)=(si(ij,5)+sk(ij,5))*abxa
        do 2022 j=2,jj1
        fi(1,ij,j)=abxa*(si(ij,j)+sk(ij,j))
        sumj(1,j)=sumj(1,j)+fi(1,ij,j)
2022    continue
        do 2023 j=7,jj2
        fi(2,ij,j)=abxa*(si(ij,j)+sk(ij,j))
        sumj(2,j)=sumj(2,j)+fi(2,ij,j)
2023    continue
        do 2024 j=14,jj3
        fi(3,ij,j)=abxa*(si(ij,j)+sk(ij,j))
        sumj(3,j)=sumj(3,j)+fi(3,ij,j)
2024    continue
        do 2025 j=2,jhe1_2g
        fi(4,ij,j)=abxa*(si(ij,j)+sk(ij,j))
        sumj(4,j)=sumj(4,j)+fi(4,ij,j)
2025    continue
        do 2026 j=2,jhe2_2g
        fi(5,ij,j)=abxa*(si(ij,j)+sk(ij,j))
        sumj(5,j)=sumj(5,j)+fi(5,ij,j)
2026    continue
       sum1=sum1+corate(ij)
       sum2=sum2+dorate(ij)
       sum3=sum3+forate(ij)
      endif
 9001 continue
      corat1=corat1/sum1
      corat2=corat2/sum2
      corat3=corat3/sum3
       do 3001 j=2,jj1
	  cr1sum(j)=crate1(1,i,j)
	  do iik=2,imax
	     cr1sum(j)=cr1sum(j)+crate1(iik,i,j)
	  enddo
          cr1sum(j)=cr1sum(j)/sumj(1,j)
 3001  continue
       do 3002 j=7,jj2
	  cr2sum(j)=crate2(1,i,j)
	  do iik=2,imax
	     cr2sum(j)=cr2sum(j)+crate2(iik,i,j)
	  enddo
          cr2sum(j)=cr2sum(j)/sumj(2,j)
 3002  continue
       do 3003 j=14,jj3
	  cr3sum(j)=crate3(1,i,j)
	  do iik=2,imax
	     cr3sum(j)=cr3sum(j)+crate3(iik,i,j)
	  enddo
          cr3sum(j)=cr3sum(j)/sumj(3,j)
 3003  continue
       do 3004 j=2,jhe1_2g
	  cr4sum(j)=crate4(1,i,j)
	  do iik=2,imax
	     cr4sum(j)=cr4sum(j)+crate4(iik,i,j)
	  enddo
          cr4sum(j)=cr4sum(j)/sumj(4,j)
 3004  continue
       do 3005 j=2,jhe2_2g
	  cr5sum(j)=crate5(1,i,j)
	  do iik=2,imax
	     cr5sum(j)=cr5sum(j)+crate5(iik,i,j)
	  enddo
          cr5sum(j)=cr5sum(j)/sumj(5,j)
 3005  continue
      do 9002 ij=1,117
      abxa=abx(ij)
      if(abxa.ge.1.d-12.or.ij.le.3) then
       abxad=abxa*den(i)
       borate(i,ij)=(corate(ij)*corat1+dorate(ij)*corat2+forate(ij)*
     & corat3)/abxad
C - the division by abxad is OK! (-> lag of one time step for the
C                                    fraction absorbed by ion 'ij'.)
        do 9501 j=2,jj1
        borate(i,ij)=borate(i,ij)+(cr1sum(j)*fi(1,ij,j))/abxad
 9501   continue
        do 9502 j=7,jj2
        borate(i,ij)=borate(i,ij)+(cr2sum(j)*fi(2,ij,j))/abxad
 9502   continue
        do 9503 j=14,jj3
        borate(i,ij)=borate(i,ij)+(cr3sum(j)*fi(3,ij,j))/abxad
 9503   continue
        do 9504 j=2,jhe1_2g
        borate(i,ij)=borate(i,ij)+(cr4sum(j)*fi(4,ij,j))/abxad
 9504   continue
        do 9505 j=2,jhe2_2g
        borate(i,ij)=borate(i,ij)+(cr5sum(j)*fi(5,ij,j))/abxad
 9505   continue
      endif
 9002 continue
C
      RETURN
      END
C************************************************************           
C************************************************************           
C                                                                       
      SUBROUTINE VOIGT(A,VS,H)                                          
      implicit real*8(a-h,o-z)                                          
C                                                                       
C  COMPUTES A VOIGT FUNCTION  H = H(A,V)                                
C  A=GAMMA/(4*PI*DNUD)   AND  V=(NU-NU0)/DNUD.  THIS  IS  DONE AFTER    
C  TRAVING (LANDOLT-B\RNSTEIN, P. 449).                                 
C                                                                       
C  CODED BY: KJELL ERIKSSON  (APR-1974). REVISED: (JULY-1975).          
C                                                                       
      DIMENSION AK(19),A1(5)                                            
      DATA AK      /-1.12470432, -0.15516677,  3.28867591, -2.34357915, 
     ,  0.42139162, -4.48480194,  9.39456063, -6.61487486,  1.98919585, 
     , -0.22041650, 0.554153432, 0.278711796,-0.188325687, 0.042991293, 
     ,-0.003278278, 0.979895023,-0.962846325, 0.532770573,-0.122727278/ 
      DATA SQP/1.772453851/,SQ2/1.414213562/                            
C                                                                       
      V = DABS(VS)                                                      
      U = A + V                                                         
      V2 = V*V                                                          
      IF (A.EQ.0.0) GO TO 140                                           
      IF (A.GT.0.2) GO TO 120                                           
      IF (V.GE.5.0) GO TO 121                                           
C                                                                       
      EX=0.                                                             
      IF(V2.LT.100.)EX = DEXP(-V2)                                      
      K = 1                                                             
C                                                                       
  100 QUO = 1.                                                          
      IF (V.LT.2.4) GO TO 101                                           
      QUO = 1./(V2 - 1.5)                                               
      M = 11                                                            
      GO TO 102                                                         
C                                                                       
  101 M = 6                                                             
      IF (V.LT.1.3) M = 1                                               
  102 DO 103 I=1,5                                                      
         A1(I) = AK(M)                                                  
         M = M + 1                                                      
  103 CONTINUE                                                          
      H1 = QUO*(A1(1) + V*(A1(2) + V*(A1(3) + V*(A1(4) + V*A1(5)))))    
      IF (K.GT.1) GO TO 110                                             
C                                                                       
C A LE 0.2  AND V LT 5.                                                 
C                                                                       
      H = H1*A + EX*(1. + A*A*(1. - 2.*V2))                             
      RETURN                                                            
C                                                                       
  110 PQS = 2./SQP                                                      
      H1P = H1 + PQS*EX                                                 
      H2P = PQS*H1P - 2.*V2*EX                                          
      H3P = (PQS*(1. - EX*(1. - 2.*V2)) - 2.*V2*H1P)/3. + PQS*H2P       
      H4P = (2.*V2*V2*EX - PQS*H1P)/3. + PQS*H3P                        
      PSI = AK(16) + A*(AK(17) + A*(AK(18) + A*AK(19)))                 
C                                                                       
C 0.2 LT A LE 1.4  AND  A + V LE 3.2                                    
C                                                                       
      H = PSI*(EX + A*(H1P + A*(H2P + A*(H3P + A*H4P))))                
      RETURN                                                            
C                                                                       
  120 IF (A.GT.1.4.OR.U.GT.3.2) GO TO 130                               
      EX=0.                                                             
      IF(V2.LT.100.)EX = DEXP(-V2)                                      
      K = 2                                                             
      GO TO 100                                                         
C                                                                       
C A LE 0.2  AND  V GE 5.                                                
C                                                                       
  121 H = A*(15. + 6.*V2 + 4.*V2*V2)/(4.*V2*V2*V2*SQP)                  
      RETURN                                                            
C                                                                       
  130 A2 = A*A                                                          
      U = SQ2*(A2 + V2)                                                 
      U2 = 1./(U*U)                                                     
C                                                                       
C A GT 1.4  OR  A + V GT 3.2                                            
C                                                                       
      H = SQ2/SQP*A/U*(1. + U2*(3.*V2 - A2) +                           
     ,        U2*U2*(15.*V2*V2 - 30.*V2*A2 + 3.*A2*A2))                 
      RETURN                                                            
C                                                                       
C A EQ 0.                                                               
C                                                                       
  140 H=0.                                                              
      IF(V2.LT.100.)H=DEXP(-V2)                                         
      RETURN                                                            
      END                                                               
C                                                                       
C************************************************************           
C************************************************************           
      SUBROUTINE SIMPA(N,H,S)                                           
      IMPLICIT REAL*8(A-H,O-Z)                                          
      COMMON/PHEXC/F(300)                                               
C     OBS! N MUST BE AN ODD NUMBER                                      
      NC=N-1                                                            
      ND=N-2                                                            
      NA=N+1                                                            
      NB=N-3                                                            
      SUM4=0.                                                           
      SUM2=0.                                                           
      DO 120 I=2,NC,2                                                   
  120 SUM4=SUM4+F(I)                                                    
      DO 130 I=3,ND,2                                                   
  130 SUM2=SUM2+F(I)                                                    
  320 S=H*(4.*SUM4+2.*SUM2+F(1)+F(N))/3.                                
      RETURN                                                            
      END                                                               
c*********************************************************************   
C*********************************************************************
      SUBROUTINE TRINIT
      implicit real*8(a-h,o-z)
C
      parameter (md=75,levhmx=55,levh=levhmx+1,levhemx=71,
     &           levhe=levhemx+1,nnhmax=10,mmatr=80)
C
      common/folines2/redsi(10),UVSI(md,4),forsi(10)
      common/column/colni(md,117),colno(md,117)
      common/hyddt/qxh(md,2,levh),hradm(levh,levh),hccoll(levh,levh)
      common/hedt/qxhe(md,2,levhe),heradm(levhe,levhe),
     &            hecoll(levhe,levhe)
      common/arglin/farg(md,6)
      COMMON/BWLOSS/FOIII(md,8)
      COMMON/HEOPT/OPTHEL(10,md)
      COMMON/YIELDA/YLA(md),YB(md),YO2(md),YA(md),IYIELD,KYIELD
      common/yieldc/ela(md),plinla(md),pconla(md),poo(md)
      COMMON/UVSUVS/UVHELI(md,1),UVCARB(md,6),UVOXYG(md,5),UVNITR(md,6)
      COMMON/TAUXX/TAXA(md,-13:75)
      COMMON/FRE/JMIN,JJ
      COMMON/EQUIV/W(30),CIN(30),FB(md,15),TWOP(md)
      COMMON/PHY/DEN(md)
      COMMON/DXA/DX(md)
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)
      COMMON/ION/XB(md,117)
      COMMON/DIF/TAU(md,-13:75)
      COMMON/TEM/TE(md)
      COMMON/PELEC/QXB(md,2,131),XU(131),JUP
      COMMON/RADIE/R(md)
      COMMON/ELEC/DEL(md),MEL
      COMMON/SPECT/TEL,FD(1,1),F0(-13:75),FEDD(md,-13:75),IPARA
      COMMON/DTAU/FLUXED(md,-13:75)
      COMMON/ABSH/ABS12(md),ABS13(md),ABSHE2(md),ABSHE3(md),ABSO1(md),
     &            ABSHE4(MD),ABSHE5(MD),ABSHE6(MD),ABSHE7(MD),
     &            ABSHE8(MD)
      COMMON/ABSCNO/ABSEUV(md,99)
      COMMON/BOWHEL/BWHELI(md,2)
      COMMON/ABSHN/ABSNV(md)
      COMMON/ADDL/ADDLIN(md,75)
      common/radabs/rfix(md),rbound(md),dxabs(3,md,md)
      common/oldtee/TEALD(md)
      COMMON/RADADD/SQW(md),TEMPST(md),WARM(md),DENT(md),DENTK(md),
     &              COLD(md),HYD(md),DXA(md),tsecc(md),RAAA(md),
     &              RAAB(md),RAAC(md),STOLIN(md,200),frro21(md),
     &              frro41(md),frrs21(md),frrs41(md),icomp(md),itee(md)
      COMMON/NVLEV/x1n5(md),p21in(md),p21out(md)
      common/difrec/crate1(md,md,-13:75),crate2(md,md,-13:75),
     &              crate3(md,md,-13:75),crate4(md,md,-13:75),
     &              crate5(md,md,-13:75)
C
      do i=1,md
	 do iij=1,md
	    do jji=-13,75
	       crate1(md,md,jji)=0.0d0
	       crate2(md,md,jji)=0.0d0
	       crate3(md,md,jji)=0.0d0
	       crate4(md,md,jji)=0.0d0
	       crate5(md,md,jji)=0.0d0
            enddo
         enddo
      yla(i)=0.0d0
      yb(i)=0.0d0
      yo2(i)=0.0d0
      ya(i)=0.0d0
      ela(i)=0.0d0
      plinla(i)=0.0d0
      pconla(i)=0.0d0
      poo(i)=0.0d0
      uvheli(i,1)=0.0d0
      twop(i)=0.0d0
      den(i)=0.0d0
      dx(i)=0.0d0
      te(i)=0.0d0
      r(i)=0.0d0
      del(i)=0.0d0
      abs12(i)=0.0d0
      abs13(i)=0.0d0
      abshe2(i)=0.0d0
      abshe3(i)=0.0d0
      abso1(i)=0.0d0
      abshe4(i)=0.0d0
      abshe5(i)=0.0d0
      abshe6(i)=0.0d0
      abshe7(i)=0.0d0
      abshe8(i)=0.0d0
      absnv(i)=0.0d0
      x1n5(i)=0.0d0
      rfix(i)=0.0d0
      rbound(i)=0.0d0
      teald(i)=0.0d0
      sqw(i)=0.0d0
      tempst(i)=0.0d0
      warm(i)=0.0d0
      dent(i)=0.0d0
      dentk(i)=0.0d0
      cold(i)=0.0d0
      hyd(i)=0.0d0
      dxa(i)=0.0d0
      icomp(i)=0
      itee(i)=0
      tsecc(i)=0.0d0
      raaa(i)=0.0d0
      raab(i)=0.0d0
      raac(i)=0.0d0
      frro21(i)=0.0d0
      frro41(i)=0.0d0
      frrs21(i)=0.0d0
      frrs41(i)=0.0d0
	do j=jmin,jj
	taxa(i,j)=0.0d0
	fl(i,j)=0.0d0
	tau(i,j)=0.0d0
	fedd(i,j)=0.0d0
	fluxed(i,j)=0.0d0
	enddo
	do j=1,200
	if(j.le.2) then
	  bwheli(i,j)=0.0d0
	  do k=1,131
	  if(k.le.levh) then
	    qxh(i,j,k)=0.0d0
          endif
	  if(k.le.levhe) then
	    qxhe(i,j,k)=0.0d0
	  endif
	  qxb(i,j,k)=0.0d0
	  enddo
        endif
	if(j.le.3) then
	  do k=1,md
	  dxabs(j,i,k)=0.0d0
	  enddo
        endif
	if(j.le.4) then
	  uvsi(i,j)=0.0d0
	endif
	if(j.le.5) then
	  uvoxyg(i,j)=0.0d0
	endif
	if(j.le.6) then
	  farg(i,j)=0.0d0
	  uvcarb(i,j)=0.0d0
	  uvnitr(i,j)=0.0d0
	endif
	if(j.le.8) then
	  foiii(i,j)=0.0d0
	endif
	if(j.le.10) then
	  opthel(j,i)=0.0d0
	endif
	if(j.le.15) then
	  fb(i,j)=0.0d0
	endif
	if(j.le.117) then
	  colni(i,j)=0.0d0
	  colno(i,j)=0.0d0
	  xb(i,j)=0.0d0
	endif
	if(j.le.75) then
	  addlin(i,j)=0.0d0
	endif
	if(j.le.99) then
	  abseuv(i,j)=0.0d0
	endif
	stolin(i,j)=0.0d0 
	enddo
      enddo
C
      RETURN
      END
C******************************************************************
C*****************************************************************      
      SUBROUTINE ELDENS(MELECT)                                         
C
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      parameter (md=75)
C
      COMMON/PELEC/QXB(md,2,131),XU(131),JUP
      COMMON/ABUN/AB(15)
      COMMON/IND/I
      COMMON/TIMES/DELTT,DEPE,MAXT
C
      DIMENSION XV(131)
C
      IF(MELECT.LE.1) GOTO 10
      DO 310 K=1,131
  310 XU(K)=QXB(I,JUP,K)                                                
      GOTO 110                                                          
   10 CONTINUE                                                          
      DO 100 K=1,131
  100 XU(K)=QXB(I,JUP-1,K)
  110 G8=XU(2)*AB(1)+AB(2)*(XU(4)+2.*XU(5))
      DO 120 K=7,12
  120 G8=G8+AB(4)*DBLE(K-6)*XU(K)
      DO 130 K=14,21
  130 G8=G8+AB(3)*DBLE(K-13)*XU(K)
      DO 121 K=23,32
  121 G8=G8+AB(6)*DBLE(K-22)*XU(K)
      DO 140 K=34,40
  140 G8=G8+AB(5)*DBLE(K-33)*XU(K)
      DO 141 K=42,49
  141 G8=G8+AB(7)*DBLE(K-41)*XU(K)
      DO 142 K=51,65
  142 G8=G8+AB(8)*DBLE(K-50)*XU(K)
      DO 143 K=67,76
  143 G8=G8+AB(9)*DBLE(K-66)*XU(K)
      DO K=78,86
        G8=G8+AB(10)*DBLE(K-77)*XU(K)
      ENDDO
      DO K=88,97
        G8=G8+AB(11)*DBLE(K-87)*XU(K)
      ENDDO
      DO K=99,109
        G8=G8+AB(12)*DBLE(K-98)*XU(K)
      ENDDO
      DO K=111,120
        G8=G8+AB(13)*DBLE(K-110)*XU(K)
      ENDDO
      DO K=122,131
        G8=G8+AB(14)*DBLE(K-121)*XU(K)
      ENDDO
      DEPE=G8                                                           
      IF(MELECT.EQ.2) GOTO 210                                          
      GOTO 800                                                          
  210 CONTINUE                                                          
      DO 220 K=1,131
  220 XV(K)=XU(K)
      XU(2)=XV(3)
      XU(3)=XV(4)
      XU(7)=XV(17)
      XU(12)=XV(6)
      XU(13)=XV(7)
      DO 230 K=4,6
  230 XU(K)=XV(K+14)
      DO 240 K=14,17
  240 XU(K)=XV(K-1)
      do 241 k=25,34
  241 xu(k)=xv(k-3)
      do 242 k=35,42
  242 xu(k)=xv(k+6)
      DO 903 K=18,24
  903 XU(K)=XV(K+15)                                                    
      DO 904 K=43,57
  904 XU(K)=XV(K+7)
      DO 905 K=58,67
  905 XU(K)=XV(K+8)
      DO K=68,76
        XU(K)=XV(K+9)
      ENDDO
      DO K=77,86
        XU(K)=XV(K+10)
      ENDDO
      DO K=87,97
        XU(K)=XV(K+11)
      ENDDO
      DO K=98,107
        XU(K)=XV(K+12)
      ENDDO
      DO K=108,117
        XU(K)=XV(K+13)
      ENDDO
  800 CONTINUE
C
      RETURN
      END
C*****************************************************************      
C*****************************************************************           
C For truncated model ions it is important to include Auger
C ionization also up to the highest ionization stage.
C*****************************************************************           
      SUBROUTINE IONS                                                   
C
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      parameter (md=75,levhmx=55,levh=levhmx+1,levhemx=71,
     &levhe=levhemx+1,nnhmax=10,mmatr=80,levhe1=16)
C
      common/hyddt/qxh(md,2,levh),hradm(levh,levh),hccoll(levh,levh)
      common/hedt/qxhe(md,2,levhe),heradm(levhe,levhe),
     &hecoll(levhe,levhe)
      COMMON/CHION/CIONH,HYCO,HEI,HEII,HEII32,HEIIBW                    
      COMMON/RADIE/R(md)                                                
      COMMON/PHY/DEN(md)                                                
      COMMON/PHQ/ZE(117),GE(117),ZK(117),GET(117)                           
      COMMON/ION/XB(md,117)                                              
      COMMON/ABU/XA(2,117)                                               
      COMMON/ABC/AL(7),ALN(8),ALO(9),ALC(7),ALS(15),ALMG(11),
     &ALAL(12),COO(8),COC(6),CON(7)
      COMMON/ABC2/ALNEO(9),ALSUL(11),CONE(8),COSU(10),zneo(8),
     &zsul(10)
      common/abc3/alfeo(16),cofe(15),zfeo(15),alsio(11),cosi(10),
     &zsio(10)
      common/abc4/alna(10),alar(11),alcca(11),cona(9),zna(9),
     &comg(10),zmg(10),coal(11),zal(11),coar(10),zar(10),coca(10),
     &zcca(10)
      COMMON/HESTUFF/ALDHE,CHEO,CHE                                     
      COMMON/IND/I                                                      
      COMMON/TIMES/DELTT,DEPE,MAXT
      COMMON/PELEC/QXB(md,2,131),XU(131),JUP                              
      COMMON/ELEC/DEL(md),MEL
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      COMMON/ABUN/AB(15)
      COMMON/ZAETA/ZA(8),ZB(6),QXA1,ZC(7)
C
      DIMENSION ZH(2),ZCA(7)
      DIMENSION ALCA(7),ZCAA(5,7),ZOX(9),ZOXA(7,9),ALOX(9)
      DIMENSION ALNI(8),ZNIA(6,8),ZNI(8)
      DIMENSION ALSU(12),ZSUA(10,12),ZSU(12)
      DIMENSION ALNE(10),ZNEA(8,10),ZNE(10)
      DIMENSION ALFE(16),ZFEA(14,16),ZFE(16)
      DIMENSION ZSI(12),ZSIA(10,12),ALSI(12)
      dimension alnat(11),znat(11),znaa(9,11),almag(12),zmag(12),
     &zmga(10,12),alalu(13),zalu(13),zala(11,13),alarg(12),
     &zarg(12),zara(10,12),alcal(12),zcal(12),zcala(10,12)
      DIMENSION DEOLD(md),ACA(14,16,16)
C--------------------------------------------------------------------   
C       IN THE RATE EQS.:
C       ALL Z:s MARK IONIZATION INTEGRALS.                              
C       ALL AL:s MARK REC. COEFFICIENTS.                                
C--------------------------------------------------------------------   
C      CONS1=R(I)/R(I-1)                                                
      CONS1=1.D0
C--------------------------------------------------------------         
C------ start of electron-density loop                                  
C                                                                       
 1772 CONTINUE                                                          
C-----Carbon                                                          
      DO 8372 KXY=2,7                                                   
      ALCA(KXY)=ALC(KXY)                                                
      KXZ=KXY-1                                                         
 8372 ZCA(KXZ)=ZB(KXZ)+DEN(I)*CONS1*DEL(I)*COC(KXZ)                   
      ZCAA(1,3)=ZK(12)                                                
      ZCAA(2,4)=ZK(13)                                                
      ZCAA(3,5)=ZK(8)                                                
      ZCAA(4,6)=ZK(9)                                                 
C------Oxygen                                                           
      DO 8373 KXY=2,9                                                   
      ALOX(KXY)=ALO(KXY)                                                
      KXZ=KXY-1                                                         
 8373 ZOX(KXZ)=ZA(KXZ)+DEN(I)*CONS1*DEL(I)*COO(KXZ)                   
      ZOXA(1,3)=ZK(14)                                                
      DO 4619 K=2,4                                                     
      KX=K+2                                                            
 4619 ZOXA(K,KX)=ZK(K+13)                                             
      ZOXA(5,7)=ZK(7)                                                 
      ZOXA(6,8)=ZK(4)                                                 
C------Sulphur                                                         
      DO 4936 KXY=2,11                                                  
      ALSU(KXY)=ALSUL(KXY)                                              
      KXZ=KXY-1                                                         
 4936 ZSU(KXZ)=ZSUL(KXZ)+DEN(I)*CONS1*DEL(I)*COSU(KXZ)                  
      zsu(10)=zsu(10)+zk(34)
      DO 4939 KXY=1,9                                                   
      KX=KXY+2                                                          
 4939 ZSUA(KXY,KX)=ZK(KXY+24)                                         
C------Nitrogen                                                         
      DO 1436 KXY=2,8                                                   
      ALNI(KXY)=ALN(KXY)                                                
      KXZ=KXY-1                                                         
 1436 ZNI(KXZ)=ZC(KXZ)+DEN(I)*CONS1*DEL(I)*CON(KXZ)                   
      DO 1439 KXY=1,5                                                   
      KX=KXY+2                                                          
 1439 ZNIA(KXY,KX)=ZK(KXY+17)                                         
C------Neon                                                         
      DO 5436 KXY=2,9                                                   
      ALNE(KXY)=ALNEO(KXY)                                              
      KXZ=KXY-1                                                         
 5436 ZNE(KXZ)=ZNEO(KXZ)+DEN(I)*CONS1*DEL(I)*CONE(KXZ)                  
      zne(8)=zne(8)+zk(42)
      DO 5439 KXY=1,7                                                   
      KX=KXY+2                                                          
 5439 ZNEA(KXY,KX)=ZK(KXY+34)                                         
C------Iron                                                         
      DO 5438 KXY=2,16                                               
      ALFE(KXY)=ALFEO(KXY)                                              
      KXZ=KXY-1                                                         
 5438 ZFE(KXZ)=ZFEO(KXZ)+DEN(I)*CONS1*DEL(I)*COFE(KXZ)                  
      zfe(15)=zfe(15)+zk(57)
	do kxy=1,14
	kx=kxy+2
	zfea(kxy,kx)=zk(kxy+42)
	enddo
C------Silicon
      DO 5478 KXY=2,11
      ALSI(KXY)=ALSIO(KXY)
      KXZ=KXY-1
 5478 ZSI(KXZ)=ZSIO(KXZ)+DEN(I)*CONS1*DEL(I)*COSI(KXZ)
      zsi(10)=zsi(10)+zk(67)
      DO KXY=1,9
        KX=KXY+2
        ZSIA(KXY,KX)=ZK(KXY+57)
      ENDDO
C------Sodium
      DO KXY=2,10
        ALNAT(KXY)=ALNA(KXY)
        KXZ=KXY-1
        ZNAT(KXZ)=ZNA(KXZ)+DEN(I)*CONS1*DEL(I)*CONA(KXZ)
      ENDDO
      znat(9)=znat(9)+zk(76)
      DO KXY=1,8
        KX=KXY+2
        ZNAA(KXY,KX)=ZK(KXY+67)
      ENDDO
C------Magnesium
      DO KXY=2,11
        ALMAG(KXY)=ALMG(KXY)
        KXZ=KXY-1
        ZMAG(KXZ)=ZMG(KXZ)+DEN(I)*CONS1*DEL(I)*COMG(KXZ)
      ENDDO
      zmag(10)=zmag(10)+zk(86)
      DO KXY=1,9
        KX=KXY+2
        ZMGA(KXY,KX)=ZK(KXY+76)
      ENDDO
C------Aluminum
      DO KXY=2,12
        ALALU(KXY)=ALAL(KXY)
        KXZ=KXY-1
        ZALU(KXZ)=ZAL(KXZ)+DEN(I)*CONS1*DEL(I)*COAL(KXZ)
      ENDDO
      zalu(11)=zalu(11)+zk(97)
      DO KXY=1,10
        KX=KXY+2
        ZALA(KXY,KX)=ZK(KXY+86)
      ENDDO
C------Argon
      DO KXY=2,11
        ALARG(KXY)=ALAR(KXY)
        KXZ=KXY-1
        ZARG(KXZ)=ZAR(KXZ)+DEN(I)*CONS1*DEL(I)*COAR(KXZ)
      ENDDO
      zarg(10)=zarg(10)+zk(107)
      DO KXY=1,9
        KX=KXY+2
        ZARA(KXY,KX)=ZK(KXY+97)
      ENDDO
C------Calcium
      DO KXY=2,11
        ALCAL(KXY)=ALCCA(KXY)
        KXZ=KXY-1
        ZCAL(KXZ)=ZCCA(KXZ)+DEN(I)*CONS1*DEL(I)*COCA(KXZ)
      ENDDO
      zcal(10)=zcal(10)+zk(117)
      DO KXY=1,9
        KX=KXY+2
        ZCALA(KXY,KX)=ZK(KXY+107)
      ENDDO
C/////////////////////////////////////////////////////////////////////  
C       COEFFICIENT MATRICES FOR THE METALS                      
C     (Carbon=1,Oxygen=2,Sulphur=3,Nitrogen=4,Neon=5,Iron=6,Silicon=7,
C      Sodium=8,Magnesium=9,Aluminum=10,Argon=11,Calcium=12)
C
C       e.g. ACA(1,JUP,K) IS THE MATRIX-ELEMENT IN THE CARBON MAT-      
C       RIX IN THE RATE EQUATION  FOR THE J:th ION DUE TO THE K:th      
C       ION.                                                            
C/////////////////////////////////////////////////////////////////////  
C------ Carbon                                                          
      DO 9480 KU=1,7                                                    
      DO 9480 KV=1,7                                                    
 9480 ACA(1,KV,KU)=0.0D0                                                
      ACA(1,1,1)=-(ZCA(1)+ZCAA(1,3))                    
      DO 342 KP=2,4                                                     
      ACA(1,KP,KP)=-(ZCA(KP)+ZCAA(KP,KP+2)+ALCA(KP) 
     &*DEN(I)*CONS1*DEL(I))                                             
  342 CONTINUE                                                          
      ACA(1,5,5)=-(ZCA(5)+ALCA(5)*DEN(I)*CONS1*DEL(I))                
      ACA(1,6,6)=-(ZCA(6)+ALCA(6)*DEN(I)*CONS1*DEL(I))                
      ACA(1,7,7)=-ALCA(7)*DEN(I)*CONS1*DEL(I)                           
      DO 344 KP=1,6                                                     
      KPE=KP+1                                                          
      ACA(1,KP,KPE)=ALCA(KPE)*DEN(I)*CONS1*DEL(I)                       
  344 ACA(1,KPE,KP)=ZCA(KP)                                           
      DO 346 KP=1,4                                                     
      KPE=KP+2                                                          
  346 ACA(1,KPE,KP)=ZCAA(KP,KPE)                                      
C-------Oxygen                                                          
      DO 9951 KU=1,9                                                    
      DO 9951 KV=1,9                                                    
 9951 ACA(2,KV,KU)=0.0D0                                                
      ACA(2,1,1)=-(ZOX(1)+ZOXA(1,3))                    
      DO 9342 KP=2,6                                                    
      ACA(2,KP,KP)=-(ZOX(KP)+ZOXA(KP,KP+2)+ALOX(KP) 
     &*DEN(I)*CONS1*DEL(I))                                             
 9342 CONTINUE                                                          
      ACA(2,7,7)=-(ZOX(7)+ALOX(7)*DEN(I)*CONS1*DEL(I))                
      ACA(2,8,8)=-(ZOX(8)+ALOX(8)*DEN(I)*CONS1*DEL(I))                
      ACA(2,9,9)=-ALOX(9)*DEN(I)*CONS1*DEL(I)                           
      DO 9344 KP=1,8                                                    
      KPE=KP+1                                                          
      ACA(2,KP,KPE)=ALOX(KPE)*DEN(I)*CONS1*DEL(I)                       
 9344 ACA(2,KPE,KP)=ZOX(KP)                                           
      DO 3991 KP=1,6                                                    
      KPE=KP+2                                                          
 3991 ACA(2,KPE,KP)=ZOXA(KP,KPE)                                      
C-------Sulphur                                                         
      DO 5451 KU=1,11                                                   
      DO 5451 KV=1,11                                                   
 5451 ACA(3,KV,KU)=0.0D0                                                
      ACA(3,1,1)=-(ZSU(1)+ZSUA(1,3))                    
      DO 5442 KP=2,9                                                    
      ACA(3,KP,KP)=-(ZSU(KP)+ZSUA(KP,KP+2)+ALSU(KP) 
     &*DEN(I)*CONS1*DEL(I))                                             
 5442 CONTINUE                                                          
      ACA(3,10,10)=-(ZSU(10)+ALSU(10)*DEN(I)*CONS1*DEL(I))              
      ACA(3,11,11)=-ALSU(11)*DEN(I)*CONS1*DEL(I)                        
      DO 5444 KP=1,10                                                   
      KPE=KP+1                                                          
      ACA(3,KP,KPE)=ALSU(KPE)*DEN(I)*CONS1*DEL(I)                       
 5444 ACA(3,KPE,KP)=ZSU(KP)                                           
      DO 2791 KP=1,9                                                    
      KPE=KP+2                                                          
 2791 ACA(3,KPE,KP)=ZSUA(KP,KPE)                                      
C-------Nitrogen                                                        
      DO 8851 KU=1,8                                                    
      DO 8851 KV=1,8                                                    
 8851 ACA(4,KV,KU)=0.0D0                                                
      ACA(4,1,1)=-(ZNI(1)+ZNIA(1,3))                    
      DO 3942 KP=2,5                                                    
      ACA(4,KP,KP)=-(ZNI(KP)+ZNIA(KP,KP+2)+ALNI(KP)                 
     &*DEN(I)*CONS1*DEL(I))                                             
 3942 CONTINUE                                                          
      ACA(4,6,6)=-(ZNI(6)+ALNI(6)*DEN(I)*CONS1*DEL(I))                
      ACA(4,7,7)=-(ZNI(7)+ALNI(7)*DEN(I)*CONS1*DEL(I))                
      ACA(4,8,8)=-ALNI(8)*DEN(I)*CONS1*DEL(I)                           
      DO 9544 KP=1,7                                                    
      KPE=KP+1                                                          
      ACA(4,KP,KPE)=ALNI(KPE)*DEN(I)*CONS1*DEL(I)                       
 9544 ACA(4,KPE,KP)=ZNI(KP)                                           
      DO 5991 KP=1,5                                                    
      KPE=KP+2                                                          
 5991 ACA(4,KPE,KP)=ZNIA(KP,KPE)                                      
C-------Neon                                                         
      DO 2151 KU=1,9                                                    
      DO 2151 KV=1,9                                                    
 2151 ACA(5,KV,KU)=0.0D0                                                
      ACA(5,1,1)=-(ZNE(1)+ZNEA(1,3))                    
      DO 2142 KP=2,7                                                    
      ACA(5,KP,KP)=-(ZNE(KP)+ZNEA(KP,KP+2)+ALNE(KP) 
     &*DEN(I)*CONS1*DEL(I))                                             
 2142 CONTINUE                                                          
      ACA(5,8,8)=-(ZNE(8)+ALNE(8)*DEN(I)*CONS1*DEL(I))                
      ACA(5,9,9)=-ALNE(9)*DEN(I)*CONS1*DEL(I)                           
      DO 2144 KP=1,8                                                    
      KPE=KP+1                                                          
      ACA(5,KP,KPE)=ALNE(KPE)*DEN(I)*CONS1*DEL(I)                       
 2144 ACA(5,KPE,KP)=ZNE(KP)                                           
      DO 2191 KP=1,7                                                    
      KPE=KP+2                                                          
 2191 ACA(5,KPE,KP)=ZNEA(KP,KPE)                                      
C-------Iron                                                         
      DO 5480 KU=1,16                                                  
      DO 5480 KV=1,16                                                   
 5480 ACA(6,KV,KU)=0.0D0                                                
      ACA(6,1,1)=-(ZFE(1)+zfea(1,3))
      DO 5342 KP=2,14                                                   
      ACA(6,KP,KP)=-(ZFE(KP)+zfea(kp,kp+2)+ALFE(KP)
     &*DEN(I)*CONS1*DEL(I))                                             
 5342 CONTINUE                                                          
      ACA(6,15,15)=-(ZFE(15)+ALFE(15)*DEN(I)*CONS1*DEL(I))            
      ACA(6,16,16)=-ALFE(16)*DEN(I)*CONS1*DEL(I)                        
      DO 5344 KP=1,15                                                   
      KPE=KP+1                                                          
      ACA(6,KP,KPE)=ALFE(KPE)*DEN(I)*CONS1*DEL(I)                       
 5344 ACA(6,KPE,KP)=ZFE(KP)                                           
      DO KP=1,14
	KPE=KP+2
	ACA(8,KPE,KP)=ZFEA(KP,KPE)
      ENDDO
C-------Silicon                                                         
      DO 5481 KU=1,11                                                   
      DO 5481 KV=1,11                                                   
 5481 ACA(7,KV,KU)=0.0D0                                                
      ACA(7,1,1)=-(ZSI(1)+ZSIA(1,3))
      DO 5347 KP=2,9                                                   
      ACA(7,KP,KP)=-(ZSI(KP)+ZSIA(KP,KP+2)+ALSI(KP)
     &*DEN(I)*CONS1*DEL(I))                                             
 5347 CONTINUE                                                          
      ACA(7,10,10)=-(ZSI(10)+ALSI(10)*DEN(I)*CONS1*DEL(I))            
      ACA(7,11,11)=-ALSI(11)*DEN(I)*CONS1*DEL(I)                        
      DO 5348 KP=1,10                                                   
      KPE=KP+1                                                          
      ACA(7,KP,KPE)=ALSI(KPE)*DEN(I)*CONS1*DEL(I)                       
 5348 ACA(7,KPE,KP)=ZSI(KP)                                           
      DO KP=1,9
	KPE=KP+2
	ACA(7,KPE,KP)=ZSIA(KP,KPE)
      ENDDO
C-------Sodium
      DO KU=1,10
         DO KV=1,10
            ACA(8,KV,KU)=0.0D0
         ENDDO
      ENDDO
      ACA(8,1,1)=-(ZNAT(1)+ZNAA(1,3))
      DO KP=2,8
         ACA(8,KP,KP)=-(ZNAT(KP)+ZNAA(KP,KP+2)+ALNAT(KP)*
     &   DEN(I)*CONS1*DEL(I))                                             
      ENDDO
      ACA(8,9,9)=-(ZNAT(9)+ALNAT(9)*DEN(I)*CONS1*DEL(I))
      ACA(8,10,10)=-ALNAT(10)*DEN(I)*CONS1*DEL(I)
C      DO KP=1,8 wrong !!!!!!!!
      DO KP=1,9
         KPE=KP+1                                                          
         ACA(8,KP,KPE)=ALNAT(KPE)*DEN(I)*CONS1*DEL(I)
         ACA(8,KPE,KP)=ZNAT(KP)
      ENDDO
      DO KP=1,8
	 KPE=KP+2
	 ACA(8,KPE,KP)=ZNAA(KP,KPE)
      ENDDO
C-------Magnesium
      DO KU=1,11
         DO KV=1,11
            ACA(9,KV,KU)=0.0D0
         ENDDO
      ENDDO
      ACA(9,1,1)=-(ZMAG(1)+ZMGA(1,3))
      DO KP=2,9
         ACA(9,KP,KP)=-(ZMAG(KP)+ZMGA(KP,KP+2)+ALMAG(KP)*
     &   DEN(I)*CONS1*DEL(I))                                             
      ENDDO
      ACA(9,10,10)=-(ZMAG(10)+ALMAG(10)*DEN(I)*CONS1*DEL(I))
      ACA(9,11,11)=-ALMAG(11)*DEN(I)*CONS1*DEL(I)
      DO KP=1,10
         KPE=KP+1                                                          
         ACA(9,KP,KPE)=ALMAG(KPE)*DEN(I)*CONS1*DEL(I)
         ACA(9,KPE,KP)=ZMAG(KP)
      ENDDO
      DO KP=1,9
	 KPE=KP+2
	 ACA(9,KPE,KP)=ZMGA(KP,KPE)
      ENDDO
C-------Aluminum
      DO KU=1,12
         DO KV=1,12
            ACA(10,KV,KU)=0.0D0
         ENDDO
      ENDDO
      ACA(10,1,1)=-(ZALU(1)+ZALA(1,3))
      DO KP=2,10
         ACA(10,KP,KP)=-(ZALU(KP)+ZALA(KP,KP+2)+ALALU(KP)*
     &   DEN(I)*CONS1*DEL(I))                                             
      ENDDO
      ACA(10,11,11)=-(ZALU(11)+ALALU(11)*DEN(I)*CONS1*DEL(I))
      ACA(10,12,12)=-ALALU(12)*DEN(I)*CONS1*DEL(I)
      DO KP=1,11
         KPE=KP+1                                                          
         ACA(10,KP,KPE)=ALALU(KPE)*DEN(I)*CONS1*DEL(I)
         ACA(10,KPE,KP)=ZALU(KP)
      ENDDO
      DO KP=1,10
	 KPE=KP+2
	 ACA(10,KPE,KP)=ZALA(KP,KPE)
      ENDDO
C-------Argon
      DO KU=1,11
         DO KV=1,11
            ACA(11,KV,KU)=0.0D0
         ENDDO
      ENDDO
      ACA(11,1,1)=-(ZARG(1)+ZARA(1,3))
      DO KP=2,9
         ACA(11,KP,KP)=-(ZARG(KP)+ZARA(KP,KP+2)+ALARG(KP)*
     &   DEN(I)*CONS1*DEL(I))                                             
      ENDDO
      ACA(11,10,10)=-(ZARG(10)+ALARG(10)*DEN(I)*CONS1*DEL(I))
      ACA(11,11,11)=-ALARG(11)*DEN(I)*CONS1*DEL(I)
      DO KP=1,10
         KPE=KP+1                                                          
         ACA(11,KP,KPE)=ALARG(KPE)*DEN(I)*CONS1*DEL(I)
         ACA(11,KPE,KP)=ZARG(KP)
      ENDDO
      DO KP=1,9
	 KPE=KP+2
	 ACA(11,KPE,KP)=ZARA(KP,KPE)
      ENDDO
C-------Calcium
      DO KU=1,11
         DO KV=1,11
            ACA(12,KV,KU)=0.0D0
         ENDDO
      ENDDO
      ACA(12,1,1)=-(ZCAL(1)+ZCALA(1,3))
      DO KP=2,9
         ACA(12,KP,KP)=-(ZCAL(KP)+ZCALA(KP,KP+2)+ALCAL(KP)*
     &   DEN(I)*CONS1*DEL(I))                                             
      ENDDO
      ACA(12,10,10)=-(ZCAL(10)+ALCAL(10)*DEN(I)*CONS1*DEL(I))
      ACA(12,11,11)=-ALCAL(11)*DEN(I)*CONS1*DEL(I)
      DO KP=1,10
         KPE=KP+1                                                          
         ACA(12,KP,KPE)=ALCAL(KPE)*DEN(I)*CONS1*DEL(I)
         ACA(12,KPE,KP)=ZCAL(KP)
      ENDDO
      DO KP=1,9
	 KPE=KP+2
	 ACA(12,KPE,KP)=ZCALA(KP,KPE)
      ENDDO
C                                                                       
C       CONTINUE WITH FURTHER ELEMENTS                                  
C                                                                       
C--------------------------------------------------------------------   
C       WE NOW SOLVE THE ION.RATES                                      
C--------------------------------------------------------------------   
C                                                                       
C       Hydrogen
      ZH(1)=ZK(1)
      do 6001 ih=1,levh
      x(ih)=qxh(i,1,ih)
      do 6002 ihh=1,levh
      a(ih,ihh)=hradm(ih,ihh)
 6002 continue
 6001 continue
      a(1,1)=a(1,1)+deltt*zh(1)
      do 6003 ih=1,levh-1
      do 6004 ihh=1,levh
      if(ihh.ne.ih) then
       a(ih,ih)=a(ih,ih)+deltt*den(i)*del(i)*hccoll(ih,ihh)
       a(ih,ihh)=a(ih,ihh)-deltt*den(i)*del(i)*hccoll(ihh,ih)
      endif
 6004 continue
 6003 continue
      do 6407 ih=1,levh
      a(levh,ih)=1.d0
 6407 continue
      x(levh)=1.d0
      call matrix(levh)
      do 6005 ih=1,levh
      if(x(ih).le.0.0d0) x(ih)=1.d-50
      if(x(ih).ge.1.d0) x(ih)=1.d0
 6005 qxh(i,2,ih)=x(ih)
      qxb(i,2,2)=qxh(i,2,levh)
      qxb(i,2,1)=1.d0-qxb(i,2,2)
      if(qxb(i,2,1).le.0.d0) qxb(i,2,1)=1.d-50
      if(qxb(i,2,2).le.0.d0) qxb(i,2,2)=1.d-50
      if(qxb(i,2,1).ge.1.d0) qxb(i,2,1)=1.d0
      if(qxb(i,2,2).ge.1.d0) qxb(i,2,2)=1.d0
C                                                                       
C       Helium
      do 7001 ih=1,levhe
      x(ih)=qxhe(i,1,ih)
      do 7002 ihh=1,levhe
      a(ih,ihh)=heradm(ih,ihh)
 7002 continue
 7001 continue
      a(1,1)=a(1,1)+deltt*zk(2)
      a(levhe1+1,1)=-deltt*zk(2)
      a(levhe1+1,levhe1+1)=a(levhe1+1,levhe1+1)+deltt*zk(3)
      do 7003 ih=1,levhe1+1
      do 7004 ihh=1,levhe1+1
      if(ihh.ne.ih) then
       a(ih,ih)=a(ih,ih)+deltt*den(i)*del(i)*hecoll(ih,ihh)
       a(ih,ihh)=a(ih,ihh)-deltt*den(i)*del(i)*hecoll(ihh,ih)
      endif
 7004 continue
 7003 continue
      do 7023 ih=levhe1+1,levhe-1
      do 7024 ihh=levhe1+1,levhe
      if(ihh.ne.ih) then
       a(ih,ih)=a(ih,ih)+deltt*den(i)*del(i)*hecoll(ih,ihh)
       a(ih,ihh)=a(ih,ihh)-deltt*den(i)*del(i)*hecoll(ihh,ih)
      endif
 7024 continue
 7023 continue
c     qhemax=dmax1(qxhe(i,1,1),qxhe(i,1,levhe1+1),qxhe(i,1,levhe))
c     if(qhemax.eq.qxhe(i,1,1)) iihe=1
c     if(qhemax.eq.qxhe(i,1,levhe1+1)) iihe=levhe1+1
c     if(qhemax.eq.qxhe(i,1,levhe)) iihe=levhe
      do 6507 ih=1,levhe
      a(levhe,ih)=1.d0
 6507 continue
      x(levhe)=1.d0
      call matrix(levhe)
      do 7005 ih=1,levhe
      if(x(ih).le.0.0d0) x(ih)=1.d-50
      if(x(ih).ge.1.d0) x(ih)=1.d0
 7005 qxhe(i,2,ih)=x(ih)
      qxb(i,2,4)=0.0d0
      do ih=levhe1+1,levhe-1
         qxb(i,2,4)=qxb(i,2,4)+x(ih)
      enddo
      qxb(i,2,5)=x(levhe)
      qxb(i,2,3)=0.0d0
      do ih=1,levhe1
         qxb(i,2,3)=qxb(i,2,3)+x(ih)
      enddo
      if(qxb(i,2,3).le.0.d0) qxb(i,2,2)=1.d-50
      if(qxb(i,2,4).le.0.d0) qxb(i,2,1)=1.d-50
      if(qxb(i,2,5).le.0.d0) qxb(i,2,1)=1.d-50
      if(qxb(i,2,3).ge.1.d0) qxb(i,2,3)=1.d0
      if(qxb(i,2,4).ge.1.d0) qxb(i,2,4)=1.d0
      if(qxb(i,2,5).ge.1.d0) qxb(i,2,5)=1.d0
C--------------------------------------------------------------------   
C        METALS                                            
C
C    Number of ionization levels=...
C    N(C)=7, N(O)=9, N(S)=11, N(N)=8, N(Ne)=9, N(Fe)=16, N(Si)=11,
C    N(Na)=10, N(Mg)=11, N(Al)=12, N(Ar)=11 and N(Ca)=11
C    Identifier (IONIX)=...
C    (Carbon=1,Oxygen=2,Sulphur=3,Nitrogen=4,Neon=5,Iron=6,Silicon=7,
C     Sodium=8,Magnesium=9,Aluminum=10,Argon=11,Calcium=12)
C                                                                       
C     New matrix routine 8/11-87; Na,Mg,Al,Ar,Ca added 18/10-95
C--------------------------------------------------------------------   
      MU=5                                                              
      N=0                                                               
      DO 613 IONIX=1,12
      MU=MU+N                                                           
      IF(IONIX.EQ.1) N=7                                                
      IF(IONIX.EQ.2) N=9                                                
      IF(IONIX.EQ.3) N=11                                               
      IF(IONIX.EQ.4) N=8                                                
      IF(IONIX.EQ.5) N=9                                                
      IF(IONIX.EQ.6) N=16
      IF(IONIX.EQ.7) N=11
      IF(IONIX.EQ.8) N=10
      IF(IONIX.EQ.9) N=11
      IF(IONIX.EQ.10) N=12
      IF(IONIX.EQ.11) N=11
      IF(IONIX.EQ.12) N=11
      M=N+1                                                             
      DO 9243 IZ=1,M                                                   
      DO 9243 JZ=1,M                                                   
 9243 A(IZ,JZ)=0.0                                                      
      DO 108 IZ=1,N-1                                              
      DO 108 JZ=1,N                                                     
  108 A(IZ,JZ)=-DELTT*ACA(IONIX,IZ,JZ)                                
      DO 1238 IZ=1,N-1                                               
      A(IZ,IZ)=1.D0+A(IZ,IZ)                                            
 1238 X(IZ)=QXB(I,JUP-1,IZ+MU)                                          
C -- Number conservation (overwrite the dominant ion's row) --
      qimax=0.0d0
       do 9239 iz=1,n
       qiold=qimax
       qimax=dmax1(qimax,qxb(i,jup-1,iz+mu))
       if(qimax.gt.qiold) izion=n
 9239  continue
      do 1239 iz=1,n
 1239 a(izion,iz)=1.d0
      x(izion)=1.d0
      CALL MATRIX (N)                                                   
      MA=MU+1                                                           
      MB=MU+N                                                           
      DO 7514 KX=MA,MB                                                  
      if(x(kx-mu).lt.0.0d0) x(kx-mu)=1.d-50
 7514 QXB(I,JUP,KX)=X(KX-MU)                                            
  613 CONTINUE                                                          
C--------------------------------------------------------------------   
C      NEXT WE COMPUTE THE NEW ELECTRON DENSITY (ACC. TO ALLEN-73)      
        DEOLD(I)=DEL(I)                                                 
      MELECT=3                                                          
      CALL ELDENS(MELECT)                                               
      DEL(I)=DEPE                                                       
 1517 TAX=DABS(DEL(I)-DEOLD(I))                                         
C--------------------------------------------------------------------   
C     WE NOW RETRANSLATE QXB(I,JUP,K) TO XB(I,K)                        
C--------------------------------------------------------------------   
      XB(I,1)=QXB(I,JUP,1)                                              
      XB(I,2)=QXB(I,JUP,3)                                              
      XB(I,3)=QXB(I,JUP,4)                                              
      DO 372 K=4,6                                                      
  372 XB(I,K)=QXB(I,JUP,K+14)                                           
      XB(I,7)=QXB(I,JUP,17)                                             
      DO 1938 K=8,11                                                    
 1938 XB(I,K)=QXB(I,JUP,K)                                              
      XB(I,12)=QXB(I,JUP,6)                                             
      XB(I,13)=QXB(I,JUP,7)                                             
      DO 7902 K=14,17                                                   
 7902 XB(I,K)=QXB(I,JUP,K-1)                                            
      DO 7909 K=18,24                                                   
 7909 XB(I,K)=QXB(I,JUP,K+15)                                           
      DO 7911 K=25,34
 7911 XB(I,K)=QXB(I,JUP,K-3)
      DO 7912 K=35,42
 7912 XB(I,K)=QXB(I,JUP,K+6)
      DO 7913 K=43,57
 7913 XB(I,K)=QXB(I,JUP,K+7)
      DO 7914 K=58,67
 7914 XB(I,K)=QXB(I,JUP,K+8)
      DO K=68,76
         XB(I,K)=QXB(I,JUP,K+9)
      ENDDO
      DO K=77,86
         XB(I,K)=QXB(I,JUP,K+10)
      ENDDO
      DO K=87,97
         XB(I,K)=QXB(I,JUP,K+11)
      ENDDO
      DO K=98,107
         XB(I,K)=QXB(I,JUP,K+12)
      ENDDO
      DO K=108,117
         XB(I,K)=QXB(I,JUP,K+13)
      ENDDO
      DO 6591 K=1,117
 6591 XA(2,K)=XB(I,K)                                                   
      IF(TAX/DEL(I).GT.1.D-6) GOTO 1772                                 
C                                                                       
C----- end of electron density-loop                                     
C----------------------------------------------------------------       
 6938 format(1x,3e10.3)
      RETURN                                                            
      END                                                               
C ***********************************************************************
C************************************************************           
C    Provides column densities for all ions.
C    colni(,) = column density towards inner edge, etc.
C************************************************************           
      subroutine colden(imax)
C
      implicit real*8(a-h,o-z)
C
      parameter (md=75)
C
      common/column/colni(md,117),colno(md,117)
      COMMON/PHY/DEN(md)                                                
      common/abun2/abunda(117)
      COMMON/ION/XB(md,117)
      common/radabs/rfix(md),rbound(md),dxabs(3,md,md)
C      
       do 100 i=1,imax
        do 101 k=1,117
        colni(i,k)=0.0d0
        colno(i,k)=0.0d0
         do 102 j=1,i
         colni(i,k)=colni(i,k)+abunda(k)*den(j)*xb(j,k)*dxabs(1,i,j)
  102    continue
         do 103 j=i,imax
         colno(i,k)=colno(i,k)+abunda(k)*den(j)*xb(j,k)*dxabs(3,i,j)
  103    continue
  101   continue
  100  continue
C
      return
      end
C************************************************************           
C************************************************************           
C    Translates between CF:s ion-induces and xc, xn etc.
C    ind = 1 (from CF to xc etc.) = 2 (the other way around)
C************************************************************           
      subroutine trlate(ind,i)
C
      implicit real*8(a-h,o-z)
C
      parameter (md=75)
C
      COMMON/ABU/XA(2,117)                                               
      common/absimp/xc(7),xn(8),xo(9),xne(9),xs(11),xfe(16),
     &xsi(11),xna(10),xmg(11),xal(12),xar(11),xca(11)
C      
      if(ind.eq.1) then
       xc(1)=xa(2,12)
       xc(2)=xa(2,13)
       do 100 k=3,6
  100  xc(k)=xa(2,k+5)
       do 101 k=1,7
  101  xn(k)=xa(2,k+17)      
       do 102 k=1,4
  102  xo(k)=xa(2,k+13)
       xo(5)=xa(2,7)
       do 103 k=6,8
  103  xo(k)=xa(2,k-2)
       do 104 k=1,8
  104  xne(k)=xa(2,k+34)
       do 105 k=1,10
  105  xs(k)=xa(2,k+24)
       do 106 k=1,15
  106  xfe(k)=xa(2,k+42)
       do 107 k=1,10
  107  xsi(k)=xa(2,k+57)
       do k=1,9
	 xna(k)=xa(2,k+67)
       enddo
       do k=1,10
	 xmg(k)=xa(2,k+76)
       enddo
       do k=1,11
	 xal(k)=xa(2,k+86)
       enddo
       do k=1,10
	 xar(k)=xa(2,k+97)
       enddo
       do k=1,10
	 xca(k)=xa(2,k+107)
       enddo
      else
       xa(2,12)=xc(1)
       xa(2,13)=xc(2)
       do 200 k=3,6
  200  xa(2,k+5)=xc(k)
       do 201 k=1,7
  201  xa(2,k+17)=xn(k)
       do 202 k=1,4
  202  xa(2,k+13)=xo(k)
       xa(2,7)=xo(5)
       do 203 k=6,8
  203  xa(2,k-2)=xo(k)
       do 204 k=1,8
  204  xa(2,k+34)=xne(k)
       do 205 k=1,10
  205  xa(2,k+24)=xs(k)
       do 206 k=1,15
  206  xa(2,k+42)=xfe(k)
       do 207 k=1,10
  207  xa(2,k+57)=xsi(k)
       do 208 k=1,67
  208  xa(1,k)=xa(2,k)
       do k=1,9
	 xa(2,k+67)=xna(k)
       enddo
       do k=1,10
	 xa(2,k+76)=xmg(k)
       enddo
       do k=1,11
	 xa(2,k+86)=xal(k)
       enddo
       do k=1,10
	 xa(2,k+97)=xar(k)
       enddo
       do k=1,10
	 xa(2,k+107)=xca(k)
       enddo
      endif
C
      return
      end
C ***********************************************************************
C ***********************************************************************
      SUBROUTINE CROSS
C ***********************************************************************
C   (28/2-93)
C   Uses the Reilman & Manson (1979) Tables to estimate cross sections.
C
C   Cross sections have been moved in energy space from the (theoretical)
C   energies given by R & M to agree with better estimates for threshold
C   energies.
C
C   Threshold energies are from: 
C    Allen ('73) for ground levels;  Gould & Jung, ApJ 373, 271 ('91) for 
C    K-shell ionizations of stages I and II;  Arnaud & Rothenflug, A & A
C    Suppl. 60, 425 ('85) for most other configurations;  Reilman & Manson
C    for remaining configurations.
C
C   For energies above 5 keV ....
C   
C   ('94-'95)
C   Updated for C, N, O and Fe with TopBase data. TopBase also used for
C   Na, Mg, Al, Ar & Ca. He I updated 17/10-95 with experimental data.
C ***********************************************************************
      implicit real*8(a-h,o-z)
C
      parameter (md=75)
C
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)       
      COMMON/FRE/JMIN,JJ                                                
      COMMON/TRES/EL(117),EK(117)
      COMMON/ENTRES/E00(117),DFLUX                                       
      COMMON/SIK/SK(117,-13:75)                                          
      common/ENEWS/SIGMA(7,89),EP(7),IC(7),IREAD(7)
      COMMON/ENEWS2/ETEMP(90),ETEMPLG(90),sgtemp(90),sgtemplg(90),
     &ee(7),e1log(-13:76)
      COMMON/ENEWS3/efix(89)
      COMMON/ENEWS4/eex(7,15,7),jmi(7,15),jma(7,15),icx(7,15,7),
     &lev(7),kread(7),ksave(7,15),ireadx(7,15,7)
      COMMON/TWOHES/he1_2g(-13:75),he2_2g(-13:75),jhe1_2g,jhe2_2g
C
      ELCH=1.6022D-12                                                   
      misp=7
C -- Empty matrices and vectors
      do i=1,misp
	lev(i)=0
	kread(i)=0
	do j=1,15
	  jmi(i,j)=1
	  if(i.eq.7) jmi(i,j)=2
	  jma(i,j)=0
	  ksave(i,j)=0
	  do k=1,7
	    icx(i,j,k)=0
	    ireadx(i,j,k)=0
	    eex(i,j,k)=0.0d0
	  enddo
	enddo
      enddo
      do k=1,117
	el(k)=0.0d0
	ek(k)=0.0d0
	e00(k)=0.0d0
	do j=-13,75
	  si(k,j)=0.0d0
	  sk(k,j)=0.0d0
	enddo
      enddo
C -- Define energy grid used in the main program
      call energy
      do j=jmin,75
	e1log(j)=dlog10(e1(j))
      enddo
C -- Define 2-photon distribution on this grid (He I and He II)
      e_he1=20.616d0
      e_he2=40.794d0
      sum_he1=0.0d0
      sum_he2=0.0d0
      iestop=0
      do j=jmin,75
        if(e(j+1).gt.e_he1) iestop=1
        ymin=e(j)/e_he1
        ymax=e(j+1)/e_he1
        if(iestop.eq.1) ymax=0.999999d0
        he1_2g(j)=twophf(ymin,ymax)
        sum_he1=sum_he1+he1_2g(j)
        if(iestop.eq.1) jhe1_2g=j
        if(iestop.eq.1) goto 7278
      enddo
 7278 continue
      iestop=0
      do j=jmin,75
        if(e(j+1).gt.e_he2) iestop=1
        ymin=e(j)/e_he2
        ymax=e(j+1)/e_he2
        if(iestop.eq.1) ymax=0.999999d0
        he2_2g(j)=twophf(ymin,ymax)
        sum_he2=sum_he2+he2_2g(j)
        if(iestop.eq.1) jhe2_2g=j
        if(iestop.eq.1) goto 7279
      enddo
 7279 continue
C
      CALL CRDATA(misp)
      E00(1)=EL(1)                                                      
      E00(2)=EL(2)                                                      
      E00(3)=EL(3)                                                      
      E00(4)=EL(12)                                                     
      E00(5)=EL(13)                                                     
      E00(6)=EL(8)                                                      
      E00(7)=EL(9)                                                      
      E00(8)=EL(10)                                                     
      E00(9)=EL(11)                                                     
      E00(10)=EL(14)                                                    
      E00(11)=EL(15)                                                    
      E00(12)=EL(16)                                                    
      E00(13)=EL(17)                                                    
      E00(14)=EL(7)                                                     
      E00(15)=EL(4)                                                     
      E00(16)=EL(5)                                                     
      E00(17)=EL(6)                                                     
      DO 9375 K=18,117                                                   
 9375 E00(K)=EL(K)                                                      
      DO 9376 K=1,117                                                   
 9376 E00(K)=ELCH*E00(K)                                                
C
C -- Define energy grid used by Reilman & Manson
      do i=1,4
	efix(i)=6.d0+dble(i-1)
      enddo
      do i=5,12
	efix(i)=10.d0+5.d0*dble(i-5)
      enddo
      do i=13,18
	efix(i)=5.d1+1.d1*dble(i-13)
      enddo
      efix(19)=1.3d2
      efix(20)=1.6d2
      efix(21)=1.9d2
      do i=22,48
	efix(i)=2.1d2+3.d1*dble(i-22)
      enddo
      do i=49,89
	efix(i)=1.d3+1.d2*dble(i-49)
      enddo
    9 format(a)
C
C - Extract cross sections.
C
      do isp=1,misp
        do m=1,lev(isp)
	  jminn=jmi(isp,m)
	  jmax=jma(isp,m)
	  do j=jminn,jmax
	    ic(j)=icx(isp,m,j)
	    ee(j)=eex(isp,m,j)
	    iread(j)=ireadx(isp,m,j)
          enddo
	  do j=jminn,jmax
	    if(iread(j).ne.0) then
	      do i=1,89
	        sigma(j,i)=0.0d0
	      enddo
	    endif
	  enddo
          CALL CRREAD(kread(isp),jminn,jmax)
	  ksve=ksave(isp,m)
	  CALL CRIPOL(jminn,jmax,ksve)
        enddo
      enddo
C 
C -- Extrapolate if jj > 57 (i.e., E > 5 keV)
C   (do not change 4,67 to anything else !!)
      if(jj.gt.57) then
	do k=4,67
	if(k.ne.11.or.k.ne.24) then
	  if(k.ne.6) then
	    do j=58,jj
	    fsige=(dlog10(e1(j))-dlog10(e1(56)))/
     &            (dlog10(e1(57))-dlog10(e1(56)))
	    if(sk(k,56).gt.1.d-100.and.sk(k,57).gt.1.d-100) then
	      sk(k,j)=1.d1**((1.d0-fsige)*dlog10(sk(k,56))+
     &                       fsige*dlog10(sk(k,57)))
            endif
	    if(si(k,56).gt.1.d-100.and.si(k,57).gt.1.d-100) then
	      si(k,j)=1.d1**((1.d0-fsige)*dlog10(si(k,56))+
     &                       fsige*dlog10(si(k,57)))
            endif
            enddo
	  endif
	endif
	enddo
C -- Iron K-shell (Extrapolate in R&M:s tables for Mg, Si and S.)
	do j=58,jj
	  do k=43,57
	  if(e1(j).gt.ek(k)) then
	    sk(k,j)=3.4d-20*(ek(k)/e1(j))**2.6
	  endif
	  enddo
	enddo
      endif
C
      CALL HE1CROSS
C
      do j=2,jj
C -- HI
      IF(J.GE.2) THEN
	call CRH(1.d0,e1(j),skh)
	sk(1,j)=skh
      ENDIF
C -- HE I (prior to loop)
C      IF(J.GE.7.AND.J.LE.15) THEN
C        SK(2,J)=SK(1,J)*(6.53*(E1(J)/24.6)-0.22)
C      ELSEIF(J.GT.15) THEN
C        SK(2,J)=SK(1,J)*(37.-19.1*(65.4/E1(J))**0.76)
C      ENDIF
C -- HE II
      IF(J.GE.14) THEN
	call CRH(2.d0,e1(j),skh)
	sk(3,j)=skh
      ENDIF
C -- C VI
      IF(J.GE.38) THEN
        SIA=1.287*(490./E1(J))**2.95-.287*(490./E1(J))**3.95
        SK(11,J)=0.194D-18*SIA
      ENDIF
C -- N VII
      IF(J.GE.45) THEN
        SK(24,J)=0.142D-18*(1.287*(667./E1(J))**2.95-.287*(667.
     &  /E1(J))**3.95)
      ENDIF
C -- O VIII
      IF(J.GE.48) THEN
        SK(6,J)=0.109D-18*(1.287*(871./E1(J))**2.95-.287*(871./E1(J))
     &  **3.95)                                                           
      ENDIF
C -- Rough Fe II correction (Le Dourneuf et al. '93, J.Phys.B, 26, L1)
      if(e1(j).gt.16.16d0.and.e1(j).lt.25.d0) then
	si(44,j)=si(44,j)*2.2d0*(e1(j)/16.16d0)**(-1.807)
      endif
      enddo
C -- Opacity Project Data for C, N, O, Fe III-XV, Na, Mg, Al, Ar and Ca.
      do k=1,6
      read(54,*)dumi1
      if(k.eq.1) kkk=12
      if(k.eq.2) kkk=13
      if(k.eq.3) kkk=8
      if(k.eq.4) kkk=9
      if(k.eq.5) kkk=10
      if(k.eq.6) kkk=11
        do j=-7,65
        read(54,*)dumen1,dumen2,sk11,sk12,sk13
	si(kkk,j)=sk12
	sk(kkk,j)=sk13
	enddo
      enddo
      do k=18,24
      read(55,*)dumi1
        do j=-7,65
        read(55,*)dumen1,dumen2,sk11,sk12,sk13
	si(k,j)=sk12
	sk(k,j)=sk13
	enddo
      enddo
      do k=1,8
      read(56,*)dumi1
      if(k.eq.1) kkk=14
      if(k.eq.2) kkk=15
      if(k.eq.3) kkk=16
      if(k.eq.4) kkk=17
      if(k.eq.5) kkk=7
      if(k.eq.6) kkk=4
      if(k.eq.7) kkk=5
      if(k.eq.8) kkk=6
        do j=-7,65
        read(56,*)dumen1,dumen2,sk11,sk12,sk13
	si(kkk,j)=sk12
	sk(kkk,j)=sk13
	enddo
      enddo
      do k=45,57
      read(59,*)dumi1
        do j=-7,65
        read(59,*)dumen1,dumen2,sk11,sk12,sk13,sk14
	si(k,j)=sk12+sk13
	sk(k,j)=sk14
	enddo
      enddo
      do k=68,76
      read(57,*)dumi1
        do j=-7,65
        read(57,*)dumen1,dumen2,sk11,sk12,sk13,sk14
	si(k,j)=sk12+sk13
	sk(k,j)=sk14
	enddo
      enddo
      do k=77,86
      read(58,*)dumi1
        do j=-7,65
        read(58,*)dumen1,dumen2,sk11,sk12,sk13,sk14
	si(k,j)=sk12+sk13
	sk(k,j)=sk14
	enddo
      enddo
      do k=87,97
      read(60,*)dumi1
        do j=-7,65
        read(60,*)dumen1,dumen2,sk11,sk12,sk13,sk14
	si(k,j)=sk12+sk13
	sk(k,j)=sk14
	enddo
      enddo
      do k=98,107
      read(61,*)dumi1
        do j=-7,65
        read(61,*)dumen1,dumen2,sk11,sk12,sk13,sk14
	si(k,j)=sk12+sk13
	sk(k,j)=sk14
	enddo
      enddo
      do k=108,117
      read(62,*)dumi1
        do j=-7,65
        read(62,*)dumen1,dumen2,sk11,sk12,sk13,sk14
	si(k,j)=sk12+sk13
	sk(k,j)=sk14
	enddo
      enddo
C
      RETURN
      END
C ***********************************************************************
C ***********************************************************************
      double precision function twophf(ymin,ymax)
C
      implicit real*8(a-h,o-z)
C
      dimension f(51)
C
      pi=3.14159265d0
      nint=25
      dy=(ymax-ymin)/dble(nint)
      do n=1,nint+1
         y=ymin+dy*dble(n-1)
         f(n)=2.623d0*dsqrt(dcos(pi*(y-0.5d0)))
      enddo
      sum=0.5d0*(f(1)+f(nint+1))
      do n=2,nint
         sum=sum+f(n)
      enddo
      twophf=sum*dy
C
      return
      end
C ***********************************************************************
C ***********************************************************************
      SUBROUTINE CRH(Z,E,SK)
C
      implicit real*8(a-h,o-z)
C
      XI=13.6*Z**2
      PS=DSQRT((E-XI)/XI)
      FE=EXPFN(4.*DATAN(PS)/PS)/(1.-EXPFN(6.283/PS))
      SK=(3.402D-16/Z**2)*(XI/E)**4.*FE
C
      RETURN
      END
C ***********************************************************************
C ***********************************************************************
      SUBROUTINE CRDATA(misp)
C
      implicit real*8(a-h,o-z)
C
      COMMON/TRES/EL(117),EK(117)
      COMMON/ENEWS4/eex(7,15,7),jmi(7,15),jma(7,15),icx(7,15,7),
     &lev(7),kread(7),ksave(7,15),ireadx(7,15,7)
C
      dimension EKDAT(117),ELDAT(117),levd(7),kreadd(7),ksave1(5),
     &ksave2(6),ksave3(7),ksave4(8),ksave5(10),ksave6(10),
     &ksave7(15),jma1(5),jma2(6),jma3(7),jma4(8),jma5(10),
     &jma6(10),jma7(15),eneca(5,3),eneni(6,3),eneox(7,3),enene(8,3),
     &enesi(10,5),enesu(10,5),enefe(15,7),ireadca(5,3),ireadni(6,3),
     &ireadox(7,3),ireadne(8,3),ireadsi(10,5),ireadsu(10,5),
     &ireadfe(15,7),icca(5,3),icni(6,3),icox(7,3),icne(8,3),
     &icsi(10,5),icsu(10,5),icfe(15,7)
C
      DATA EKDAT/13.6,25.4,54.4,670.,739.32,870.,644.,325.,343.,
     &392.1,490.,297.4,315.2,546.,567.9,583.3,613.8,412.4,432.2,
     &446.9,471.0,493.0,552.06,667.03,2479.9,2496.2,2425.,2492.1,
     &2512.3,2534.3,2558.9,2624.7,2694.9,2769.3,869.4,895.4,911.7,
     &946.5,985.9,1029.6,1073.0,1107.0,7123.6,7126.0,7127.0,7146.0,
     &7170.0,7196.0,7225.0,7257.0,7291.0,7330.0,7372.0,7415.0,
     &7461.0,7508.0,7557.0,1822.9,1833.5,1845.6,1862.8,1881.7,
     &1934.7,1991.0,2025.5,2118.2,2187.8,1131.,1148.,1167.,1189.,
     &1214.,1244.,1280.,1325.,1363.,1292.5,1297.3,1314.4,1349.1,
     &1392.2,1441.1,1493.6,1550.3,1610.9,1667.5,1586.8,1605.8,
     &1625.8,1648.8,1673.8,1702.8,1735.8,1774.8,1821.8,1915.8,
     &1961.8,3129.5,3150.5,3173.5,3197.5,3224.5,3252.5,3283.5,
     &3317.5,3355.5,3397.5,3943.9,3965.9,3988.9,4013.9,4039.9,
     &4067.9,4097.9,4130.9,4166.9,4205.9/
      DATA ELDAT/13.59,25.4,54.4,138.12,739.32,870.,113.9,47.89,
     &64.49,392.1,490.,11.26,24.383,13.618,35.117,54.934,77.413,
     &14.534,29.601,47.448,77.472,97.89,552.06,667.03,10.36,23.313,
     &34.83,47.3,72.68,88.05,280.01,328.33,379.1,447.1,21.564,
     &40.962,63.45,97.11,126.,158.,207.,239.,7.87,16.16,30.651,54.8,
     &75.5,100.,128.3,151.12,235.,262.1,290.4,331.,361.,392.,457.,
     &8.151,16.345,33.492,45.141,166.77,205.08,246.49,303.16,351.1,
     &401.4,5.139,47.286,71.64,98.91,138.4,172.15,204.48,264.19,
     &299.9,7.646,15.035,80.143,109.31,141.27,186.51,224.95,265.92,
     &328.0,367.5,5.986,18.826,28.448,119.99,153.75,190.47,241.44,
     &284.59,330.2,398.6,442.0,15.759,27.629,40.74,59.81,75.04,
     &91.01,124.4,143.45,422.6,478.9,6.113,11.871,50.91,67.15,
     &84.43,108.78,127.7,147.4,188.7,211.3/
      data levd/5,6,7,8,10,10,15/,kreadd/41,42,43,44,45,46,47/
      data ksave1/12,13,8,9,10/,ksave2/18,19,20,21,22,23/,
     &ksave3/14,15,16,17,7,4,5/,ksave4/35,36,37,38,39,40,41,42/,
     &ksave5/58,59,60,61,62,63,64,65,66,67/,
     &ksave6/25,26,27,28,29,30,31,32,33,34/,
     &ksave7/43,44,45,46,47,48,49,50,51,52,53,54,55,56,57/
      data jma1/3,3,2,2,1/,jma2/3,3,3,2,2,1/,jma3/3,3,3,3,2,2,1/,
     &jma4/3,3,3,3,3,3,2,2/,jma5/5,5,4,4,3,3,3,3,3,3/,
     &jma6/5,5,5,5,4,4,3,3,3,3/,jma7/7,7,6,6,6,6,6,6,5,5,5,5,5,5,4/
      data eneca/297.4,315.2,325.,343.,392.1,16.6,30.9,47.89,64.49,
     &0.,11.26,24.383,3*0./
      data eneni/412.4,432.2,446.9,471.,493.,552.06,20.3,36.7,55.8,
     &77.472,97.89,0.,14.534,29.601,47.448,3*0./
      data eneox/546.,567.,583.3,613.8,644.,670.,739.32,28.5,42.6,
     &63.8,87.6,113.90,138.12,0.,13.618,35.117,54.934,77.413,3*0./
      data enene/869.4,895.4,911.7,946.5,985.9,1029.6,1073.,1107.,
     &48.5,66.4,86.2,108.,139.,172.,207.26,239.09,21.564,40.962,
     &63.45,97.11,126.21,157.93,2*0./
      data enesi/1848.6,1862.2,1845.6,1862.8,1881.7,1934.3,1991.,
     &2025.5,2118.2,2187.8,150.8,161.1,176.6,193.5,217.,250.,285.,
     &321.,371.,423.,108.2,118.6,133.,148.,166.77,205.08,246.49,
     &303.2,351.1,401.4,13.5,22.9,33.492,45.141,6*0.,8.151,16.345,
     &8*0./
      data enesu/2479.9,2496.2,2425.,2492.1,2512.3,2534.3,2558.9,
     &2624.7,2694.9,2769.3,224.5,237.3,252.1,268.8,288.2,309.7,
     &343.,384.,426.,469.,171.3,184.5,199.5,216.2,239.,257.,280.01,
     &328.33,379.1,447.1,20.2,30.7,43.8,57.6,72.68,88.05,4*0.,
     &10.36,23.33,34.83,47.3,6*0./
      data enefe/7123.6,7126.,7127.,7146.,7170.,7196.,7225.,7257.,
     &7291.,7330.,7372.,7415.,7461.,7508.,7557.,829.,838.,849.1,
     &871.,896.3,924.6,955.6,989.2,1025.3,1062.5,1101.7,1141.2,
     &1182.6,1225.3,1254.3,721.9,731.,742.,763.9,789.2,817.4,
     &848.5,882.2,918.3,956.1,995.4,1035.1,1078.3,1121.8,1185.,
     &98.8,108.,141.,162.,184.,205.,227.,249.,271.,297.,324.,
     &356.,388.,421.,457.,59.,81.,103.,125.,147.,169.,190.,213.,
     &235.,262.1,290.4,331.,361.,392.,0.,9.0,17.5,30.651,54.8,
     &75.5,100.,128.3,151.12,7*0.,7.87,16.16,13*0./
      data ireadca/1,4*0,4*1,0,2*1,3*0/,ireadni/1,5*0,5*1,0,3*1,
     &3*0/,ireadox/1,6*0,6*1,0,4*1,3*0/,ireadne/1,7*0,14*1,2*0/,
     &ireadsi/1,9*0,1,4*0,6*1,4*0,9*1,6*0,2*1,8*0/,
     &ireadsu/1,9*0,1,6*0,4*1,7*0,8*1,4*0,4*1,6*0/,
     &ireadfe/15*0,1,14*0,1,14*0,1,2*0,13*1,2*0,11*1,0,1,2*0,5*1,
     &7*0,2*1,13*0/
      data icca/5*25,7,10,13,15,0,4,8,3*0/,icni/6*29,8,11,14,16,18,
     &0,6,9,12,3*0/,icox/7*33,9,13,15,17,19,20,0,6,11,14,16,3*0/,
     &icne/8*44,12,15,17,19,20,21,22,23,7,12,15,18,19,20,2*0/,
     &icsi/10*58,5*20,24,25,26,28,30,5*19,22,24,25,27,29,6,8,10,
     &13,6*0,2,6,8*0/,icsu/10*64,7*23,28,30,31,8*21,28,30,8,10,
     &12,14,16,17,4*0,6,8,10,13,6*0/,icfe/15*0,15*43,15*40,3*18,
     &20,21,21,23,23,24,25,26,27,28,30,31,3*15,19,19,20,21,22,
     &23,24,25,26,27,28,0,3*6,14,16,18,19,20,7*0,3,7,13*0/

C
      do i=1,117
	ek(i)=ekdat(i)
	el(i)=eldat(i)
      enddo
      do isp=1,misp
	kread(isp)=kreadd(isp)
	lev(isp)=levd(isp)
	do j=1,lev(isp)
	  if(isp.eq.1) then
	    jma(isp,j)=jma1(j) 
	    ksave(isp,j)=ksave1(j)
	    do k=1,3
	      icx(isp,j,k)=icca(j,k)
	      ireadx(isp,j,k)=ireadca(j,k)
	      eex(isp,j,k)=eneca(j,k)
	    enddo
	  elseif(isp.eq.2) then
	    jma(isp,j)=jma2(j) 
	    ksave(isp,j)=ksave2(j)
	    do k=1,3
	      icx(isp,j,k)=icni(j,k)
	      ireadx(isp,j,k)=ireadni(j,k)
	      eex(isp,j,k)=eneni(j,k)
	    enddo
	  elseif(isp.eq.3) then
	    jma(isp,j)=jma3(j) 
	    ksave(isp,j)=ksave3(j)
	    do k=1,3
	      icx(isp,j,k)=icox(j,k)
	      ireadx(isp,j,k)=ireadox(j,k)
	      eex(isp,j,k)=eneox(j,k)
	    enddo
	  elseif(isp.eq.4) then
	    jma(isp,j)=jma4(j) 
	    ksave(isp,j)=ksave4(j)
	    do k=1,3
	      icx(isp,j,k)=icne(j,k)
	      ireadx(isp,j,k)=ireadne(j,k)
	      eex(isp,j,k)=enene(j,k)
	    enddo
	  elseif(isp.eq.5) then
	    jma(isp,j)=jma5(j) 
	    ksave(isp,j)=ksave5(j)
	    do k=1,5
	      icx(isp,j,k)=icsi(j,k)
	      ireadx(isp,j,k)=ireadsi(j,k)
	      eex(isp,j,k)=enesi(j,k)
	    enddo
	  elseif(isp.eq.6) then
	    jma(isp,j)=jma6(j) 
	    ksave(isp,j)=ksave6(j)
	    do k=1,5
	      icx(isp,j,k)=icsu(j,k)
	      ireadx(isp,j,k)=ireadsu(j,k)
	      eex(isp,j,k)=enesu(j,k)
	    enddo
	  elseif(isp.eq.7) then
	    jma(isp,j)=jma7(j) 
	    ksave(isp,j)=ksave7(j)
	    do k=1,7
	      icx(isp,j,k)=icfe(j,k)
	      ireadx(isp,j,k)=ireadfe(j,k)
	      eex(isp,j,k)=enefe(j,k)
	    enddo
	  endif
	enddo
      enddo
C
      RETURN
      END
C ***********************************************************************
C ***********************************************************************
      SUBROUTINE CRREAD(nfile,jminn,jmax)
C
      implicit real*8(a-h,o-z)
      character*78 cdum
C
      common/ENEWS/SIGMA(7,89),EP(7),IC(7),IREAD(7)
C
      do j=jminn,jmax
	if(iread(j).eq.1) then
          read(nfile,*)EP(j)
	else
	  read(nfile,*)dummy1
	endif
	if(iread(j).eq.0) goto 101
  	do i=1,89
	  sigma(j,i)=0.0d0
	enddo
        do i=ic(j),89
	  read(nfile,9)cdum(1:78)
	  if(cdum(75:75).eq.' ') then
	    cdum(75:75)='+'
	    write(50,9)cdum(68:77)
	  elseif(cdum(76:76).eq.' ') then
	    cdum(76:76)='+'
	    write(50,9)cdum(69:78)  
	  endif
	  if(cdum(75:75).eq.'-') then
	    write(50,9)cdum(68:77)
	  elseif(cdum(76:76).eq.'-') then
	    write(50,9)cdum(69:78)  
	  endif
        enddo
	rewind(50)
	do i=ic(j),89
	  read(50,*)sigma(j,i)
	enddo
  101   continue
	rewind(50)
      enddo
    9 format(a)
C
      RETURN
      END
C ***********************************************************************
C ***********************************************************************
      SUBROUTINE CRIPOL(jminn,jmax,ik)
C
      implicit real*8(a-h,o-z)
C
      parameter (md=75)
C
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)
      COMMON/FRE/JMIN,JJ
      COMMON/SIK/SK(117,-13:75)
      common/ENEWS/SIGMA(7,89),EP(7),IC(7),IREAD(7)
      COMMON/ENEWS2/ETEMP(90),ETEMPLG(90),sgtemp(90),sgtemplg(90),
     &ee(7),e1log(-13:76)
      COMMON/ENEWS3/efix(89)
C
      do j=jminn,jmax
	ediff=ee(j)-ep(j)
	icj=ic(j)
	do i=icj+1,90
	  etemp(i)=efix(i-1)+ediff
	  etemplg(i)=dlog10(etemp(i))
	  sgtemp(i)=1.d-18*sigma(j,i-1)
	  sgtemplg(i)=dlog10(sgtemp(i))
        enddo
	etemp(icj)=ee(j)
	etemplg(icj)=dlog10(etemp(icj))
	sgtemplg(icj)=sgtemplg(icj+1)+((etemplg(icj)-etemplg(icj+1))/
     &  (etemplg(icj+2)-etemplg(icj+1)))*(sgtemplg(icj+2)-
     &  sgtemplg(icj+1))
	sgtemp(icj)=1.d1**sgtemplg(icj)
        do ij=jmin,jj
          if(e1(ij).ge.ee(j)) then
            do i=icj,89
              if(e1(ij).ge.etemp(i).and.e1(ij).lt.etemp(i+1)) then
                sg=sgtemplg(i)+(e1log(ij)-etemplg(i))*((sgtemplg(i+1)
     &		-sgtemplg(i))/(etemplg(i+1)-etemplg(i)))
    	        if(j.eq.1) sk(ik,ij)=sk(ik,ij)+1.d1**sg
  	        if(j.gt.1) si(ik,ij)=si(ik,ij)+1.d1**sg
                goto 100
     	      endif
	    enddo
  100       continue
	  endif
        enddo
      enddo
C
      RETURN
      END
C ***********************************************************************
C ***********************************************************************
      SUBROUTINE ENERGY                                                 
C
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      parameter (md=75)
C
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)
      COMMON/FRE/JMIN,JJ                                                
C
C     ***********************************************************       
C     *****                                                             
C     ENERGY INTERVALS IN EV                                            
C     *****                                                             
C     ***********************************************************       
      DO J=JMIN,1                                                  
        E(J)=3.4*10.**(DBLE(J+13)*.04013733)                              
        E1(J)=3.4*10.**((DBLE(J)+13.5)*.04013733)                         
      ENDDO
      DO J=2,6                                                      
        E(J)=13.598*10.**(DBLE(J-2)*0.0514461)                            
        E1(J)=13.598*10.**((DBLE(J)-1.5)*0.0514461)                       
      ENDDO
      DO J=7,13                                                     
        E(J)=24.587*10.**(DBLE(J-7)*0.0492887)                            
        E1(J)=24.587*10.**((DBLE(J)-6.5)*0.0492887)                       
      ENDDO
      DO J=14,21                                                    
        E(J)=54.416*10.**(DBLE(J-14)*0.0509579)                           
        E1(J)=54.416*10.**((DBLE(J)-13.5)*0.0509579)                      
      ENDDO
      DO J=22,27                                                    
        E(J)=139.12*10.**(DBLE(J-22)*0.0526138)                           
        E1(J)=139.12*10.**((DBLE(J)-21.5)*0.0526138)                      
      ENDDO
      E(28)=297.4
      E(29)=315.2
      E(30)=325.0
      E(31)=343.0
      E(32)=369.0
      E(33)=392.1
      E(34)=412.4
      E(35)=432.2
      E(36)=446.9
      E(37)=471.0
      E(38)=493.0
      E(39)=511.0
      E(40)=546.0
      E(41)=567.9
      E(42)=583.3
      E(43)=613.8
      E(44)=644.0
      E(45)=667.03
      E(46)=739.32
      E(47)=802.0
      E1(47)=835.3
      do j=28,46
        E1(j)=dsqrt(e(j)*e(j+1))
      enddo
      DO J=48,65                                                    
        E(J)=870.*10.**(DBLE(J-48)*0.0750)                                
        E1(J)=870.*10.**((DBLE(J)-47.5)*0.0750)                           
      ENDDO
      DO J=66,76                                                    
        E(J)=1.6387D4*10.**(DBLE(J-65)*0.15)                              
        E1(J)=1.6387D4*10.**((DBLE(J)-64.5)*0.15)                         
      ENDDO
C
      RETURN                                                            
      END                                                               
C ***********************************************************************
C ***********************************************************************
      SUBROUTINE HE1CROSS
C ***********************************************************************
C  High precision He I photoionization cross sections (mapped onto the
C  energy grid used in the program.) Samson et al. (J.Phys.B 27, 887,'94).
C  Samson et al. corrected by Pont & Shakeshaft (J.Phys.B 28, L571, '95).
C ***********************************************************************
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      parameter (md=75)
C
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),e1(-13:76),e(-13:76)
      common/SIK/SK(117,-13:75)
      COMMON/FRE/JMIN,JJ
C
      dimension eh(200),skhe(200),indhe(75)
C
      iomax=jj
      do io=1,1000
        read(63,*,err=999,end=999)eh(io),skhe(io)
	jhe=io
      enddo
  999 continue
C step in the energy space used in the program
      do io=1,iomax+1
	if(e(io).lt.eh(1)) then
	  indhe(io)=0
	  goto 1001
	else
	  do ip=1,jhe-1
	  if(eh(ip).le.e(io).and.eh(ip+1).gt.e(io)) then
	    indhe(io)=ip
	  endif
	  enddo
	endif
 1001 continue
      enddo
      do io=1,iomax
      sk(2,io)=0.0d0
	emin=e(io)
	emax=e(io+1)
	if(indhe(io).eq.0.and.indhe(io+1).eq.0) then
	  sk(2,io)=0.0d0
	  goto 1002
	elseif(indhe(io).eq.0.and.indhe(io+1).ne.0) then
	  imin=1
	  emin=eh(imin)
	  skmin=skhe(imin)
        else
	  imin=indhe(io)
	  qq=dlog10(skhe(imin+1)/skhe(imin))/
     &                dlog10(eh(imin+1)/eh(imin))
	  skmin=skhe(imin)*(emin/eh(imin))**qq
	endif
	imax=indhe(io+1)
	qq=dlog10(skhe(imax+1)/skhe(imax))/
     &                dlog10(eh(imax+1)/eh(imax))
	skmax=skhe(imax)*(emax/eh(imax))**qq
	if(imin.eq.imax) then
	   if(emin.eq.emax) goto 1002
	   qq=dlog10(skmax/skmin)/dlog10(emax/emin)+1.d0
	   sk(2,io)=skmin*(emin/(qq*(emax-emin)))*
     &	            ((emax/emin)**qq-1.d0)
	else
	   qq1=dlog10(skhe(imin+1)/skmin)/dlog10(eh(imin+1)/emin)+
     &         1.d0
	   sksum=skmin*(emin/qq1)*((eh(imin+1)/emin)**qq1-1.d0)
	   qq2=dlog10(skmax/skhe(imax))/dlog10(emax/eh(imax))+1.d0
	   sksum=sksum+skhe(imax)*(eh(imax)/qq2)*
     &	          ((emax/eh(imax))**qq2-1.d0)
	   if((imax-imin).ge.2) then
	     do iopp=imin+1,imax-1
	     qq3=dlog10(skhe(iopp+1)/skhe(iopp))/dlog10(eh(iopp+1)/
     &	          eh(iopp))+1.d0
	     sksum=sksum+skhe(iopp)*(eh(iopp+1)/qq3)*
     &	           ((eh(iopp+1)/eh(iopp))**qq3-1.d0)
	     enddo
	   endif
	   sk(2,io)=sksum/(emax-emin)
	endif
 1002 continue
      sk(2,io)=1.d-18*sk(2,io)
      enddo
C
      return
      end
C ***********************************************************************
C************************************************************
      SUBROUTINE FLUX2J(ishiwo,tdmin,tday,rpafix,burmul)
C************************************************************
C Simulates asymmetry effects in the ionizing radiation.
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)
C
      parameter (md=75)
C
      common/anglemu/xnjmu(100),njmu
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)
      COMMON/FRE/JMIN,JJ
      COMMON/LUMI/FSURF(-13:75)
C
      DIMENSION fhelp(-13:75),fold(-13:75)
C
      day=8.64d4
      qtc=1.d0
      cvel=2.998d10
      vej=1.5d9
      asym=1.5d0
      asym1=asym**2-1.d0
          do j=jmin,jj
          fhelp(j)=0.0d0
          fold(j)=0.0d0
          fsurf(j)=0.0d0
          enddo
      if(ishiwo.eq.0) then
        call nomshi(tday,rpa15,teb)
	r0fix=1.565d12
      elseif(ishiwo.eq.1) then
        call wos10l(tday,rpa15,teb)
	r0fix=3.d12
      elseif(ishiwo.eq.3) then
        call ensbur(tday,rpa15,teb,qtc)
	r0fix=3.46d12
      elseif(ishiwo.eq.4) then
        call ensbur2(tday,rpa15,teb,qtc)
	r0fix=2.44d12
        continue
      endif
      vejr0=vej/r0fix
C
      dsec=day*(tday-tdmin)
      rpafix=rpa15
      if(dsec.le.0.0d0) then
	xmum=0.0d0
      else
	xmum=dsqrt(((1.d0+vejr0*dsec)**2-1.d0)/asym1)
	if(xmum.ge.1.d0) xmum=1.d0
      endif
C
          do nnjmu=1,njmu
          if(nnjmu.ne.1) then
                do j=jmin,jj
                fold(j)=fhelp(j)
                enddo
            tdaymu=tday-(dsqrt(1.d0+xnjmu(nnjmu)**2*asym1)-1.d0)/
     &	    (vejr0*day)
	  else
	    tdaymu=tday
          endif
          if(tdaymu.ge.tdmin) then
            if(ishiwo.eq.0) call nomshi(tdaymu,rpa15,teb)
            if(ishiwo.eq.1) call wos10l(tdaymu,rpa15,teb)
            if(ishiwo.eq.3) call ensbur(tdaymu,rpa15,teb,qtc)
            if(ishiwo.eq.4) call ensbur2(tdaymu,rpa15,teb,qtc)
            cons=qtc*(rpa15/rpafix)**2*xnjmu(nnjmu)
                do j=jmin,jj
                aaex=(1.16054d4*e(j))/teb
                if(aaex.gt.1.d2) then
                  flff=1.d-50
                else
                  flff=(2.5205d10*e(j)**3)/(dexp(aaex)-1.d0)
                endif
                fhelp(j)=cons*flff
                if(nnjmu.ne.1) then
                  fsurf(j)=fsurf(j)+0.5d0*(fhelp(j)+fold(j))*
     &            (xnjmu(nnjmu)-xnjmu(nnjmu-1))
                endif
                enddo
          else
            if(ishiwo.eq.0) call nomshi(tdmin,rpa15,teb)
            if(ishiwo.eq.1) call wos10l(tdmin,rpa15,teb)
            if(ishiwo.eq.3) call ensbur(tdmin,rpa15,teb,qtc)
            if(ishiwo.eq.4) call ensbur2(tdmin,rpa15,teb,qtc)
            cons=qtc*(rpa15/rpafix)**2*xmum
                do j=jmin,jj
                aaex=(1.16054d4*e(j))/teb
                if(aaex.gt.1.d2) then
                  flff=1.d-50
                else
                  flff=(2.5205d10*e(j)**3)/(dexp(aaex)-1.d0)
                endif
                fhelp(j)=cons*flff
                fsurf(j)=fsurf(j)+0.5d0*(fhelp(j)+fold(j))*
     &          (xmum-xnjmu(nnjmu-1))
                enddo
            goto 1001
          endif
      enddo
 1001 continue
C
        do jjj=jmin,jj
	fsurf(jjj)=burmul*fsurf(jjj)
	enddo
C
      RETURN
      END
C************************************************************           
C************************************************************
      SUBROUTINE FLUX1J(ishiwo,tdmin,tday,rpafix,burmul)
C************************************************************
C Includes light-travel time effects in the ionizing radiation.
C (The original FLUX1J)
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)
C
      parameter (md=75)
c
      common/anglemu/xnjmu(100),njmu
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)
      COMMON/FRE/JMIN,JJ
      COMMON/LUMI/FSURF(-13:75)
C
      DIMENSION fhelp(-13:75),fold(-13:75)
C
      day=8.64d4
      fourpi=1.256637d1
      totx=0.0d0
C     fracx=0.243d0
      fracx=0.0d0
      tevx=2.5d2
      dtx=1.728d3
      qtc=1.d0
          do j=jmin,jj
          fhelp(j)=0.0d0
          fold(j)=0.0d0
          fsurf(j)=0.0d0
          enddo
      if(ishiwo.eq.0) then
        call nomshi(tday,rpa15,teb)
	totx=4.077d46
      elseif(ishiwo.eq.1) then
        call wos10l(tday,rpa15,teb)
	totx=2.562d46
      elseif(ishiwo.eq.3) then
        call ensbur(tday,rpa15,teb,qtc)
	totx=7.139d46
      elseif(ishiwo.eq.4) then
        call ensbur2(tday,rpa15,teb,qtc)
	totx=1.515d47
      elseif(ishiwo.eq.5) then
        call ensbur3(tday,rpa15,teb,qtc)
      endif
C
      chx=(1.d-30*totx*fracx)/(fourpi**2*tevx*dtx)
      dsec=day*(tday-tdmin)
      rpafix=rpa15
      xmum=1.d0-(2.998d-5*dsec)/rpafix
C
          do nnjmu=njmu,1,-1
          if(nnjmu.ne.njmu) then
                do j=jmin,jj
                fold(j)=fhelp(j)
                enddo
          endif
          tdaymu=tday-((1.d0-xnjmu(nnjmu))*rpafix)/(2.998d-5*day)
	  ddayx=tdaymu-tdmin
          if(tdaymu.ge.tdmin) then
            if(ishiwo.eq.0) call nomshi(tdaymu,rpa15,teb)
            if(ishiwo.eq.1) call wos10l(tdaymu,rpa15,teb)
            if(ishiwo.eq.3) call ensbur(tdaymu,rpa15,teb,qtc)
            if(ishiwo.eq.4) call ensbur2(tdaymu,rpa15,teb,qtc)
            if(ishiwo.eq.5) call ensbur3(tdaymu,rpa15,teb,qtc)
            cons=qtc*(rpa15/rpafix)**2*xnjmu(nnjmu)
                do j=jmin,jj
                aaex=(1.16054d4*e1(j))/teb
                if(aaex.gt.1.d2) then
                  flff=1.d-50
                else
                  flff=(2.5205d10*e1(j)**3)/(dexp(aaex)-1.d0)
                endif
                fhelp(j)=cons*flff
		if(ddayx.le.0.03d0.and.ddayx.ge.0.01d0) then
	  	  chxlum=chx/rpa15**2
		else
		  chxlum=0.0d0
		endif
		fhelp(j)=fhelp(j)+chxlum*expfn(e1(j)/tevx)
                if(nnjmu.ne.njmu) then
                  fsurf(j)=fsurf(j)+0.5d0*(fhelp(j)+fold(j))*
     &            (xnjmu(nnjmu+1)-xnjmu(nnjmu))
                endif
                enddo
          else
            if(ishiwo.eq.0) call nomshi(tdmin,rpa15,teb)
            if(ishiwo.eq.1) call wos10l(tdmin,rpa15,teb)
            if(ishiwo.eq.3) call ensbur(tdmin,rpa15,teb,qtc)
            if(ishiwo.eq.4) call ensbur2(tdmin,rpa15,teb,qtc)
            if(ishiwo.eq.5) call ensbur3(tdmin,rpa15,teb,qtc)
            cons=qtc*(rpa15/rpafix)**2*xmum
                do j=jmin,jj
                aaex=(1.16054d4*e1(j))/teb
                if(aaex.gt.1.d2) then
                  flff=1.d-50
                else
                  flff=(2.5205d10*e1(j)**3)/(dexp(aaex)-1.d0)
                endif
                fhelp(j)=cons*flff
		if(ddayx.le.0.03d0.and.ddayx.ge.0.01d0) then
	  	  chxlum=chx/rpa15**2
		else
		  chxlum=0.0d0
		endif
		fhelp(j)=fhelp(j)+chxlum*expfn(e1(j)/tevx)
                fsurf(j)=fsurf(j)+0.5d0*(fhelp(j)+fold(j))*
     &          (xnjmu(nnjmu+1)-xmum)
                enddo
            goto 1001
          endif
      enddo
 1001 continue
C
        do jjj=jmin,jj
	fsurf(jjj)=burmul*fsurf(jjj)
	enddo
C
      RETURN
      END
C************************************************************           
C************************************************************
      SUBROUTINE WOS10L(T,R,TE)
C************************************************************
C
C -- A linear fit to Woosley's 10L model of SN 1987a
C
C -- T is time in days after core collapse (Tmin=0.1 days)
C -- R is photospheric radius in 1E15 cm.
C -- TE is the photospheric effective temperature in K.
C
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)
C
      DIMENSION TIME(17),TEMP(17),RADIUS(17),templg(17)
C
      data time/0.100d0,0.1015d0,0.104d0,0.108d0,0.112d0,0.120d0,
     &0.132d0,0.152d0,0.172d0,0.192d0,0.232d0,0.3d0,0.4d0,0.5d0,
     &1.5d0,3.d0,5.d0/
      data radius/3.d-3,4.014d-3,4.556d-3,4.882d-3,5.478d-3,6.896d-3,
     &9.302d-3,1.255d-2,1.526d-2,1.773d-2,2.447d-2,3.045d-2,4.25d-2,
     &5.42d-2,0.122d0,0.24d0,0.35d0/
      data temp/1.862d5,1.995d5,1.738d5,1.413d5,1.202d5,9.550d4,
     &7.586d4,6.026d4,5.248d4,4.677d4,3.802d4,3.236d4,2.570d4,2.188d4,
     &1.4d4,8.d3,5.d3/
C
      IF(T.GT.TIME(17)) GOTO 200
      DO 100 I=1,16
      IF(T.GE.TIME(I).AND.T.LT.TIME(I+1)) GOTO 101
      GOTO 100
  101 QQ=(T-TIME(I))/(TIME(I+1)-TIME(I))
      R=RADIUS(I)+QQ*(RADIUS(I+1)-RADIUS(I))
      templg(i)=dlog10(temp(i))
      templg(i+1)=dlog10(temp(i+1))
      TELG=TEMPLG(I)+QQ*(TEMPLG(I+1)-TEMPLG(I))
      TE=1.d1**TELG
      GOTO 9999
  100 CONTINUE
  200 CONTINUE
      TE=TEMP(17)
      R=RADIUS(17)
 9999 CONTINUE
C
      RETURN
      END
C************************************************************
C************************************************************
      SUBROUTINE FIXFLX(T,R,TE,phion,tev)
C************************************************************
C -- Keep the flux fixed during a burst of DT seconds
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)
C
      TIN=0.08D0
      BFRAC1=1.0D0
      DT=5.D2
      IF(T.GT.(TIN+(DT*BFRAC1)/8.64D4)) TEV=0.5D0
      TE=1.1605D4*TEV
      R=1.D-3*DSQRT(1.D57/(PHION*DT))
C
      RETURN
      END
C************************************************************
C************************************************************
      SUBROUTINE BARBLI1(TDAYS,RPA15,qtc,ishiwo)
C************************************************************
C
C -- Read Bartunov and Blinnikov's '*.ph'-files.
C --   ishiwo = 6 (nommxf.ph)       ishiwo = 7 (ut87af.ph)
C --   ishiwo = 8 (nmxfso17.ph)
C
C************************************************************
      implicit real*8(a-h,o-z)
C
      common/bartbl/bbtime(1000),bbfreq(100),bbflum(100,1000),
     &iread
C
      h=6.626d-27
      e=1.6022d-12
      he=h/e
      qtc=1.d0
      rpa15=1.d-2
      if(ishiwo.eq.6) kread=66
      if(ishiwo.eq.7) kread=67
      if(ishiwo.eq.8) kread=68
C
      read(kread,*)(bbfreq(ila),ila=1,100)
      iread=0
      do ir=1,1500
	read(kread,*,err=999,end=999)bbtime(ir),iidum,xdum,
     &	                         (bbflum(ila,ir),ila=1,100)
	iread=iread+1
      enddo
  999 continue
      bbtime1=bbtime(1)
      do ila=1,100
        bbfreq(ila)=he*1.d1**(bbfreq(ila))
      enddo
      do ir=1,iread
	do ila=1,100
	  bbflum(ila,ir)=(1.d1**bbflum(ila,ir))/he
        enddo
	bbtime(ir)=bbtime(ir)-bbtime1
      enddo
C
      return
      end
C************************************************************
C************************************************************
      SUBROUTINE BARBLI(TDAYS,RPA15,qtc)
C************************************************************
C
C -- Interpolate in Bartunov and Blinnikov's grids
C
C************************************************************
      implicit real*8(a-h,o-z)
C
      parameter (md=75)
C
      COMMON/FRE/JMIN,JJ
      COMMON/LUMI/FSURF(-13:75)
      COMMON/INT/FL(md,-13:75),SI(117,-13:75),E1(-13:76),E(-13:76)
      common/bartbl/bbtime(1000),bbfreq(100),bbflum(100,1000),
     &iread
C
      pi=3.14159d0
      qtc=1.d0
      rpa15=1.d-2
      const=1.d-30/(4.d0*pi*rpa15)**2
C
      if(tdays.ge.bbtime(iread)) then
        irsave=iread-1 
        qqir=1.d0
      else
      do ir=1,iread-1
        if(tdays.ge.bbtime(ir).and.tdays.lt.bbtime(ir+1)) then
          irsave=ir
          qqir=(tdays-bbtime(ir))/(bbtime(ir+1)-bbtime(ir))  
        endif
      enddo
      endif
C
      do j=jmin,jj
        do ila=1,99
        if(e1(j).gt.bbfreq(ila).and.e1(j).le.bbfreq(ila+1)) then
          ilsave=ila
          qqila=(e1(j)-bbfreq(ila))/(bbfreq(ila+1)-bbfreq(ila))
        endif
        enddo
        fmin=bbflum(ilsave,irsave)*(1.d0-qqila)+
     &       bbflum(ilsave+1,irsave)*qqila
        fmax=bbflum(ilsave,irsave+1)*(1.d0-qqila)+
     &       bbflum(ilsave+1,irsave+1)*qqila
        fsurf(j)=(fmin*(1.d0-qqir)+fmax*qqir)*const
      enddo
C
      return
      end
C************************************************************
C************************************************************
      SUBROUTINE NOMSHI(T,R,TE)
C************************************************************
C
C -- A linear fit to Nomoto and Shigeyama's 11E1Y6 model of 87a.
C
C -- T is time in days after core collapse (Tmin=0.08 days)
C -- R is photospheric radius in 1E15 cm.
C -- TE is the photospheric effective temperature in K.
C
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)
C
      DIMENSION TIME(21),TEMP(21),RADIUS(21),templg(21)
C
      data time/0.08d0,0.085d0,0.09d0,0.095d0,0.1d0,0.105d0,0.11d0,
     &0.12d0,0.13d0,0.151d0,0.172d0,0.219d0,0.252d0,0.302d0,0.403d0,
     &0.5d0,0.705d0,1.0d0,1.5d0,3.d0,5.d0/
      data radius/1.565d-3,2.156d-3,3.23d-3,4.155d-3,4.965d-3,5.703
     &d-3,6.38d-3,7.635d-3,8.864d-3,1.117d-2,1.336d-2,1.807d-2,
     &2.108d-2,2.549d-2,3.405d-2,4.186d-2,5.617d-2,7.486d-2,1.22d-1,
     &2.40d-1,3.5d-1/
      data temp/5.47d5,2.08d5,1.50d5,1.24d5,1.09d5,9.81d4,9.03d4,
     &7.94d4,7.16d4,6.11d4,5.42d4,4.44d4,4.00d4,3.52d4,2.90d4,
     &2.52d4,2.06d4,1.70d4,1.4d4,8.d3,5.d3/
C     DATA RADIUS/1.86D-3,2.13D-3,4.31D-3,9.42D-3,1.52D-2,3.37D-2,5.7
C    &1D-2,1.22D-1,2.40D-1,3.5D-1/
C     DATA TEMP/6.0D5,4.0D5,1.6D5,9.0D4,5.3D4,3.2D4,2.3D4,1.4D4,
C    &8.D3,5.D3/
C
      IF(T.GT.TIME(21)) GOTO 200
      DO 100 I=1,20
      IF(T.GE.TIME(I).AND.T.LT.TIME(I+1)) GOTO 101
      GOTO 100
  101 QQ=(T-TIME(I))/(TIME(I+1)-TIME(I))
      R=RADIUS(I)+QQ*(RADIUS(I+1)-RADIUS(I))
      templg(i)=dlog10(temp(i))
      templg(i+1)=dlog10(temp(i+1))
      TELG=TEMPLG(I)+QQ*(TEMPLG(I+1)-TEMPLG(I))
      TE=1.d1**TELG
      GOTO 9999
  100 CONTINUE
  200 CONTINUE
      TE=TEMP(21)
      R=RADIUS(21)
 9999 CONTINUE
      RETURN
      END
C************************************************************           
C************************************************************
      SUBROUTINE ENSBUR(T,R,TE,qtcte)
C************************************************************
C
C -- A linear fit to Ensman & Burrows 'full1'-model. Extra-
C    polated after 0.5 days, and joined to observations after
C    1.5 days.
C
C -- T is time in days after core collapse (Tmin=0.08 days)
C -- R is photospheric radius in 1E15 cm.
C -- TE is the photospheric effective temperature in K.
C    qtcte is correction due to color temperature instead of
C    TE
C
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)
C
      DIMENSION TIME(33),TEMP(33),RADIUS(33),templg(33),tcte(33)
C
      data time/0.08d0,0.080116d0,0.080232d0,0.080347d0,0.080463d0,
     &0.080579d0,0.08081d0,0.081042d0,0.081505d0,0.082199d0,
     &0.08278d0,0.08359d0,0.084514d0,0.085d0,0.09d0,0.095d0,
     &0.1d0,0.105d0,0.11d0,0.12d0,0.13d0,0.15d0,0.17d0,0.20d0,
     &0.25d0,0.30d0,0.40d0,0.5d0,0.70d0,1.0d0,1.5d0,3.d0,5.d0/
      data radius/3.46d-3,3.24d-3,3.19d-3,2.99d-3,2.99d-3,3.28d-3,
     &3.37d-3,3.44d-3,3.43d-3,3.55d-3,3.42d-3,3.68d-3,4.29d-3,
     &4.75d-3,5.46d-3,6.38d-3,7.83d-3,8.68d-3,1.031d-2,1.227d-2,
     &1.525d-2,1.945d-2,2.414d-2,3.108d-2,4.351d-2,5.118d-2,
     &6.962d-2,9.111d-2,1.020d-1,1.113d-1,1.138d-1,3.108d-1,7.34d-1/
      data temp/2.20d5,3.40d5,4.70d5,5.45d5,5.45d5,4.90d5,3.85d5,
     &3.30d5,2.70d5,2.30d5,2.15d5,1.90d5,1.70d5,1.58d5,1.29d5,
     &1.07d5,9.33d4,8.51d4,7.59d4,6.76d4,5.89d4,5.01d4,4.37d4,
     &3.72d4,3.02d4,2.69d4,2.19d4,1.86d4,1.66d4,1.50d4,1.4d4,8.d3,
     &5.d3/
      data tcte/5.09d0,3.21d0,2.18d0,1.78d0,1.63d0,1.63d0,1.71d0,
     &1.76d0,1.76d0,1.63d0,1.42d0,1.056d0,1.d0,1.127d0,1.286d0,
     &1.402d0,1.514d0,1.551d0,1.659d0,1.66d0,1.778d0,3*1.738d0,
     &1.698d0,1.66d0,2*1.698d0,1.56d0,1.35d0,3*1.d0/
C     DATA RADIUS/1.86D-3,2.13D-3,4.31D-3,9.42D-3,1.52D-2,3.37D-2,5.7
C    &1D-2,1.22D-1,2.40D-1,3.5D-1/
C     DATA TEMP/6.0D5,4.0D5,1.6D5,9.0D4,5.3D4,3.2D4,2.3D4,1.4D4,
C    &8.D3,5.D3/
C
      IF(T.GT.TIME(33)) GOTO 200
      DO 100 I=1,32
      IF(T.GE.TIME(I).AND.T.LT.TIME(I+1)) GOTO 101
      GOTO 100
  101 QQ=(T-TIME(I))/(TIME(I+1)-TIME(I))
      R=RADIUS(I)+QQ*(RADIUS(I+1)-RADIUS(I))
      tct=tcte(i)+QQ*(tcte(i+1)-tcte(i))
      templg(i)=dlog10(temp(i)*tcte(i))
      templg(i+1)=dlog10(temp(i+1)*tcte(i+1))
      TELG=TEMPLG(I)+QQ*(TEMPLG(I+1)-TEMPLG(I))
      TE=1.d1**TELG
      GOTO 9999
  100 CONTINUE
  200 CONTINUE
      tct=tcte(33)
      TE=TEMP(33)*tct
      R=RADIUS(33)
 9999 CONTINUE
      qtcte=1.d0/(tct**4)
      RETURN
      END
C************************************************************           
C************************************************************
      SUBROUTINE ENSBUR2(T,R,TE,qtcte)
C************************************************************
C
C -- A linear fit to Ensman & Burrows 'full2'-model. Extra-
C    polated after 0.5 days, and joined to observations after
C    1.5 days.
C
C -- T is time in days after core collapse (Tmin=0.08 days)
C -- R is photospheric radius in 1E15 cm.
C -- TE is the photospheric effective temperature in K.
C    qtcte is correction due to color temperature instead of
C    TE
C
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)
C
      DIMENSION TIME(33),TEMP(33),RADIUS(33),templg(33),tcte(33)
C
      data time/0.08d0,0.080116d0,0.080231d0,0.080289d0,0.080347d0,
     &0.080463d0,0.080694d0,0.080926d0,0.081157d0,0.081620d0,
     &0.082083d0,0.082315d0,0.0830d0,0.085d0,0.088d0,0.092d0,
     &0.096d0,0.100d0,0.105d0,0.11d0,0.12d0,0.14d0,0.16d0,0.20d0,
     &0.25d0,0.30d0,0.40d0,0.5d0,0.70d0,1.0d0,1.5d0,3.d0,5.d0/
      data radius/2.44d-3,3.29d-3,3.03d-3,3.14d-3,3.34d-3,3.37d-3,
     &3.29d-3,3.19d-3,3.39d-3,4.29d-3,3.91d-3,4.01d-3,4.11d-3,
     &4.99d-3,6.69d-3,7.86d-3,10.56d-3,11.32d-3,1.284d-2,1.475d-2,
     &1.920d-2,2.622d-2,3.185d-2,4.651d-2,6.586d-2,7.483d-2,
     &11.437d-2,13.633d-2,1.5609d-1,1.627d-1,1.664d-1,4.544d-1,10.73d-1/
      data temp/1.70d5,4.20d5,6.70d5,7.30d5,6.80d5,5.60d5,4.25d5,
     &3.65d5,3.25d5,2.60d5,2.50d5,2.40d5,2.20d5,1.78d5,1.41d5,
     &1.20d5,10.0d4,9.33d4,8.51d4,7.76d4,6.61d4,5.37d4,4.68d4,
     &3.72d4,3.02d4,2.69d4,2.09d4,1.86d4,1.66d4,1.50d4,1.4d4,8.d3,
     &5.d3/
      data tcte/8.18d0,3.14d0,1.78d0,1.575d0,1.574d0,1.714d0,1.824d0,
     &1.863d0,1.908d0,2.00d0,1.32d0,1.000d0,1.182d0,1.444d0,1.660d0,
     &1.783d0,1.910d0,1.822d0,1.857d0,1.907d0,1.906d0,1.955d0,
     &1.778d0,1.817d0,1.904d0,1.74d0,1.904d0,1.806d0,1.56d0,1.35d0,
     &3*1.d0/
C     DATA RADIUS/1.86D-3,2.13D-3,4.31D-3,9.42D-3,1.52D-2,3.37D-2,5.7
C    &1D-2,1.22D-1,2.40D-1,3.5D-1/
C     DATA TEMP/6.0D5,4.0D5,1.6D5,9.0D4,5.3D4,3.2D4,2.3D4,1.4D4,
C    &8.D3,5.D3/
C
      IF(T.GT.TIME(33)) GOTO 200
      DO 100 I=1,32
      IF(T.GE.TIME(I).AND.T.LT.TIME(I+1)) GOTO 101
      GOTO 100
  101 QQ=(T-TIME(I))/(TIME(I+1)-TIME(I))
      R=RADIUS(I)+QQ*(RADIUS(I+1)-RADIUS(I))
      tct=tcte(i)+QQ*(tcte(i+1)-tcte(i))
      templg(i)=dlog10(temp(i)*tcte(i))
      templg(i+1)=dlog10(temp(i+1)*tcte(i+1))
      TELG=TEMPLG(I)+QQ*(TEMPLG(I+1)-TEMPLG(I))
      TE=1.d1**TELG
      GOTO 9999
  100 CONTINUE
  200 CONTINUE
      tct=tcte(33)
      TE=TEMP(33)*tct
      R=RADIUS(33)
 9999 CONTINUE
      qtcte=1.d0/(tct**4)
C --- change the spectrum (keeping the integrated luminosity constant).
C      if(t.lt.1.d0) then
C       corrt=0.822d0*(t/0.08d0)**(.0776d0)
C       te=corrt*te
C       r=r/corrt**2
C      endif
C
      RETURN
      END
C************************************************************           
C************************************************************
      SUBROUTINE ENSBUR3(T,R,TE,qtcte)
C************************************************************
C
C -- A linear fit to an extrapolated model to Ensman & Burrows
C    'full'-models. The below model has higher color temp.
C    than 500full1 and 500full2, but is in the same way as these 
C    extrapolateded after 0.5 days, and joined to observations
C    after 1.5 days. Maximum color temperture is 2E6 K. The tem-
C    poral behavior of the effective temperature and luminosity
C    is the same as in 500full2.
C
C -- T is time in days after core collapse (Tmin=0.08 days)
C -- R is photospheric radius in 1E15 cm.
C -- TE is the photospheric effective temperature in K.
C    qtcte is correction due to color temperature instead of
C    TE
C
C************************************************************
      IMPLICIT REAL*8(A-H,O-Z)
C
      DIMENSION TIME(33),TEMP(33),RADIUS(33),templg(33),tcte(33)
C
      data time/0.08d0,0.080116d0,0.080231d0,0.080289d0,0.080347d0,
     &0.080463d0,0.080694d0,0.080926d0,0.081157d0,0.081620d0,
     &0.082083d0,0.082315d0,0.0830d0,0.085d0,0.088d0,0.092d0,
     &0.096d0,0.100d0,0.105d0,0.11d0,0.12d0,0.14d0,0.16d0,0.20d0,
     &0.25d0,0.30d0,0.40d0,0.5d0,0.70d0,1.0d0,1.5d0,3.d0,5.d0/
      data radius/2.44d-3,3.29d-3,3.03d-3,3.14d-3,3.34d-3,3.37d-3,
     &3.29d-3,3.19d-3,3.39d-3,4.29d-3,3.91d-3,4.01d-3,4.11d-3,
     &4.99d-3,6.69d-3,7.86d-3,10.56d-3,11.32d-3,1.284d-2,1.475d-2,
     &1.920d-2,2.622d-2,3.185d-2,4.651d-2,6.586d-2,7.483d-2,
     &11.437d-2,13.633d-2,1.5609d-1,1.627d-1,1.664d-1,4.544d-1,10.73d-1/
      data temp/1.70d5,4.20d5,6.70d5,7.30d5,6.80d5,5.60d5,4.25d5,
     &3.65d5,3.25d5,2.60d5,2.50d5,2.40d5,2.20d5,1.78d5,1.41d5,
     &1.20d5,10.0d4,9.33d4,8.51d4,7.76d4,6.61d4,5.37d4,4.68d4,
     &3.72d4,3.02d4,2.69d4,2.09d4,1.86d4,1.66d4,1.50d4,1.4d4,8.d3,
     &5.d3/
C     data tcte/11.765d0,4.3584d0,2.3424d0,2.0452d0,1.9057d0,2.0013d0,
C    &2.0875d0,2.2430d0,2.3757d0,2.5462d0,1.7820d0,1.3500d0,1.5957d0,
C    &2.4423d0,2.6719d0,2.8071d0,2.8727d0,2.5151d0,2.5456d0,2.5469d0,
C    &2.3748d0,2.3528d0,1.8688d0,1.9948d0,2.3676d0,1.9200d0,2.1847d0,
C    &2.0490d0,1.5600d0,1.3500d0,3*1.d0/
      data tcte/11.765d0,4.358d0,26*3.d0,1.56d0,1.35d0,3*1.d0/
C     DATA RADIUS/1.86D-3,2.13D-3,4.31D-3,9.42D-3,1.52D-2,3.37D-2,5.7
C    &1D-2,1.22D-1,2.40D-1,3.5D-1/
C     DATA TEMP/6.0D5,4.0D5,1.6D5,9.0D4,5.3D4,3.2D4,2.3D4,1.4D4,
C    &8.D3,5.D3/
C
      IF(T.GT.TIME(33)) GOTO 200
      DO 100 I=1,32
      IF(T.GE.TIME(I).AND.T.LT.TIME(I+1)) GOTO 101
      GOTO 100
  101 QQ=(T-TIME(I))/(TIME(I+1)-TIME(I))
      R=RADIUS(I)+QQ*(RADIUS(I+1)-RADIUS(I))
      tct=tcte(i)+QQ*(tcte(i+1)-tcte(i))
      templg(i)=dlog10(temp(i)*tcte(i))
      templg(i+1)=dlog10(temp(i+1)*tcte(i+1))
      TELG=TEMPLG(I)+QQ*(TEMPLG(I+1)-TEMPLG(I))
      TE=1.d1**TELG
      GOTO 9999
  100 CONTINUE
  200 CONTINUE
      tct=tcte(33)
      TE=TEMP(33)*tct
      R=RADIUS(33)
 9999 CONTINUE
      qtcte=1.d0/(tct**4)
      RETURN
      END
C************************************************************ 
C************************************************************
      subroutine shock(tday,clum,estart,gam2)
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      real*8 n,m,m1,m2,np,ne1,ne2
C
      dmdt=1.82d-2
      n=9.58d0
      m=(n-3.d0)/(n-2.d0)
      tsec=8.64d4*(tday-0.08d0)
      rp0=3.46d12
      rc=1.87d10*tsec**m
      r2=0.98d0*rc
      vc=(m*rc)/tsec
      rs=1.24d0*rc
      vs=1.24d0*vc
      ts=1.36d-9*vs**2
      te=3.d9
C thin shell approx.
c      np=(3.d37*dmdt)/rs**2
      m1=6.30d14*dmdt*(rs-rp0)
      m2=0.5d0*(n-4.d0)*m1
      ne1=(1.43d23*m1)/(rs**3-rc**3)
      ne2=(1.43d23*m2)/(rc**3-r2**3)
      tau1=6.65d-25*ne1*(rs-rc)
      tau2=6.65d-25*ne2*(rc-r2)
      call ENSBUR(TDAY,RP,TCOL,qtcte)
      rp=1.d15*rp
      teff=tcol*qtcte**.25d0
      tc=ts/(n-3)**2
      tc=0.7d0*tc
      w=0.5d0*(1.d0-dsqrt(1.d0-(rp/rc)**2))
      const=4.812d-13*(1.d0-w)*rp**2*teff**4
      shock1=3.2d41*dmdt*(vs/1.d9)**3
      shock2=(0.5d0*shock1*(n-4.d0))/(n-3.d0)**2
 1002 continue
      alpha1=te/5.93d9
      call gamma(gam1,tau1,alpha1)
      comp1=const*tau1*te*(1.d0+4.4d-10*te)
      if(shock1.lt.comp1) te=0.9999d0*te
      if(shock1.lt.comp1) goto 1002
 1001 continue
      alpha2=tc/5.93d9
      call gamma(gam2,tau2,alpha2)
      comp2=const*tau2*tc*(1.d0+4.4d-10*tc)
      if(shock2.lt.comp2) tc=0.9999d0*tc
      if(shock2.lt.comp2) goto 1001
      epeak=2.7d0*(tcol/1.16d4)
      estart=epeak*(1.d0+(te/1.48d9)*(1.d0+4.4d-10*te))
      clum=(comp1*(gam1-1.d0))/estart
      if(tday.gt.1.d0) then
      clum=1.d-50
       estart=1.d0
       gam2=1.d0
      endif
c     estart=epeak*(1.d0+(tc/1.48d9)*(1.d0+4.4d-10*tc))
c     clum=(comp2*(gam2-1.d0))/estart
c     if(tday.gt.1.d0) then
c      clum=1.d-50
c      estart=1.d0
c      gam2=1.d0
c     endif
C
      return
      end
C************************************************************
C************************************************************
      subroutine gamma(ga,t,a)
C     
      implicit real*8(a-h,o-z)
C  
      d=0.5772d0
      g=1.5d0-d-dlog(t)
      ga=dsqrt(2.25d0-(dlog(0.5d0*g*t))/a)-1.5d0
C  
      return
      end
C************************************************************
C************************************************************ 
      SUBROUTINE FEII(TE,EDENS,COLTOTI,COLTOT,FECOOL)
C************************************************************           
C  Fe II is treated as a 16-level atom with escape probs for the
C  fine structure lines. Corrections to the cooling are made for
C  T > 3,000 K, using results from 116-level calcs.. Collision
C  strengths between the lowest 9 levels are from Berrington et al.
C  (J.Phys.B 21, 339 (1988)), and the rest are from C. Fransson 
C  (priv.comm. 1991, van Regemorter estimates ?). 
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
C                                                                       
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      COMMON/FEIIC/OME(16,16),AIJ(16,16),DLAIR(16,16) 
      common/FEIIC2/adamp(16,16),asum(16)
C
      DIMENSION ENE(16),G(16),CIJ(16,16),FBFII(16,16),
     &tfix(5),afix(5),bfix(5),p(16,16),xh(16)
C
      DATA G/10.D0,8.D0,6.D0,4.D0,2.D0,10.D0,8.D0,6.D0,4.D0,8.D0,
     &6.D0,4.D0,2.D0,6.D0,4.D0,2.D0/
      DATA ENE/0.0D0,0.047705D0,0.082777D0,0.10695D0,0.12114D0,
     &0.23217D0,0.30129D0,0.35186D0,0.38652D0,0.98632D0,1.04046D0,
     &1.07624D0,1.09686D0,1.67062D0,1.69526D0,1.72398D0/
      data tfix/3.4771,3.699,4.0,4.301,4.6021/
      data afix/-22.7525,-21.1822,-19.8686,-19.1382,-18.7849/
      data bfix/.0818,.0551,.0265,.0081,-.0039/
C
      TEV=TE/1.1605D4
      CONS=(8.63D-6*EDENS)/DSQRT(TE)
      amass=55.85d0
C
      tlog=dlog10(te)
       do ii=1,16
        do jj=1,16
        cij(ii,jj)=0.0d0
        p(ii,jj)=0.0d0
        enddo
       enddo
       do 111 ii=1,15
        do 112 jj=ii+1,16
        exiijj=expfn((ene(jj)-ene(ii))/tev)
        cij(ii,jj)=(cons*ome(ii,jj)*exiijj)/g(ii)
        cij(jj,ii)=(g(ii)*cij(ii,jj))/(exiijj*g(jj))
  112   continue
  111  continue
C
      m=0
 9988 continue
      m=m+1
       do 101 ii=1,16
        do 102 jj=1,16
        a(ii,jj)=0.0d0
  102   continue
       if(m.ne.1) xh(ii)=x(ii)
       x(ii)=0.0d0
  101  continue
       do ii=2,16
        do jj=1,ii-1
        if(aij(ii,jj).ne.0.0d0) then
         if(m.eq.1.and.jj.ne.1) tcorr=0.0d0
         if(m.eq.1.and.jj.eq.1) tcorr=1.d0
         if(m.ne.1) tcorr=xh(jj)*(1.d0-(xh(ii)*g(jj))/(xh(jj)*g(ii)))
         call escacf((ene(ii)-ene(jj)),aij(ii,jj),adamp(ii,jj),g(jj),
     &   g(ii),te,amass,tcorr,coltoti,coltot,p(ii,jj))
        endif
        enddo
       enddo
       do 113 ii=1,16
        do 114 jj=1,16
        if(ii.eq.16) then
         a(ii,jj)=1.d0
        else
         if(ii.eq.jj) goto 114
         a(ii,ii)=a(ii,ii)-p(ii,jj)*aij(ii,jj)-cij(ii,jj)
         a(ii,jj)=p(jj,ii)*aij(jj,ii)+cij(jj,ii)
        endif
  114   continue
  113  continue
      x(16)=1.d0
      CALL MATRIX(16)
      if(m.lt.2) goto 9988
C
      cool=0.0d0
       do 131 ii=1,15
        do 132 jj=ii+1,16
        fbfii(jj,ii)=(1.6022d-12*aij(jj,ii)*(ene(jj)-ene(ii))*x(jj))
     &  /edens
        cool=cool+fbfii(jj,ii)
  132   continue
  131  continue
      fecool=cool
C
C - Add cooling from higher levels if T > 3E3 K.
C
      if(te.ge.3.d3) then
      ipsave=0
       do 8201 ip=1,4
       if(tlog.ge.tfix(ip).and.tlog.lt.tfix(ip+1)) then
        ipsave=ip
       endif
 8201  continue
      if(tlog.ge.tfix(5)) ipsave=4
      cte=(tlog-tfix(ipsave))/(tfix(ipsave+1)-tfix(ipsave))
      af=afix(ipsave)*(1.d0-cte)+afix(ipsave+1)*cte
      bf=bfix(ipsave)*(1.d0-cte)+bfix(ipsave+1)*cte
      fehigh=1.d1**(af+bf*dlog10(edens))
      fecool=fecool+fehigh
      endif
C
      RETURN
      END
C************************************************************           
C************************************************************           
      SUBROUTINE FEIV(TE,EDENS,FECOOL)
C************************************************************           
C
C  Fe IV is treated as a 22-level atom.
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
C                                                                       
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      common/fe4/aij(22,22),ome(22,22)
C                                                                       
      DIMENSION FBFIV(22,22),ENE(22),G(22),CIJ(22,22)
C
      DATA G/6.D0,6.D0,12.D0,10.D0,8.D0,6.D0,4.D0,2.D0,8.D0,2.D0,
     &4.D0,6.D0,12.D0,14.D0,6.D0,4.D0,8.D0,6.D0,10.D0,8.D0,6.D0,4.D0/
      DATA ENE/0.0D0,3.9939D0,3.999D0,4.0059D0,4.0064D0,4.3512D0,
     &4.3608D0,4.3716D0,4.8092D0,4.8196D0,4.8276D0,4.8294D0,5.8543D0,
     &5.864D0,6.0881D0,6.1483D0,6.3527D0,6.4303D0,6.5258D0,6.5351D0,
     &6.5529D0,6.5583D0/
C
      TEV=TE/1.1605D4
      CONS=(8.63D-6*EDENS)/DSQRT(TE)
       do 101 ii=1,22
        do 102 jj=1,22
        a(ii,jj)=0.0d0
        cij(ii,jj)=0.0d0
  102   continue
       x(ii)=0.0d0
  101  continue
        do 111 ii=1,21
         do 112 jj=ii+1,22
         exiijj=expfn((ene(jj)-ene(ii))/tev)
         cij(ii,jj)=(cons*ome(ii,jj)*exiijj)/g(ii)
         cij(jj,ii)=(g(ii)*cij(ii,jj))/(exiijj*g(jj))
  112    continue
  111   continue
        do 113 ii=1,22
         do 114 jj=1,22
         if(ii.eq.1) then
          a(ii,jj)=1.d0
         else
          if(ii.eq.jj) goto 114
          a(ii,ii)=a(ii,ii)-aij(ii,jj)-cij(ii,jj)
          a(ii,jj)=aij(jj,ii)+cij(jj,ii)
         endif
  114    continue
  113   continue
       x(1)=1.d0
       CALL MATRIX(22)
        do 7891 ii=1,22
        if(x(ii).lt.0.0d0) x(ii)=1.d-50
 7891   continue
       cool=0.0d0
        do 131 ii=1,21
         do 132 jj=ii+1,22
         fbfiv(jj,ii)=(1.6022d-12*aij(jj,ii)*(ene(jj)-ene(ii))*x(jj))
     &   /edens
         cool=cool+fbfiv(jj,ii)
  132    continue
  131   continue
       fecool=cool
C
       RETURN
       END
C************************************************************           
C************************************************************           
      SUBROUTINE FEVII(TE,EDENS,FECOOL)
C************************************************************           
C
C  Calculates the forbidden line emission of Fe VII. Atomic data
C  from Nussbaumer&Storey (A&A, 1982, 113 p.21), and Keenan&
C  Norrington (A&A, 1987, 181 p.370).
C  Fe VII is treated as a 9-level atom.
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
C                                                                       
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      COMMON/FBFE7/FEADD(17)
C                                                                       
      DIMENSION OME(9,9),ENE(9),G(9),CIJ(9,9),AIJ(9,9),FBFVII(9,9)
C
      DATA G/5.D0,7.D0,9.D0,5.D0,1.D0,3.D0,5.D0,9.D0,1.D0/
      DATA ENE/0.0D0,0.1304D0,0.1587D0,2.168D0,2.486D0,2.534D0,
     &2.64D0,3.588D0,8.318D0/
C
      T4=TE/1.D4
      TEV=TE/1.1605D4
      CONS=(8.63D-6*EDENS)/DSQRT(TE)
       do 101 ii=1,9
        do 102 jj=1,9
        ome(ii,jj)=0.0d0
        a(ii,jj)=0.0d0
        aij(ii,jj)=0.0d0
        cij(ii,jj)=0.0d0
  102   continue
       x(ii)=0.0d0
  101  continue
       ome(1,2)=1.49d0/t4**.264
       ome(1,3)=0.389d0/t4**.222
       ome(1,4)=0.511d0/t4**.096
       ome(1,5)=0.152d0/t4**.022
       ome(1,6)=0.283d0/t4**.033
       ome(1,7)=0.149d0/t4**.061
       ome(1,8)=0.518d0/t4**.020
       ome(1,9)=0.0357d0/t4**.043
       ome(2,3)=1.57d0/t4**.215
       ome(2,4)=0.666d0/t4**.103
       ome(2,5)=0.0967d0/t4**.053
       ome(2,6)=0.376d0/t4**.025
       ome(2,7)=0.415d0/t4**.039
       ome(2,8)=0.734d0/t4**.020
       ome(2,9)=0.052d0/t4**.042
       ome(3,4)=0.793d0/t4**.115
       ome(3,5)=0.0558d0/t4**.041
       ome(3,6)=0.244d0/t4**.042
       ome(3,7)=0.940d0/t4**.029
       ome(3,8)=0.956d0/t4**.020
       ome(3,9)=0.0705d0/t4**.043
       ome(4,5)=0.0965d0/t4**.076
       ome(4,6)=0.286d0/t4**.088
       ome(4,7)=0.516d0/t4**.180
       ome(4,8)=0.387d0/t4**.017
       ome(4,9)=0.24d0/t4**.019
       ome(5,6)=0.153d0/t4**.072
       ome(5,7)=0.131d0/t4**.045
       ome(5,8)=0.103d0/t4**.028
       ome(5,9)=0.0178d0/t4**.031
       ome(6,7)=0.486d0/t4**.052
       ome(6,8)=0.311d0/t4**.027
       ome(6,9)=0.0545d0/t4**.031
       ome(7,8)=0.523d0/t4**.026
       ome(7,9)=0.105d0/t4**.029
       ome(8,9)=0.0319d0/t4**.061
       aij(9,1)=1.67d-1
       aij(9,4)=2.17d1
       aij(9,6)=6.93d0
       aij(9,7)=1.39d0
       aij(8,1)=5.55d-4
       aij(8,2)=3.14d-1
       aij(8,3)=4.54d-1
       aij(8,4)=1.23d-3
       aij(8,7)=1.74d-5
       aij(7,1)=1.93d-2
       aij(7,2)=7.69d-2
       aij(7,3)=4.96d-2
       aij(7,4)=1.56d-1
       aij(7,5)=1.39d-8
       aij(7,6)=7.62d-3
       aij(6,1)=3.55d-2
       aij(6,2)=5.25d-2
       aij(6,4)=4.37d-2
       aij(6,5)=1.06d-3
       aij(5,1)=9.37d-2
       aij(5,4)=2.07d-7
       aij(4,1)=3.60d-1
       aij(4,2)=5.77d-1
       aij(4,3)=1.55d-3
       aij(3,1)=1.34d-9
       aij(3,2)=4.24d-2
       aij(2,1)=2.98d-2
        do 111 ii=1,8
         do 112 jj=ii+1,9
         exiijj=expfn((ene(jj)-ene(ii))/tev)
         cij(ii,jj)=(cons*ome(ii,jj)*exiijj)/g(ii)
         cij(jj,ii)=(g(ii)*cij(ii,jj))/(exiijj*g(jj))
  112    continue
  111   continue
        do 113 ii=1,9
         do 114 jj=1,9
         if(ii.eq.1) then
          a(ii,jj)=1.d0
         else
          if(ii.eq.jj) goto 114
          a(ii,ii)=a(ii,ii)-aij(ii,jj)-cij(ii,jj)
          a(ii,jj)=aij(jj,ii)+cij(jj,ii)
         endif
  114    continue
  113   continue
       x(1)=1.d0
       CALL MATRIX(9)
        do 7891 ii=1,9
        if(x(ii).lt.0.0d0) x(ii)=1.d-50
 7891   continue
       cool=0.0d0
        do 131 ii=1,8
         do 132 jj=ii+1,9
         fbfvii(jj,ii)=(1.6022d-12*aij(jj,ii)*(ene(jj)-ene(ii))*x(jj))
     &   /edens
         cool=cool+fbfvii(jj,ii)
  132    continue
  131   continue
       feadd(1)=fbfvii(9,1)
       feadd(2)=fbfvii(9,4)
       feadd(3)=fbfvii(9,6)
       feadd(4)=fbfvii(9,7)
       feadd(5)=fbfvii(8,1)
       feadd(6)=fbfvii(8,2)
       feadd(7)=fbfvii(8,3)
       feadd(8)=fbfvii(7,1)
       feadd(9)=fbfvii(6,1)
       feadd(10)=fbfvii(7,2)
       feadd(11)=fbfvii(5,1)
       feadd(12)=fbfvii(6,2)
       feadd(13)=fbfvii(7,3)
       feadd(14)=fbfvii(4,1)
       feadd(15)=fbfvii(4,2)
       feadd(16)=fbfvii(4,3)
       feadd(17)=fbfvii(8,4)
       FECOOL=cool
C
      RETURN
      END
C************************************************************           
C************************************************************           
      SUBROUTINE FEVII_NEW(TE,EDENS,FECOOL)
C************************************************************           
C
C  Calculates the forbidden line emission of Fe VII. Atomic data
C  from Nussbaumer&Storey (A&A, 1982, 113 p.21), and Keenan&
C  Norrington (A&A, 1987, 181 p.370).
C  Fe VII is treated as a 9-level atom.
C  Updated with Berrington et al. (2000) and corrected by
C  P. Young in 2005 for typos in the Berrington et al. paper.
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
C                                                                       
      parameter (mmatr=80)
C
      COMMON/MSTUFF/X(mmatr),A(mmatr,mmatr),EPS,NERR
      COMMON/FBFE7/FEADD(17)
C                                                                       
      DIMENSION OME(9,9),ENE(9),G(9),CIJ(9,9),AIJ(9,9),FBFVII(9,9),
     &om12(8),om13(8),om14(8),om15(8),om16(8),om17(8),om18(8),
     &om19(8),om23(8),om24(8),om25(8),om26(8),om27(8),om28(8),
     &om29(8),om34(8),om35(8),om36(8),om37(8),om38(8),om39(8),
     &om45(8),om46(8),om47(8),om48(8),om49(8),om56(8),om57(8),
     &om58(8),om59(8),om67(8),om68(8),om69(8),om78(8),om79(8),
     &om89(8),tberr(8)
C
      DATA G/5.D0,7.D0,9.D0,5.D0,1.D0,3.D0,5.D0,9.D0,1.D0/
      DATA ENE/0.0D0,0.1304D0,0.2927D0,2.168D0,2.486D0,2.534D0,
     &2.64D0,3.588D0,8.318D0/
      data om12/3.71d0,3.35d0,2.98d0,2.69d0,2.48d0,2.20d0,
     &1.79d0,1.35d0/
      data om13/1.30d0,1.17d0,1.02d0,.962d0,.862d0,.776d0,
     &.634d0,.469d0/
      data om14/1.01d0,.959d0,.918d0,.873d0,.813d0,.721d0,
     &.593d0,.455d0/
      data om15/.295d0,.299d0,.299d0,.302d0,.305d0,.292d0,
     &.259d0,.217d0/
      data om16/.619d0,.633d0,.628d0,.623d0,.619d0,.585d0,
     &.504d0,.403d0/
      data om17/.516d0,.549d0,.548d0,.534d0,.518d0,.473d0,
     &.387d0,.288d0/
      data om18/1.14d0,1.24d0,1.26d0,1.20d0,1.11d0,.971d0,
     &.786d0,.595d0/
      data om19/.134d0,.115d0,.097d0,.087d0,.080d0,.071d0,
     &.057d0,.043d0/
      data om23/4.55d0,4.11d0,3.64d0,3.30d0,3.04d0,2.71d0,
     &2.21d0,1.66d0/
      data om24/1.38d0,1.29d0,1.23d0,1.16d0,1.08d0,.952d0,
     &.78d0,.596d0/
      data om25/.228d0,.235d0,.231d0,.226d0,.222d0,.207d0,
     &.174d0,.133d0/
      data om26/.807d0,.833d0,.837d0,.837d0,.837d0,.794d0,
     &.693d0,.564d0/
      data om27/1.02d0,1.06d0,1.06d0,1.05d0,1.03d0,.964d0,
     &.818d0,.637d0/
      data om28/1.59d0,1.74d0,1.76d0,1.68d0,1.56d0,1.36d0,
     &1.11d0,.837d0/
      data om29/.190d0,.162d0,.138d0,.123d0,.113d0,.1d0,
     &.081d0,.061d0/
      data om34/1.72d0,1.60d0,1.51d0,1.42d0,1.32d0,1.16d0,
     &.943d0,.716d0/
      data om35/.171d0,.187d0,.191d0,.186d0,.181d0,.167d0,
     &.139d0,.104d0/
      data om36/.649d0,.690d0,.692d0,.677d0,.663d0,.616d0,
     &.516d0,.390d0/
      data om37/1.89d0,1.94d0,1.95d0,1.94d0,1.93d0,1.84d0,
     &1.60d0,1.30d0/
      data om38/2.06d0,2.25d0,2.28d0,2.18d0,2.02d0,1.77d0,
     &1.43d0,1.09d0/
      data om39/.249d0,.213d0,.181d0,.162d0,.149d0,.132d0,
     &.107d0,.080d0/
      data om45/.159d0,.172d0,.179d0,.178d0,.170d0,.154d0,
     &.128d0,.099d0/
      data om46/.490d0,.531d0,.553d0,.544d0,.515d0,.462d0,
     &.382d0,.292d0/
      data om47/1.022d0,1.06d0,1.07d0,1.03d0,.952d0,.833d0,
     &.668d0,.495d0/
      data om48/2.01d0,2.02d0,1.91d0,1.76d0,1.62d0,1.43d0,
     &1.16d0,.889d0/
      data om49/.509d0,.473d0,.468d0,.483d0,.492d0,.480d0,
     &.439d0,.384d0/
      data om56/.350d0,.370d0,.380d0,.380d0,.389d0,.402d0,
     &.375d0,.306d0/
      data om57/.315d0,.324d0,.324d0,.329d0,.347d0,.363d0,
     &.342d0,.287d0/
      data om58/.155d0,.164d0,.178d0,.186d0,.183d0,.166d0,
     &.138d0,.106d0/
      data om59/.030d0,.035d0,.040d0,.041d0,.039d0,.034d0,
     &.028d0,.021d0/
      data om67/1.13d0,1.17d0,1.18d0,1.20d0,1.25d0,1.31d0,
     &1.23d0,1.03d0/
      data om68/.466d0,.495d0,.536d0,.558d0,.549d0,.500d0,
     &.416d0,.320d0/
      data om69/.088d0,.105d0,.120d0,.123d0,.116d0,.103d0,
     &.084d0,.064d0/
      data om78/.857d0,.903d0,.961d0,.988d0,.965d0,.875d0,
     &.726d0,.558d0/
      data om79/.173d0,.197d0,.220d0,.225d0,.215d0,.192d0,
     &.159d0,.122d0/
      data om89/.255d0,.310d0,.346d0,.345d0,.320d0,.273d0,
     &.209d0,.146d0/
      data tberr/4.3d0,4.5d0,4.75d0,5.0d0,5.25d0,5.5d0,5.75d0,
     &6.0d0/
C
      T4=TE/1.D4
      TEV=TE/1.1605D4
      TLG=DLOG10(TE)
      CONS=(8.63D-6*EDENS)/DSQRT(TE)
       do 101 ii=1,9
        do 102 jj=1,9
        ome(ii,jj)=0.0d0
        a(ii,jj)=0.0d0
        aij(ii,jj)=0.0d0
        cij(ii,jj)=0.0d0
  102   continue
       x(ii)=0.0d0
  101  continue
      do 7870 iis=1,7
         tefix=tberr(iis)
         if(tlg.ge.tberr(iis).and.tlg.lt.tberr(iis+1)) then
            iit=iis
            goto 7871
         endif
 7870 continue
 7871 continue
      if(tlg.lt.4.3d0) iit=1
      if(tlg.lt.4.3d0) tefix=4.3d0
      if(tlg.ge.6.d0) iit=7
      if(tlg.ge.6.d0) tefix=5.75d0
      qq=(tlg-tefix)/(tberr(iit+1)-tberr(iit))
       ome(1,2)=1.d1**((dlog10(om12(iit)))*(1.d0-qq)+
     &              (dlog10(om12(iit+1)))*qq)
       ome(1,3)=1.d1**((dlog10(om13(iit)))*(1.d0-qq)+
     &              (dlog10(om13(iit+1)))*qq)
       ome(1,4)=1.d1**((dlog10(om14(iit)))*(1.d0-qq)+
     &              (dlog10(om14(iit+1)))*qq)
       ome(1,5)=1.d1**((dlog10(om15(iit)))*(1.d0-qq)+
     &              (dlog10(om15(iit+1)))*qq)
       ome(1,6)=1.d1**((dlog10(om16(iit)))*(1.d0-qq)+
     &              (dlog10(om16(iit+1)))*qq)
       ome(1,7)=1.d1**((dlog10(om17(iit)))*(1.d0-qq)+
     &              (dlog10(om17(iit+1)))*qq)
       ome(1,8)=1.d1**((dlog10(om18(iit)))*(1.d0-qq)+
     &              (dlog10(om18(iit+1)))*qq)
       ome(1,9)=1.d1**((dlog10(om19(iit)))*(1.d0-qq)+
     &              (dlog10(om19(iit+1)))*qq)
       ome(2,3)=1.d1**((dlog10(om23(iit)))*(1.d0-qq)+
     &              (dlog10(om23(iit+1)))*qq)
       ome(2,4)=1.d1**((dlog10(om24(iit)))*(1.d0-qq)+
     &              (dlog10(om24(iit+1)))*qq)
       ome(2,5)=1.d1**((dlog10(om25(iit)))*(1.d0-qq)+
     &              (dlog10(om25(iit+1)))*qq)
       ome(2,6)=1.d1**((dlog10(om26(iit)))*(1.d0-qq)+
     &              (dlog10(om26(iit+1)))*qq)
       ome(2,7)=1.d1**((dlog10(om27(iit)))*(1.d0-qq)+
     &              (dlog10(om27(iit+1)))*qq)
       ome(2,8)=1.d1**((dlog10(om28(iit)))*(1.d0-qq)+
     &              (dlog10(om28(iit+1)))*qq)
       ome(2,9)=1.d1**((dlog10(om29(iit)))*(1.d0-qq)+
     &              (dlog10(om29(iit+1)))*qq)
       ome(3,4)=1.d1**((dlog10(om34(iit)))*(1.d0-qq)+
     &              (dlog10(om34(iit+1)))*qq)
       ome(3,5)=1.d1**((dlog10(om35(iit)))*(1.d0-qq)+
     &              (dlog10(om35(iit+1)))*qq)
       ome(3,6)=1.d1**((dlog10(om36(iit)))*(1.d0-qq)+
     &              (dlog10(om36(iit+1)))*qq)
       ome(3,7)=1.d1**((dlog10(om37(iit)))*(1.d0-qq)+
     &              (dlog10(om37(iit+1)))*qq)
       ome(3,8)=1.d1**((dlog10(om38(iit)))*(1.d0-qq)+
     &              (dlog10(om38(iit+1)))*qq)
       ome(3,9)=1.d1**((dlog10(om39(iit)))*(1.d0-qq)+
     &              (dlog10(om39(iit+1)))*qq)
       ome(4,5)=1.d1**((dlog10(om45(iit)))*(1.d0-qq)+
     &              (dlog10(om45(iit+1)))*qq)
       ome(4,6)=1.d1**((dlog10(om46(iit)))*(1.d0-qq)+
     &              (dlog10(om46(iit+1)))*qq)
       ome(4,7)=1.d1**((dlog10(om47(iit)))*(1.d0-qq)+
     &              (dlog10(om47(iit+1)))*qq)
       ome(4,8)=1.d1**((dlog10(om48(iit)))*(1.d0-qq)+
     &              (dlog10(om48(iit+1)))*qq)
       ome(4,9)=1.d1**((dlog10(om49(iit)))*(1.d0-qq)+
     &              (dlog10(om49(iit+1)))*qq)
       ome(5,6)=1.d1**((dlog10(om56(iit)))*(1.d0-qq)+
     &              (dlog10(om56(iit+1)))*qq)
       ome(5,7)=1.d1**((dlog10(om57(iit)))*(1.d0-qq)+
     &              (dlog10(om57(iit+1)))*qq)
       ome(5,8)=1.d1**((dlog10(om58(iit)))*(1.d0-qq)+
     &              (dlog10(om58(iit+1)))*qq)
       ome(5,9)=1.d1**((dlog10(om59(iit)))*(1.d0-qq)+
     &              (dlog10(om59(iit+1)))*qq)
       ome(6,7)=1.d1**((dlog10(om67(iit)))*(1.d0-qq)+
     &              (dlog10(om67(iit+1)))*qq)
       ome(6,8)=1.d1**((dlog10(om68(iit)))*(1.d0-qq)+
     &              (dlog10(om68(iit+1)))*qq)
       ome(6,9)=1.d1**((dlog10(om69(iit)))*(1.d0-qq)+
     &              (dlog10(om69(iit+1)))*qq)
       ome(7,8)=1.d1**((dlog10(om78(iit)))*(1.d0-qq)+
     &              (dlog10(om78(iit+1)))*qq)
       ome(7,9)=1.d1**((dlog10(om79(iit)))*(1.d0-qq)+
     &              (dlog10(om79(iit+1)))*qq)
       ome(8,9)=1.d1**((dlog10(om89(iit)))*(1.d0-qq)+
     &              (dlog10(om89(iit+1)))*qq)
C       write(6,1011)tlg,ome(1,2)
 1011  format(1x,20e12.4)
C       goto 9988
       aij(9,1)=1.34d-1
       aij(9,4)=2.67d1
       aij(9,6)=6.88d0
       aij(9,7)=1.11d0
       aij(8,1)=9.59d-4
       aij(8,2)=3.43d-1
       aij(8,3)=5.03d-1
       aij(8,4)=4.14d-3
       aij(8,7)=4.54d-5
C       aij(7,1)=1.74d-1
       aij(7,1)=1.50d-2
       aij(7,2)=6.97d-2
       aij(7,3)=7.35d-2
C       aij(7,4)=1.91d-1
       aij(7,4)=1.82d-1
       aij(7,5)=1.39d-8
       aij(7,6)=7.43d-3
       aij(6,1)=5.02d-2
       aij(6,2)=7.62d-2
       aij(6,4)=5.72d-2
       aij(6,5)=1.15d-3
       aij(5,1)=1.35d-1
       aij(5,4)=4.72d-7
C       aij(4,1)=3.25d0
       aij(4,1)=3.72d-1
       aij(4,2)=6.03d-1
       aij(4,3)=1.39d-3
       aij(3,1)=1.67d-9
       aij(3,2)=4.66d-2
       aij(2,1)=3.25d-2
        do 111 ii=1,8
         do 112 jj=ii+1,9
         exiijj=expfn((ene(jj)-ene(ii))/tev)
         cij(ii,jj)=(cons*ome(ii,jj)*exiijj)/g(ii)
         cij(jj,ii)=(g(ii)*cij(ii,jj))/(exiijj*g(jj))
  112    continue
  111   continue
        do 113 ii=1,9
         do 114 jj=1,9
         if(ii.eq.1) then
          a(ii,jj)=1.d0
         else
          if(ii.eq.jj) goto 114
          a(ii,ii)=a(ii,ii)-aij(ii,jj)-cij(ii,jj)
          a(ii,jj)=aij(jj,ii)+cij(jj,ii)
         endif
  114    continue
  113   continue
       x(1)=1.d0
       CALL MATRIX(9)
        do 7891 ii=1,9
        if(x(ii).lt.0.0d0) x(ii)=1.d-50
 7891   continue
       cool=0.0d0
        do 131 ii=1,8
         do 132 jj=ii+1,9
         fbfvii(jj,ii)=(1.6022d-12*aij(jj,ii)*(ene(jj)-ene(ii))*x(jj))
     &   /edens
         cool=cool+fbfvii(jj,ii)
  132    continue
  131   continue
       feadd(1)=fbfvii(9,1)
       feadd(2)=fbfvii(9,4)
       feadd(3)=fbfvii(9,6)
       feadd(4)=fbfvii(9,7)
       feadd(5)=fbfvii(8,1)
       feadd(6)=fbfvii(8,2)
       feadd(7)=fbfvii(8,3)
       feadd(8)=fbfvii(7,1)
       feadd(9)=fbfvii(6,1)
       feadd(10)=fbfvii(7,2)
       feadd(11)=fbfvii(5,1)
       feadd(12)=fbfvii(6,2)
       feadd(13)=fbfvii(7,3)
       feadd(14)=fbfvii(4,1)
       feadd(15)=fbfvii(4,2)
       feadd(16)=fbfvii(4,3)
       feadd(17)=fbfvii(8,4)
       FECOOL=cool
C
 9988 continue
      RETURN
      END
C************************************************************           
C************************************************************           
      SUBROUTINE MATRIX(N)                                              
C************************************************************           
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      parameter (mmatr=80)
C
      COMMON/MSTUFF/B(mmatr),A(mmatr,mmatr),EPS,NERR
C
      DIMENSION LASTN(mmatr),ASAVE(mmatr)
C                                                                       
C -- Find the column index of the last non-zero element in each row --  
C -- Speeds up the solution of loose matrices considerably --           
C                                                                       
      DO 100 I=1,N                                                      
      DO 101 J=N,I,-1                                                   
      JSAVE=J                                                           
      IF(A(I,J).NE.0.0D0) GOTO 102                                      
  101 CONTINUE                                                          
  102 LASTN(I)=JSAVE                                                    
  100 CONTINUE                                                          
C                                                                       
C -- ************************** --                                      
C -- Forward elimination scheme --                                      
C -- ************************** --                                      
C                                                                       
      DO 200 I=1,N-1                                                    
C                                                                       
C -- Partial pivoting routine --                                        
C                                                                       
      AI=0.0D0                                                          
      DO 201 L=I,N                                                      
      ASAVE(L)=0.0D0                                                    
      IF(DABS(A(L,I)).GT.AI) LSAVE=L                                    
      IF(DABS(A(L,I)).GT.AI) AI=DABS(A(L,I))                            
  201 CONTINUE                                                          
      IF(LSAVE.EQ.I) GOTO 202                                           
C                                                                       
C -- Interchange rows A(I,?) and A(LSAVE,?) if LSAVE.NE.I --            
C                                                                       
      LAST=MAX0(LASTN(I),LASTN(LSAVE))                                  
      DO 203 L=I,LAST                                                   
      ASAVE(L)=A(I,L)                                                   
      A(I,L)=A(LSAVE,L)                                                 
  203 A(LSAVE,L)=ASAVE(L)                                               
      BSAVE=B(I)                                                        
      B(I)=B(LSAVE)                                                     
      B(LSAVE)=BSAVE                                                    
      LASTNI=LASTN(I)                                                   
      LASTN(I)=LASTN(LSAVE)                                             
      LASTN(LSAVE)=LASTNI                                               
  202 CONTINUE                                                          
C                                                                       
C -- Elimination routine --                                             
C                                                                       
      BI=B(I)                                                           
      AII=A(I,I)                                                        
      DO 301 J=I+1,N                                                    
      IF(A(J,I).EQ.0.0D0.OR.(I+1).GT.LASTN(I)) GOTO 301                 
      AJIAII=A(J,I)/A(I,I)                                              
      DO 302 K=I+1,LASTN(I)                                             
  302 A(J,K)=A(J,K)-AJIAII*A(I,K)                                       
      B(J)=B(J)-AJIAII*BI                                               
      LASTN(J)=MAX0(LASTN(J),LASTN(I))                                  
  301 CONTINUE                                                          
C                                                                       
C -- ************************** --                                      
C -- End of forward elimination --                                      
C -- ************************** --                                      
C                                                                       
  200 CONTINUE                                                          
C                                                                       
C -- *************** --                                                 
C -- Back-substitute --                                                 
C -- *************** --                                                 
C                                                                       
      DO 400 K=N,1,-1                                                   
      BK=B(K)                                                           
      IF((K+1).GT.N) GOTO 401                                           
      DO 402 L=K+1,N                                                    
      BK=BK-A(K,L)*B(L)                                                 
  402 CONTINUE                                                          
  401 B(K)=BK/A(K,K)                                                    
  400 CONTINUE                                                          
C                                                                       
      RETURN                                                            
      END                                                               
C************************************************************           
C***************************************************                    
      DOUBLE PRECISION FUNCTION SIMUL(N,A,B,X)                          
C
      IMPLICIT REAL*8(A-H,O-Z)                                          
C
      parameter(nf=40,nfp1=41,nf2=80,nf21=81)                           
C
      dimension a(nf21,nf21),b(nf21),x(nf21)                            
      DIMENSION LASTN(nf21),ASAVE(nf21),BS(nf21)                        
C                                                                       
C -- Find the column index of the last non-zero element in each row --  
C -- Speeds up the solution of loose matrices considerably --           
C                                                                       
      DO 100 I=1,N                                                      
      BS(I)=B(I)                                                        
      DO 101 J=N,I,-1                                                   
      JSAVE=J                                                           
      IF(A(I,J).NE.0.0D0) GOTO 102                                      
  101 CONTINUE                                                          
  102 LASTN(I)=JSAVE                                                    
  100 CONTINUE                                                          
C                                                                       
C -- ************************** --                                      
C -- Forward elimination scheme --                                      
C -- ************************** --                                      
C                                                                       
      DO 200 I=1,N-1                                                    
C                                                                       
C -- Partial pivoting routine --                                        
C                                                                       
      AI=0.0D0                                                          
      DO 201 L=I,N                                                      
      ASAVE(L)=0.0D0                                                    
      IF(DABS(A(L,I)).GT.AI) LSAVE=L                                    
      IF(DABS(A(L,I)).GT.AI) AI=DABS(A(L,I))                            
  201 CONTINUE                                                          
      IF(LSAVE.EQ.I) GOTO 202                                           
C                                                                       
C -- Interchange rows A(I,?) and A(LSAVE,?) if LSAVE.NE.I --            
C                                                                       
      LAST=MAX0(LASTN(I),LASTN(LSAVE))                                  
      DO 203 L=I,LAST                                                   
      ASAVE(L)=A(I,L)                                                   
      A(I,L)=A(LSAVE,L)                                                 
  203 A(LSAVE,L)=ASAVE(L)                                               
      BSAVE=B(I)                                                        
      B(I)=B(LSAVE)                                                     
      B(LSAVE)=BSAVE                                                    
      LASTNI=LASTN(I)                                                   
      LASTN(I)=LASTN(LSAVE)                                             
      LASTN(LSAVE)=LASTNI                                               
  202 CONTINUE                                                          
C                                                                       
C -- Elimination routine --                                             
C                                                                       
      BI=B(I)                                                           
      AII=A(I,I)                                                        
      DO 301 J=I+1,N                                                    
      IF(A(J,I).EQ.0.0D0.OR.(I+1).GT.LASTN(I)) GOTO 301                 
      AJIAII=A(J,I)/A(I,I)                                              
      DO 302 K=I+1,LASTN(I)                                             
  302 A(J,K)=A(J,K)-AJIAII*A(I,K)                                       
      B(J)=B(J)-AJIAII*BI                                               
      LASTN(J)=MAX0(LASTN(J),LASTN(I))                                  
  301 CONTINUE                                                          
C                                                                       
C -- ************************** --                                      
C -- End of forward elimination --                                      
C -- ************************** --                                      
C                                                                       
  200 CONTINUE                                                          
C                                                                       
C -- *************** --                                                 
C -- Back-substitute --                                                 
C -- *************** --                                                 
C                                                                       
      DO 400 K=N,1,-1                                                   
      BK=B(K)                                                           
      IF((K+1).GT.N) GOTO 401                                           
      DO 402 L=K+1,N                                                    
      BK=BK-A(K,L)*B(L)                                                 
  402 CONTINUE                                                          
  401 B(K)=BK/A(K,K)                                                    
  400 CONTINUE                                                          
      DO 9 I=1,N                                                        
      X(I)=B(I)                                                         
9     B(I)=BS(I)                                                        
C                                                                       
      simul=0.                                                          
      RETURN                                                            
      END                                                               
C***************************************************************           
C***************************************************************           
      SUBROUTINE BOWEN(EO2,TE,DENO3,DENHE2,CONOPA,YLA,YB,YO2,YA,
     &xela)
C***************************************************************           
      implicit real*8(a-h,o-z)                                          
      real*8 jm                                                         
C                                                                       
      parameter(nf=40,nfp1=41,nf2=80,nf21=81)                           
C                                                                       
      common/a41/ALFLA(-NF:NF),ALFO1(-NF:NF),BETAB,BETAO2,ALFC,ELA      
      common/a43/ f(-nf:nf),w(-nf:nf)                                   
      common/a44/nstep,NBIN1,WMAX                                       
      COMMON/CONSTS/PI,PISQRT,CLIGHT                                    
c                                                                       
      dimension dmat(-nf:nf,-nf:nf),s(-nf:nf),aa(nf21,nf21),b(nf21)     
      DIMENSION FILA(-NF:NF),FIO1(-NF:NF)                               
      dimension jm(nf21)                                                
C                                                                       
      DATA BO2/0.245/,BB/0.018/                                         
      DATA FO1/0.11/,GAMO1/7.6D9/,FLA/0.4162/,GAMLA/1.D10/              
     &,FO2/0.059/                                                       
      DATA AHE/4./,AO/16./                                              
C                                                                       
      vla=CLIGHT/303.783e-8                                             
      vo1=CLIGHT/303.799e-8                                             
      nstep=14                                                          
      nbin1=6                                                           
      wmax=6.d0                                                         
      t4=te/1.e4                                                        
      dvla=vla*12.85e5*sqrt(t4/ahe)/CLIGHT                              
      dvo1=vo1*12.85e5*sqrt(t4/ao)/CLIGHT                               
      dx=(dabs(vla-vo1))/dvla                                           
      dx1=(2.d0*dx)/dble(nbin1)                                         
      dx2=(wmax-2.d0*dx)/dble(nstep-nbin1)                              
C                                                                       
      call freq(dvla,dx1,dx2)                                           
C                                                                       
C -- Define normalized profile functions --                             
C                                                                       
      ALAVOI=GAMLA/(4.D0*PI*dvla)                                       
      AO1VOI=GAMO1/(4.D0*PI*dvo1)                                       
      CALL VOIGT(ALAVOI,0.0D0,FILA0)                                    
      CALL VOIGT(AO1VOI,0.0D0,FIO10)                                    
      FILA0=FILA0/PISQRT                                                
      FIO10=FIO10/PISQRT                                                
C                                                                       
      C1=dvla/dvo1                                                      
      C2=(vla-vo1)/dvo1                                                 
      DO 800 I=-NSTEP,NSTEP,1                                           
      CALL VOIGT(ALAVOI,F(I),H)                                         
      FILA(I)=H/PISQRT                                                  
      U=C1*F(I)+C2                                                      
      CALL VOIGT(AO1VOI,U,H)                                            
      FIO1(I)=H/PISQRT                                                  
  800 CONTINUE
C                                                                       
C -- Define ALFA-functions --                                           
C                                                                       
      CAPLA=(0.0265*FLA*DENHE2)/dvla                                    
      CAPO1=(0.0265*FO1*DENO3)/dvo1                                     
      CAPLA0=CAPLA*FILA0                                                
      ALFC=CONOPA/CAPLA0                                                
      DO 900 I=-NSTEP,NSTEP,1                                           
      ALFLA(I)=FILA(I)/FILA0                                            
      ALFO1(I)=(FIO1(I)*CAPO1)/CAPLA0                                   
  900 CONTINUE
C                                                                       
C -- Calculate escape probabilities --                                  
C                                                                       
      ela=xela                                                          
C                                                                       
C -- Define Beta-factors --                                             
C                                                                       
      DENOM=1.D0-BO2*(1.D0-eo2)                                         
      BETAO2=(BO2*eo2)/DENOM                                            
      BETAB=BB/DENOM                                                    
      BETAO1=BETAO2+BETAB                                               
C                                                                       
C -- Set up the matrix --                                               
C                                                                       
      C3=1.d0-ela                                                       
      C4=1.d0-BETAO1                                                    
c                                                                       
      do 1 i=-nstep,nstep,1                                             
      s(i)=-ALFLA(I)                                                    
      AK=AKNEER(F(I),ak1)                                               
      C33=C3*AK*ALFLA(I)                                                
      C44=C4*ALFO1(I)                                                   
      C55=C3*AK1*ALFLA(I)                                               
      DO 11 J=-NSTEP,NSTEP,1                                            
      dmat(i,j)=w(j)*(C55*FILA(J)+C44*FIO1(J))                          
   11 CONTINUE                                                          
       dmat(i,i)=dmat(i,i)+C33-(ALFC+ALFLA(I)+ALFO1(I))                 
  1   continue                                                          
C --                                                                    
      do 2 i=-nstep,nstep,1                                             
      aa(i+nstep+1,2*nstep+2)=s(i)                                      
      b(i+nstep+1)=s(i)                                                 
      do 2 j=-nstep,nstep,1                                             
2     aa(i+nstep+1,j+nstep+1)=dmat(i,j)                                 
C                                                                       
C -- Solve the matrix equation --                                       
C                                                                       
      n=2*nstep+1                                                       
      sqw=simul(n,aa,b,jm)                                              
9     format(i4,5e12.4)                                                 
      call yield(yla,yb,yo2,ya,jm,pdestr)                               
      ytot=yla+yb+yo2+ya                                                
      yla=yla/ytot                                                      
      yb=yb/ytot                                                        
      yo2=yo2/ytot                                                      
      ya=ya/ytot                                                        
C                                                                       
      return                                                            
      end                                                               
C                                                                       
C************************************************************           
      subroutine freq(dvla,dx1,dx2)                                     
C                                                                       
C --employ Simpson weights (nstep and NBIN1 must be even)--             
C                                                                       
      implicit real*8(a-h,o-z)                                          
      parameter(nf=40,nfp1=41,nf2=80,nf21=81)                           
      common/a43/ f(-nf:nf),w(-nf:nf)                                   
      common/a44/nstep,NBIN1,WMAX                                       
c                                                                       
      nbin2=nstep-nbin1                                                 
      f(0)=0.0d0                                                        
      wei1=(2.d0*dx1)/3.d0                                              
      wei2=(2.d0*dx2)/3.d0                                              
      w(0)=wei1                                                         
       do 9011 i=1,nbin1                                                
        f(i)=dx1*dble(i)                                                
 9011  f(-i)=-f(i)                                                      
       do 9012 i=1,nbin2                                                
        f(nbin1+i)=dx2*dble(i)+f(nbin1)                                 
 9012  f(-nbin1-i)=-f(nbin1+i)                                          
      nh1=nbin1/2                                                       
      nh2=nbin2/2                                                       
       do 9013 i=1,nh1                                                  
        n1=2*i                                                          
        w(n1-1)=2.d0*wei1                                               
        w(n1)=wei1                                                      
        w(-n1+1)=2.d0*wei1                                              
 9013  w(-n1)=wei1                                                      
      w(nbin1)=w(nbin1)+(wei2-wei1)/2.d0                                
      w(-nbin1)=w(nbin1)                                                
       do 9014 i=1,nh2                                                  
        n2=2*i+nbin1                                                    
        w(n2-1)=2.d0*wei2                                               
        w(n2)=wei2                                                      
        w(-n2+1)=2.d0*wei2                                              
 9014  w(-n2)=wei2                                                      
      w(nstep)=w(nstep)-wei2/2.d0                                       
      w(-nstep)=w(nstep)                                                
c                                                                       
      RETURN                                                            
      end                                                               
C                                                                       
C*****************************************************************      
      subroutine yield(yla,yb,yo2,ya,jem,pdestr)                        
      implicit real*8(a-h,o-z)                                          
      real*8 jem                                                        
      parameter(nf=40,nfp1=41,nf2=80,nf21=81)                           
      common/a41/ALFLA(-NF:NF),ALFO1(-NF:NF),BETAB,BETAO2,ALFC,ELA      
      common/a43/ f(-nf:nf),w(-nf:nf)                                   
      common/a44/nstep,NBIN1,WMAX                                       
c                                                                       
      dimension jem(nf21)                                               
c                                                                       
      ytot=0.                                                           
      yb=0.                                                             
      yo2=0.                                                            
      ya=0.                                                             
      yla=0.                                                            
      sum1=0.                                                           
      sum2=0.                                                           
c                                                                       
      do 1 i=-nstep,nstep,1                                             
      flux=dmax1(0.0d0,jem(nstep+i+1))           
      ytot=ytot+w(i)*flux                                               
      yb=yb+w(i)*flux*ALFO1(I)                                          
      yo2=yo2+w(i)*flux*ALFO1(I)                                        
      yla=yla+w(i)*flux*ALFLA(I)                                        
      sum1=sum1+alfla(i)*w(i)*(alfc+alfo1(i))                           
      sum2=sum2+alfla(i)*w(i)*(alfla(i)+alfc+alfo1(i))                  
  1   continue                                                          
c                                                                       
      ya=ALFC*ytot                                                      
      yb=BETAB*yb                                                       
      yo2=BETAO2*yo2                                                    
      yla=ELA*yla                                                       
      if(yla.le.1.d-100) yla=1.d-100
      if(yb.le.1.d-100) yb=1.d-100
      if(yo2.le.1.d-100) yo2=1.d-100
      if(ya.le.1.d-100) ya=1.d-100
      sumyla=yla+yb+yo2+ya
      yla=yla/sumyla
      yb=yb/sumyla
      yo2=yo2/sumyla
      ya=ya/sumyla
      pdestr=sum1/sum2                                                  
C                                                                       
      return                                                            
      end                                                               
C***********************************************************            
      double precision function akneer(x,ak1)                           
      implicit real*8(a-h,o-z)                                          
      ak1=1.d0                                                          
      if((dabs(x)).gt.2.) ak1=expfn((.5*dabs(x)-1.)**2)                 
      akneer=1.D0-ak1                                                   
      return                                                            
      end                                                               
C***********************************************************            
